#!/usr/bin/env python3
"""
Analog Contract Deployment CLI
Deploy and manage analog contracts on the post-blockchain mainnet
"""

import argparse
import requests
import json
import sys
from pathlib import Path
from datetime import datetime

API_BASE = 'http://localhost:8080/api'

def deploy_contract(contract_file, name=None):
    """Deploy an analog contract from file"""
    try:
        contract_path = Path(contract_file)
        if not contract_path.exists():
            print(f"❌ Error: Contract file not found: {contract_file}")
            return False

        with open(contract_path, 'r') as f:
            contract_code = f.read()

        if not name:
            name = contract_path.stem

        print(f"📝 Deploying contract: {name}")
        print(f"📄 Source: {contract_file}")
        print(f"📊 Size: {len(contract_code)} bytes")
        print()

        response = requests.post(
            f'{API_BASE}/contracts/deploy',
            json={
                'name': name,
                'contract_code': contract_code
            },
            timeout=10
        )

        if response.status_code == 200:
            data = response.json()
            if data.get('success'):
                print("✅ Contract deployed successfully!")
                print(f"   Contract ID: {data['contract_id']}")
                print(f"   Status: {data['status']}")
                print(f"   Deployed at: {datetime.fromtimestamp(data['deployed_at']).isoformat()}")
                return True
            else:
                print(f"❌ Deployment failed: {data.get('error', 'Unknown error')}")
                return False
        else:
            print(f"❌ HTTP Error {response.status_code}: {response.text}")
            return False

    except FileNotFoundError:
        print(f"❌ Error: Contract file not found: {contract_file}")
        return False
    except requests.exceptions.ConnectionError:
        print("❌ Error: Cannot connect to analog API server")
        print("   Make sure the server is running: python analog_api_server.py")
        return False
    except Exception as e:
        print(f"❌ Error: {e}")
        return False

def list_contracts():
    """List all deployed contracts"""
    try:
        response = requests.get(f'{API_BASE}/contracts/list', timeout=10)
        if response.status_code == 200:
            data = response.json()
            if data.get('success'):
                contracts = data.get('contracts', [])
                if contracts:
                    print(f"📋 Deployed Contracts ({len(contracts)}):")
                    print()
                    for contract in contracts:
                        print(f"  • {contract['name']}")
                        print(f"    ID: {contract['contract_id']}")
                        print(f"    Status: {contract['status']}")
                        print(f"    Deployed: {contract['deployed_at']}")
                        print()
                else:
                    print("📋 No contracts deployed yet")
                return True
            else:
                print(f"❌ Error: {data.get('error', 'Unknown error')}")
                return False
        else:
            print(f"❌ HTTP Error {response.status_code}")
            return False
    except requests.exceptions.ConnectionError:
        print("❌ Error: Cannot connect to analog API server")
        return False
    except Exception as e:
        print(f"❌ Error: {e}")
        return False

def get_contract(contract_id):
    """Get contract details"""
    try:
        response = requests.get(f'{API_BASE}/contracts/{contract_id}', timeout=10)
        if response.status_code == 200:
            data = response.json()
            if data.get('success'):
                print(f"📄 Contract: {data['name']}")
                print(f"   ID: {data['contract_id']}")
                print(f"   Status: {data['status']}")
                print(f"   Deployed: {data['deployed_at']}")
                print()
                print("Code:")
                print("-" * 60)
                print(data['contract_code'])
                print("-" * 60)
                return True
            else:
                print(f"❌ Error: {data.get('error', 'Contract not found')}")
                return False
        else:
            print(f"❌ HTTP Error {response.status_code}")
            return False
    except requests.exceptions.ConnectionError:
        print("❌ Error: Cannot connect to analog API server")
        return False
    except Exception as e:
        print(f"❌ Error: {e}")
        return False

def check_status():
    """Check consensus status"""
    try:
        response = requests.get(f'{API_BASE}/consensus/status', timeout=10)
        if response.status_code == 200:
            data = response.json()
            if data.get('success'):
                print("⚡ Analog Consensus Status")
                print("=" * 60)
                print(f"  Consensus Type: {data['consensus_type']}")
                print(f"  Evolution Count: {data['evolution_count']:,}")
                print(f"  Consensus Locked: {'🔒 YES' if data['consensus_locked'] else '🔓 NO'}")
                print(f"  Phase Variance: {data['phase_variance'] * 100:.4f}%")
                print(f"  Consensus Steps: {data['consensus_steps']:,}")
                print(f"  Evolution Rate: {data['target_hz']:,} Hz")
                print(f"  Uptime: {data['uptime_seconds']:.1f} seconds")
                print(f"  Dimensions: {data['dimensions']}D Complex")
                print("=" * 60)
                return True
            else:
                print(f"❌ Error: {data.get('error', 'Unknown error')}")
                return False
        else:
            print(f"❌ HTTP Error {response.status_code}")
            return False
    except requests.exceptions.ConnectionError:
        print("❌ Error: Cannot connect to analog API server")
        print("   Make sure the server is running: python analog_api_server.py")
        return False
    except Exception as e:
        print(f"❌ Error: {e}")
        return False

def show_comparison():
    """Show POA vs Analog comparison"""
    try:
        response = requests.get(f'{API_BASE}/comparison/poa', timeout=10)
        if response.status_code == 200:
            data = response.json()
            if data.get('success'):
                comp = data['comparison']

                print("📊 POA vs. Analog Comparison")
                print("=" * 70)
                print()
                print("❌ POA (Proof of Authority) - OBSOLETE:")
                for key, value in comp['poa'].items():
                    print(f"  • {key.title()}: {value}")
                print()
                print("✅ Analog (Mathematical Consensus) - ACTIVE:")
                for key, value in comp['analog'].items():
                    print(f"  • {key.title()}: {value}")
                print()
                print("📈 Improvement Factor:")
                for key, value in comp['improvement_factor'].items():
                    print(f"  • {key.title()}: {value}")
                print()
                print("=" * 70)
                return True
        return False
    except:
        return False

def main():
    parser = argparse.ArgumentParser(
        description='Analog Contract Deployment CLI - Post-Blockchain Consensus System',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  %(prog)s deploy my_contract.ac --name "My Contract"
  %(prog)s list
  %(prog)s get analog_1234567890
  %(prog)s status
  %(prog)s compare
        """
    )

    subparsers = parser.add_subparsers(dest='command', help='Command to execute')

    # Deploy command
    deploy_parser = subparsers.add_parser('deploy', help='Deploy a contract')
    deploy_parser.add_argument('contract_file', help='Path to contract file')
    deploy_parser.add_argument('--name', help='Contract name (optional)')

    # List command
    subparsers.add_parser('list', help='List all deployed contracts')

    # Get command
    get_parser = subparsers.add_parser('get', help='Get contract details')
    get_parser.add_argument('contract_id', help='Contract ID')

    # Status command
    subparsers.add_parser('status', help='Check consensus status')

    # Compare command
    subparsers.add_parser('compare', help='Show POA vs Analog comparison')

    args = parser.parse_args()

    if not args.command:
        parser.print_help()
        sys.exit(1)

    print()
    print("=" * 70)
    print("ANALOG CONTRACT CLI")
    print("Post-Blockchain Consensus System")
    print("=" * 70)
    print()

    success = False

    if args.command == 'deploy':
        success = deploy_contract(args.contract_file, args.name)
    elif args.command == 'list':
        success = list_contracts()
    elif args.command == 'get':
        success = get_contract(args.contract_id)
    elif args.command == 'status':
        success = check_status()
    elif args.command == 'compare':
        success = show_comparison()

    print()
    sys.exit(0 if success else 1)

if __name__ == '__main__':
    main()
