# Migration from POA to Analog Mainnet

## Overview

This document explains the complete migration from **Proof of Authority (POA)** blockchain consensus to **Analog Mathematical Consensus**.

---

## What Changed

### POA (Old System) ❌

- **Consensus**: Authority-based voting with 2-3 validator nodes
- **Timing**: Discrete blocks every 5-15 seconds
- **Finality**: Delayed (multiple confirmations required)
- **Security**: Trust-based (51% validator control)
- **Infrastructure**: LXC containers for validators, RPC nodes, miners
- **Energy**: Mining overhead for block production

### Analog (New System) ✅

- **Consensus**: Mathematical convergence (CV < 0.1%)
- **Timing**: Continuous evolution at 32,768 Hz (30.517 μs period)
- **Finality**: Instant (when consensus locked)
- **Security**: Cryptographic proof via chaotic dynamics
- **Infrastructure**: Single C engine + HTTP API + NetCat sync
- **Energy**: Pure computation (no mining)

### Performance Improvement

**1,000,000× faster finality** (microseconds vs. seconds)

---

## Architecture Changes

### Mathematical Foundation

The analog consensus operates via the recursive formula:

```
Dₙ(r) = √(φ·Fₙ·2ⁿ·Pₙ·Ω)·r^k
```

Where:
- **φ** = Golden ratio (1.618...)
- **Fₙ** = Fibonacci number
- **Pₙ** = nth prime
- **Ω** = Angular frequency (2π × 32,768)
- **r** = 8D complex state vector
- **k** = Recursive depth

### Consensus Detection

Consensus is achieved when:

1. **Phase Variance**: CV < 0.1% across all 8 dimensions
2. **Evolution Stability**: RK4 integrator converges
3. **Deterministic Lock**: SHA-256 validation passes

No validators or authority nodes required.

---

## Removed Components

The following POA infrastructure is **no longer needed**:

### 1. LXC Containers
```bash
# REMOVED:
poa-rpc-lxc       # RPC node
poa-miner1-lxc    # Miner 1
poa-miner2-lxc    # Miner 2
```

### 2. POA Scripts
```bash
# REMOVED:
init-poa-lxc.sh
provision-poa-validators.sh
poa-contract-deploy.js
```

### 3. POA Configuration
```json
// REMOVED: config/poa-config/*
{
  "validators": [...],
  "block_time": 15,
  "consensus": "poa"
}
```

### 4. Test Files
```python
# REMOVED POA-specific tests:
test_integration.py (POA sections)
test_poa_blockchain.py
verify_poa_consensus.py
```

---

## New Components

### 1. Analog Consensus Engine (C)

**Location**: `analog-mainnet/src/`

- `hdgl_bridge_v40.c` — Core 32,768 Hz evolution engine
- `hdgl_http_api.c` — REST API for consensus metrics
- `hdgl_netcat.c` — Phase-coherent peer sync

**Compilation**:
```bash
cd analog-mainnet/src
gcc -o hdgl_bridge hdgl_bridge_v40.c hdgl_http_api.c hdgl_netcat.c \
    -lm -lpthread -lwinmm -lws2_32
```

### 2. Web Services (Python)

**Location**: `analog-mainnet/web/`

- `analog_api_server.py` — Flask API server (port 8080)
- `dashboard.html` — Real-time consensus dashboard

**Start**:
```bash
python analog_api_server.py
```

### 3. Contract Deployment (Python)

**Location**: `analog-mainnet/contracts/`

- `analog_contract_cli.py` — CLI for contract deployment

**Usage**:
```bash
python analog_contract_cli.py deploy my_contract.ac --name "My Contract"
python analog_contract_cli.py status
python analog_contract_cli.py list
```

### 4. Orchestration (Python)

**Location**: `analog-mainnet/orchestration/`

- `orchestrate.py` — Base4096 stream processing

**Generate Config**:
```bash
python orchestrate.py --config-only -o ../config/runtime_config.json
```

---

## Deployment Guide

### Quick Start (One Command)

```bash
cd analog-mainnet
python deploy.py
```

This will:
1. Compile C consensus engine
2. Generate runtime configuration
3. Start consensus engine (32,768 Hz)
4. Start web services (port 8080)
5. Deploy dashboard

### Manual Deployment

#### Step 1: Compile Engine
```bash
cd analog-mainnet/src
gcc -o hdgl_bridge hdgl_bridge_v40.c hdgl_http_api.c hdgl_netcat.c \
    -lm -lpthread -lwinmm -lws2_32
```

#### Step 2: Generate Config
```bash
cd ../orchestration
python orchestrate.py --config-only -o ../config/runtime_config.json
```

#### Step 3: Start Engine
```bash
cd ../src
set HDGL_CONFIG=../config/runtime_config.json
./hdgl_bridge
```

#### Step 4: Start Web Services
```bash
cd ../web
python analog_api_server.py
```

#### Step 5: Access Dashboard
Open browser to: http://localhost:8080

---

## API Endpoints

### Consensus Engine (Port 9999)

```bash
# Get consensus status
curl http://localhost:9999/api/status

# Get Prometheus metrics
curl http://localhost:9999/metrics

# Get NetCat peer status
curl http://localhost:9999/api/netcat
```

### Web Services (Port 8080)

```bash
# Consensus status (via web API)
curl http://localhost:8080/api/consensus/status

# Network peers
curl http://localhost:8080/api/network/peers

# System info
curl http://localhost:8080/api/system/info

# POA comparison
curl http://localhost:8080/api/comparison/poa

# Deploy contract
curl -X POST http://localhost:8080/api/contracts/deploy \
  -H "Content-Type: application/json" \
  -d '{"name": "Test", "contract_code": "..."}'
```

---

## Configuration

### Runtime Configuration

**File**: `analog-mainnet/config/runtime_config.json`

```json
{
  "consensus": {
    "target_hz": 32768,
    "dimensions": 8,
    "cv_threshold": 0.001,
    "phase_sync_threshold": 0.95
  },
  "network": {
    "http_port": 9999,
    "netcat_port": 9095,
    "max_peers": 32
  },
  "legacy": {
    "poa_mode": false,
    "blockchain_compatibility": false,
    "validators": []
  }
}
```

Note: **`poa_mode` is set to `false`** — POA completely disabled.

---

## Contract Deployment

### Analog Contract Format

Analog contracts are **NOT** blockchain transactions. They execute within the analog consensus field.

**Example Contract** (`example.ac`):
```python
# Analog Contract Example
# Executes at 32,768 Hz within consensus field

DIMENSIONS = 8
CV_THRESHOLD = 0.001

def on_consensus_lock():
    """Called when consensus locks (CV < 0.1%)"""
    print("Consensus achieved!")
    return True

def on_evolution_step(state):
    """Called every evolution step (30.517 μs)"""
    # Contract logic here
    return state
```

### Deploy Contract
```bash
python contracts/analog_contract_cli.py deploy example.ac
```

---

## Monitoring

### Dashboard

Access at: http://localhost:8080

Shows real-time:
- Evolution count
- Phase variance (CV)
- Consensus lock status
- Network peers
- Performance metrics

### Logs

```bash
# Consensus engine logs
type logs\consensus_engine.log

# Web services logs
type logs\web_services.log
```

### Metrics (Prometheus)

```bash
curl http://localhost:9999/metrics
```

---

## Comparison Table

| Feature | POA (Old) | Analog (New) | Improvement |
|---------|-----------|--------------|-------------|
| Consensus | Authority voting | Mathematical convergence | Deterministic |
| Timing | 5-15 sec blocks | 30.517 μs evolution | 1,000,000× faster |
| Validators | 2-3 nodes | 0 (deterministic) | No trust needed |
| Finality | Delayed | Instant (when locked) | Immediate |
| Attack Resistance | 51% authority | Reversing chaos | Cryptographic |
| Energy | Mining overhead | Pure computation | Efficient |
| Infrastructure | 3+ LXC containers | 1 C engine | Simplified |

---

## Troubleshooting

### Engine Not Starting

```bash
# Check compilation
cd analog-mainnet/src
gcc -o hdgl_bridge hdgl_bridge_v40.c hdgl_http_api.c hdgl_netcat.c \
    -lm -lpthread -lwinmm -lws2_32

# Check config exists
ls ../config/runtime_config.json

# Run directly
./hdgl_bridge
```

### Web Services Not Responding

```bash
# Check Python dependencies
pip install flask flask-cors requests

# Run directly
cd analog-mainnet/web
python analog_api_server.py
```

### Consensus Not Locking

Check phase variance in dashboard:
- If CV > 0.1%, consensus is still evolving (normal)
- If CV stuck, check RTC sync
- If CV = 0, engine may be frozen

---

## Migration Checklist

- [x] ✅ POA containers removed
- [x] ✅ Analog consensus engine compiled
- [x] ✅ Web services deployed
- [x] ✅ Dashboard accessible
- [x] ✅ Contract deployment working
- [x] ✅ Configuration generated
- [x] ✅ APIs responding
- [x] ✅ Consensus locking

---

## Support

For issues, check:
1. `logs/consensus_engine.log`
2. `logs/web_services.log`
3. Dashboard at http://localhost:8080
4. API status at http://localhost:9999/api/status

---

**Analog Mainnet** — The post-blockchain future, today.
