#!/usr/bin/env python3
"""
Find the equilibrium point where damping = forcing
Then calculate GAMMA to shift that equilibrium to variance = 0.01
"""
import math

# ============================================================================
# CURRENT EQUILIBRIUM STATE
# ============================================================================
print("=" * 70)
print("EQUILIBRIUM ANALYSIS")
print("=" * 70)

# Measured equilibrium
GAMMA_current = 0.02
equilibrium_variance_current = 7.27
forcing_rate = 0.019896  # /sec (measured from previous analysis)

print(f"Current GAMMA: {GAMMA_current}")
print(f"Equilibrium variance: σ²_eq = {equilibrium_variance_current:.2f}")
print(f"Forcing rate: {forcing_rate:.6f} /sec")

# At equilibrium: forcing = damping
# But variance oscillates around a mean, so the forcing is variance-dependent
#
# Model: forcing = f₀ + f₁*σ² (forcing increases with variance)
#        damping = GAMMA (constant damping rate)
#
# At equilibrium: f₀ + f₁*σ²_eq = GAMMA
# With current values: f₀ + f₁*(7.27) = 0.02

# ============================================================================
# TARGET EQUILIBRIUM
# ============================================================================
target_variance = 0.01
print(f"\nTarget equilibrium variance: σ²_target = {target_variance}")

# Assume forcing scales linearly with variance:
# forcing(σ²) = baseline + k*σ²
#
# At current equilibrium: baseline + k*(7.27) = 0.02
# We need: baseline + k*(0.01) = GAMMA_new
#
# If we assume baseline is small and forcing ∝ σ²:
# forcing(σ²) ≈ (forcing_rate / σ²_current) * σ²
#            ≈ (0.019896 / 7.27) * σ²
#            ≈ 0.002737 * σ²

forcing_per_variance = forcing_rate / equilibrium_variance_current
print(f"\nAssuming forcing ∝ variance:")
print(f"  forcing(σ²) ≈ {forcing_per_variance:.6f} × σ²")

target_forcing = forcing_per_variance * target_variance
required_GAMMA = target_forcing
print(f"\nAt σ² = {target_variance}:")
print(f"  forcing = {target_forcing:.6f} /sec")
print(f"  Required GAMMA = {required_GAMMA:.6f}")

print(f"\n🎯 CHANGE GAMMA FROM {GAMMA_current} TO {required_GAMMA:.6f}")
print(f"   Decrease by factor: {required_GAMMA / GAMMA_current:.4f}×")

# ============================================================================
# ALTERNATIVE: FORCING IS CONSTANT, DAMPING NEEDS TO OVERWHELM IT
# ============================================================================
print(f"\n{'='*70}")
print("ALTERNATIVE: If forcing is NOT variance-dependent")
print("=" * 70)

# If forcing is constant at 0.019896 /sec regardless of variance,
# then we need GAMMA >> forcing to drive variance down
#
# Model: dσ²/dt = forcing - GAMMA*σ²
# Equilibrium: σ²_eq = forcing / GAMMA
#
# Current: σ²_eq = 0.019896 / 0.02 = 0.995 (but we measure 7.27!)
# This means forcing is NOT constant!

calculated_eq = forcing_rate / GAMMA_current
print(f"If forcing were constant:")
print(f"  Predicted equilibrium: σ²_eq = {calculated_eq:.3f}")
print(f"  Actual equilibrium: σ²_eq = {equilibrium_variance_current:.2f}")
print(f"  Mismatch: {equilibrium_variance_current / calculated_eq:.1f}×")

print(f"\nConclusion: Forcing INCREASES with variance!")
print(f"This is expected - larger variance → larger dimensions → stronger coupling")

# ============================================================================
# CORRECT MODEL: FORCING ∝ σ²
# ============================================================================
print(f"\n{'='*70}")
print("CORRECT MODEL: dσ²/dt = k*σ² - GAMMA*σ²")
print("=" * 70)

# dσ²/dt = k*σ² - GAMMA*σ² = (k - GAMMA)*σ²
# At equilibrium: k = GAMMA
# So: k = 0.02 / (some factor related to current variance)
#
# Actually, the equation should be:
# dσ²/dt = a*σ² - GAMMA*b*σ²
# where a = forcing coefficient, b = damping effectiveness
#
# At equilibrium: a*σ² = GAMMA*b*σ²
# So: GAMMA = a/b

# From measurements:
# - Forcing adds ~0.02 /sec when σ² ≈ 7.27
# - Damping removes ~0.02 /sec when σ² ≈ 7.27
# - Net effect: σ² oscillates but doesn't change on average

# The forcing per unit variance:
a = forcing_rate / equilibrium_variance_current  # ≈ 0.00274 per variance unit
print(f"Forcing coefficient: a = {a:.6f}")
print(f"  (forcing = a × σ²)")

# The damping per unit variance:
b = GAMMA_current / equilibrium_variance_current  # ≈ 0.00275 per variance unit
print(f"Current damping per variance: GAMMA/σ² = {b:.6f}")

# To achieve σ²_target = 0.01:
# We need: a*0.01 = GAMMA_new*b_new*0.01
# If b stays proportional to 1/σ² (i.e., damping effectiveness is constant):
# Then: GAMMA_new = a*(σ²_target) / (GAMMA/σ²_current)*(σ²_target)
#     = a / (GAMMA/σ²_current)
#     = a * σ²_current / GAMMA

required_GAMMA_v2 = a * equilibrium_variance_current / GAMMA_current
print(f"\nRequired GAMMA (method 2): {required_GAMMA_v2:.6f}")
print(f"  = {a:.6f} × {equilibrium_variance_current:.2f} / {GAMMA_current}")
print(f"  = {required_GAMMA_v2:.6f}")

# Hmm, that gives us the same GAMMA. That can't be right.
# The issue is we want to CHANGE the equilibrium point!

# ============================================================================
# FINAL CORRECT APPROACH
# ============================================================================
print(f"\n{'='*70}")
print("FINAL APPROACH: Shift equilibrium by changing GAMMA")
print("=" * 70)

# At equilibrium: forcing(σ²) = damping(σ², GAMMA)
#
# If forcing(σ²) = k*σ² (linear in variance)
# And damping(σ², GAMMA) = GAMMA*σ² (linear in both)
# Then: k*σ² = GAMMA*σ²
#       k = GAMMA
#       σ² cancels out - any σ² is an equilibrium!
#
# This means forcing MUST have a constant term:
# forcing(σ²) = f₀ + k*σ²
#
# At equilibrium: f₀ + k*σ²_eq = GAMMA*σ²_eq
#                f₀ = σ²_eq*(GAMMA - k)

# From current state:
# f₀ + k*7.27 = 0.02*7.27
# f₀ = 7.27*(0.02 - k)

# If k ≈ 0 (forcing is mostly constant):
# f₀ ≈ 7.27*0.02 = 0.1454

# For new equilibrium at σ²=0.01:
# f₀ + k*0.01 = GAMMA_new*0.01
# 0.1454 + k*0.01 = GAMMA_new*0.01
# GAMMA_new = 0.1454/0.01 + k = 14.54 + k

print(f"\nIf forcing has constant component f₀:")
print(f"  f₀ ≈ {equilibrium_variance_current * GAMMA_current:.4f}")
print(f"  To achieve σ²={target_variance}:")
print(f"  GAMMA_new ≈ f₀/σ²_target = {equilibrium_variance_current * GAMMA_current / target_variance:.2f}")

required_GAMMA_final = equilibrium_variance_current * GAMMA_current / target_variance
print(f"\n🎯 FINAL ANSWER: GAMMA = {required_GAMMA_final:.4f}")
print(f"   (Increase from {GAMMA_current} by {required_GAMMA_final/GAMMA_current:.1f}×)")
