#!/usr/bin/env python3
"""
Mock Analog Consensus Engine
Simulates the C engine for testing without compilation
"""

from flask import Flask, jsonify
import time
import math
import random

app = Flask(__name__)

# State
start_time = time.time()
evolution_count = 0
consensus_count = 0
locked = False
phase_variance = 0.05  # Start above threshold

def update_state():
    """Update simulated consensus state"""
    global evolution_count, consensus_count, locked, phase_variance

    # Simulate 32,768 Hz evolution
    evolution_count += 1

    # Gradually reduce phase variance (simulate convergence)
    if phase_variance > 0.0001:
        phase_variance *= 0.9999
        phase_variance += random.uniform(-0.0001, 0.0001)
        phase_variance = max(0, phase_variance)

    # Lock when CV < 0.1%
    if phase_variance < 0.001 and not locked:
        locked = True
        consensus_count += 1
    elif phase_variance >= 0.001:
        locked = False

@app.route('/api/status')
def status():
    """Consensus status endpoint"""
    update_state()

    return jsonify({
        'evolution_count': evolution_count,
        'consensus_count': consensus_count,
        'locked': locked,
        'phase_variance': phase_variance,
        'target_hz': 32768,
        'uptime_seconds': time.time() - start_time,
        'dimensions': 8
    })

@app.route('/metrics')
def metrics():
    """Prometheus metrics endpoint"""
    update_state()

    return f"""# HELP hdgl_evolution_count Total evolution steps
# TYPE hdgl_evolution_count counter
hdgl_evolution_count {evolution_count}

# HELP hdgl_consensus_count Total consensus locks
# TYPE hdgl_consensus_count counter
hdgl_consensus_count {consensus_count}

# HELP hdgl_phase_variance Current phase variance (CV)
# TYPE hdgl_phase_variance gauge
hdgl_phase_variance {phase_variance:.6f}

# HELP hdgl_locked Consensus lock status (1=locked, 0=unlocked)
# TYPE hdgl_locked gauge
hdgl_locked {1 if locked else 0}

# HELP hdgl_target_hz Target evolution frequency
# TYPE hdgl_target_hz gauge
hdgl_target_hz 32768
""", 200, {'Content-Type': 'text/plain'}

@app.route('/api/netcat')
def netcat():
    """NetCat peer status endpoint"""
    update_state()

    return jsonify({
        'connected_peers': random.randint(2, 5),
        'phase_sync': 0.95 + random.uniform(0, 0.05),
        'consensus_reached': locked,
        'peers': [
            {'id': 'peer1', 'phase_sync': 0.96},
            {'id': 'peer2', 'phase_sync': 0.98},
            {'id': 'peer3', 'phase_sync': 0.95}
        ]
    })

@app.route('/health')
def health():
    """Health check endpoint"""
    return jsonify({
        'status': 'healthy',
        'timestamp': time.time()
    })

if __name__ == '__main__':
    print("=" * 70)
    print("MOCK ANALOG CONSENSUS ENGINE")
    print("Simulating 32,768 Hz evolution for testing")
    print("=" * 70)
    print("\nEndpoints:")
    print("  • Status: http://localhost:9999/api/status")
    print("  • Metrics: http://localhost:9999/metrics")
    print("  • NetCat: http://localhost:9999/api/netcat")
    print("\n" + "=" * 70)

    app.run(host='0.0.0.0', port=9999, debug=False)
