#!/usr/bin/env pwsh
# Monitor convergence of hybrid feedback system

Write-Host "`n🔬 Hybrid Analog-Digital Feedback Convergence Monitor" -ForegroundColor Cyan
Write-Host "=" * 70 -ForegroundColor DarkGray
Write-Host "Monitoring phase variance for convergence..."
Write-Host "Expected: Exponential decay from ~6.0 → ~0.001"
Write-Host "Target: Consensus lock when variance < 0.001" -ForegroundColor Yellow
Write-Host "=" * 70 -ForegroundColor DarkGray
Write-Host ""

$startTime = Get-Date
$samples = @()
$lockAchieved = $false

for ($i = 0; $i -lt 60; $i++) {
    Start-Sleep -Seconds 1

    try {
        $response = Invoke-WebRequest -Uri "http://localhost:9999/api/status" -UseBasicParsing -ErrorAction SilentlyContinue

        if ($response) {
            $data = $response.Content | ConvertFrom-Json
            $variance = [double]$data.phase_variance
            $locked = [int]$data.locked
            $evolution = [long]$data.evolution_count

            $samples += $variance

            # Calculate trend (last 10 samples)
            $trend = ""
            if ($samples.Count -ge 10) {
                $recent = $samples[-10..-1]
                $avg_first5 = ($recent[0..4] | Measure-Object -Average).Average
                $avg_last5 = ($recent[5..9] | Measure-Object -Average).Average

                if ($avg_last5 -lt $avg_first5 * 0.9) {
                    $trend = "📉 Converging"
                } elseif ($avg_last5 -gt $avg_first5 * 1.1) {
                    $trend = "📈 Diverging"
                } else {
                    $trend = "━ Stable"
                }
            }

            $status = if ($locked -eq 1) { "🔒 LOCKED" } else { "🔓 Unlocked" }

            $output = "[t={0:D3}s] evo={1:D9} | variance={2:F6} | {3} | {4}" -f $i, $evolution, $variance, $status, $trend
            Write-Host $output            if ($locked -eq 1 -and -not $lockAchieved) {
                $lockTime = (Get-Date) - $startTime
                Write-Host "`n✅ CONSENSUS LOCKED after $($lockTime.TotalSeconds.ToString('F1')) seconds!" -ForegroundColor Green
                $lockAchieved = $true
            }
        }
    }
    catch {
        Write-Host "[t=$($i)s] ⚠ Unable to connect to consensus engine" -ForegroundColor DarkYellow
    }
}

Write-Host "`n" + ("=" * 70) -ForegroundColor DarkGray
Write-Host "📊 Analysis Summary" -ForegroundColor Cyan
Write-Host ("=" * 70) -ForegroundColor DarkGray

if ($samples.Count -gt 0) {
    $avgVariance = ($samples | Measure-Object -Average).Average
    $minVariance = ($samples | Measure-Object -Minimum).Minimum
    $maxVariance = ($samples | Measure-Object -Maximum).Maximum

    Write-Host "Samples collected: $($samples.Count)"
    Write-Host "Average variance: $($avgVariance.ToString('F6'))"
    Write-Host "Min variance: $($minVariance.ToString('F6'))"
    Write-Host "Max variance: $($maxVariance.ToString('F6'))"
    Write-Host "Range: $($maxVariance.ToString('F6')) → $($minVariance.ToString('F6'))"

    # Check for convergence trend
    if ($samples.Count -ge 20) {
        $first10 = ($samples[0..9] | Measure-Object -Average).Average
        $last10 = ($samples[-10..-1] | Measure-Object -Average).Average
        $change = (($last10 - $first10) / $first10) * 100

        Write-Host "`nConvergence Analysis:"
        Write-Host "  First 10s average: $($first10.ToString('F6'))"
        Write-Host "  Last 10s average:  $($last10.ToString('F6'))"
        Write-Host "  Change: $($change.ToString('F2'))%"

        if ($change -lt -10) {
            Write-Host "`n✅ System is CONVERGING (hybrid feedback active)" -ForegroundColor Green
        } elseif ($change -gt 10) {
            Write-Host "`n⚠ System is DIVERGING (check feedback parameters)" -ForegroundColor Yellow
        } else {
            Write-Host "`n⚠ System is OSCILLATING (pure analog behavior)" -ForegroundColor Yellow
            Write-Host "   Hybrid feedback may be too weak or not active" -ForegroundColor DarkGray
        }
    }

    if ($lockAchieved) {
        Write-Host "`n🎯 Result: CONSENSUS ACHIEVED!" -ForegroundColor Green
    } else {
        Write-Host "`n⏳ Result: Convergence in progress (needs more time)" -ForegroundColor Yellow

        # Estimate time to convergence
        if ($samples.Count -ge 30 -and $last10 -lt $first10) {
            $rate = ($first10 - $last10) / 50.0  # Change per second
            $timeToThreshold = ($last10 - 0.001) / $rate
            if ($timeToThreshold -gt 0) {
                Write-Host "   Estimated time to threshold: $($timeToThreshold.ToString('F0')) seconds" -ForegroundColor DarkGray
            }
        }
    }
}

Write-Host ""
