#!/usr/bin/env bash
# Production Test Suite (C + Docker/LXC)
# Tests live consensus engine instead of Python simulation

set -e

TEST_MODE=${1:-docker}  # docker or lxc
TOTAL_TESTS=0
PASSED_TESTS=0
FAILED_TESTS=0

print_test() {
    local name=$1
    local result=$2
    local message=$3

    TOTAL_TESTS=$((TOTAL_TESTS + 1))

    if [ "$result" = "PASS" ]; then
        echo "✅ $name"
        PASSED_TESTS=$((PASSED_TESTS + 1))
    else
        echo "❌ $name: $message"
        FAILED_TESTS=$((FAILED_TESTS + 1))
    fi
}

echo "========================================================================"
echo "  ANALOG MAINNET PRODUCTION TEST SUITE"
echo "  Testing Live C Consensus Engine"
echo "  Mode: $TEST_MODE"
echo "========================================================================"

# Set URLs based on deployment mode
if [ "$TEST_MODE" = "docker" ]; then
    PRIMARY_URL="http://localhost:9999"
    PEER1_URL="http://localhost:10001"
    PEER2_URL="http://localhost:10002"
elif [ "$TEST_MODE" = "lxc" ]; then
    PRIMARY_URL="http://10.100.0.10:9999"
    PEER1_URL="http://10.100.0.11:9999"
    PEER2_URL="http://10.100.0.12:9999"
else
    echo "❌ Invalid test mode: $TEST_MODE"
    exit 1
fi

# Test 1: Primary node responding
echo ""
echo "[1/8] Testing primary consensus node..."
if curl -sf "$PRIMARY_URL/api/status" > /dev/null 2>&1; then
    print_test "Primary node HTTP API" "PASS"
else
    print_test "Primary node HTTP API" "FAIL" "Not responding"
fi

# Test 2: Peer nodes responding
echo ""
echo "[2/8] Testing peer nodes..."
if curl -sf "$PEER1_URL/api/status" > /dev/null 2>&1; then
    print_test "Peer 1 HTTP API" "PASS"
else
    print_test "Peer 1 HTTP API" "FAIL" "Not responding"
fi

if curl -sf "$PEER2_URL/api/status" > /dev/null 2>&1; then
    print_test "Peer 2 HTTP API" "PASS"
else
    print_test "Peer 2 HTTP API" "FAIL" "Not responding"
fi

# Test 3: Consensus data structure
echo ""
echo "[3/8] Testing consensus data structure..."
response=$(curl -sf "$PRIMARY_URL/api/status" 2>/dev/null)
if echo "$response" | grep -q "evolution_count"; then
    print_test "Field: evolution_count" "PASS"
else
    print_test "Field: evolution_count" "FAIL" "Missing"
fi

if echo "$response" | grep -q "phase_variance"; then
    print_test "Field: phase_variance" "PASS"
else
    print_test "Field: phase_variance" "FAIL" "Missing"
fi

if echo "$response" | grep -q "consensus_count"; then
    print_test "Field: consensus_count" "PASS"
else
    print_test "Field: consensus_count" "FAIL" "Missing"
fi

if echo "$response" | grep -q "locked"; then
    print_test "Field: locked" "PASS"
else
    print_test "Field: locked" "FAIL" "Missing"
fi

# Test 4: Consensus parameters
echo ""
echo "[4/8] Testing consensus parameters..."
target_hz=$(echo "$response" | grep -o '"target_hz":[0-9]*' | cut -d: -f2)
if [ "$target_hz" = "32768" ]; then
    print_test "Target Hz = 32768" "PASS"
else
    print_test "Target Hz = 32768" "FAIL" "Got $target_hz"
fi

dimensions=$(echo "$response" | grep -o '"dimensions":[0-9]*' | cut -d: -f2)
if [ "$dimensions" = "8" ]; then
    print_test "Dimensions = 8" "PASS"
else
    print_test "Dimensions = 8" "FAIL" "Got $dimensions"
fi

# Test 5: Evolution is running
echo ""
echo "[5/8] Testing live evolution..."
evolution1=$(echo "$response" | grep -o '"evolution_count":[0-9]*' | cut -d: -f2)
sleep 1
response2=$(curl -sf "$PRIMARY_URL/api/status" 2>/dev/null)
evolution2=$(echo "$response2" | grep -o '"evolution_count":[0-9]*' | cut -d: -f2)

if [ "$evolution2" -gt "$evolution1" ]; then
    print_test "Evolution progressing" "PASS"
    delta=$((evolution2 - evolution1))
    echo "  Evolution delta: $delta steps/second"
else
    print_test "Evolution progressing" "FAIL" "Stalled at $evolution1"
fi

# Test 6: Prometheus metrics
echo ""
echo "[6/8] Testing Prometheus metrics..."
if curl -sf "$PRIMARY_URL/metrics" | grep -q "hdgl_evolution_count"; then
    print_test "Prometheus metrics endpoint" "PASS"
else
    print_test "Prometheus metrics endpoint" "FAIL" "Not available"
fi

# Test 7: NetCat peer synchronization
echo ""
echo "[7/8] Testing peer synchronization..."
if curl -sf "$PRIMARY_URL/api/netcat" > /dev/null 2>&1; then
    print_test "NetCat sync endpoint" "PASS"

    netcat_response=$(curl -sf "$PRIMARY_URL/api/netcat" 2>/dev/null)
    peer_count=$(echo "$netcat_response" | grep -o '"connected_peers":[0-9]*' | cut -d: -f2)

    if [ -n "$peer_count" ] && [ "$peer_count" -gt 0 ]; then
        print_test "Peers connected: $peer_count" "PASS"
    else
        print_test "Peers connected" "FAIL" "No peers"
    fi
else
    print_test "NetCat sync endpoint" "FAIL" "Not responding"
fi

# Test 8: Health checks
echo ""
echo "[8/8] Testing health endpoints..."
if curl -sf "$PRIMARY_URL/health" > /dev/null 2>&1; then
    print_test "Primary node health" "PASS"
else
    print_test "Primary node health" "FAIL" "Not healthy"
fi

# Summary
echo ""
echo "========================================================================"
echo "  TEST SUMMARY"
echo "========================================================================"
echo ""
echo "✅ Passed:  $PASSED_TESTS/$TOTAL_TESTS"
echo "❌ Failed:  $FAILED_TESTS/$TOTAL_TESTS"

if [ $FAILED_TESTS -eq 0 ]; then
    PERCENTAGE=100
else
    PERCENTAGE=$((PASSED_TESTS * 100 / TOTAL_TESTS))
fi
echo "📊 Coverage: ${PERCENTAGE}%"

echo ""
if [ $FAILED_TESTS -eq 0 ]; then
    echo "🎉 ALL TESTS PASSED - PRODUCTION SYSTEM OPERATIONAL"
    echo ""
    echo "✅ Live C consensus engine running at 32,768 Hz"
    echo "✅ All nodes responding"
    echo "✅ Peer synchronization active"
    echo "✅ Evolution progressing"
    echo "✅ No Python simulation - pure production code"
    exit 0
else
    echo "⚠️  $FAILED_TESTS test(s) failed - check deployment"
    exit 1
fi
