#!/usr/bin/env python3
"""
Analog Mainnet Integration Test
Tests full stack with mock consensus engine
"""

import sys
import time
import subprocess
import requests
from pathlib import Path
import signal

class IntegrationTester:
    def __init__(self):
        self.root = Path(__file__).parent
        self.processes = []
        self.passed = 0
        self.failed = 0

    def cleanup(self):
        """Cleanup all processes"""
        print("\n🛑 Stopping all services...")
        for name, proc in self.processes:
            try:
                proc.terminate()
                proc.wait(timeout=3)
                print(f"   ✅ Stopped {name}")
            except:
                try:
                    proc.kill()
                except:
                    pass

    def print_test(self, name, passed, message=""):
        """Print test result"""
        if passed:
            print(f"✅ {name}")
            self.passed += 1
        else:
            print(f"❌ {name}: {message}")
            self.failed += 1

    def start_mock_engine(self):
        """Start mock consensus engine"""
        print("\n[1/3] Starting mock consensus engine...")

        engine_path = self.root / "mock_consensus_engine.py"
        if not engine_path.exists():
            print("❌ Mock engine not found")
            return False

        proc = subprocess.Popen(
            [sys.executable, str(engine_path)],
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT,
            creationflags=subprocess.CREATE_NEW_PROCESS_GROUP if sys.platform == 'win32' else 0
        )

        self.processes.append(("Mock Engine", proc))

        # Wait for engine to start
        for i in range(10):
            try:
                response = requests.get('http://localhost:9999/api/status', timeout=1)
                if response.status_code == 200:
                    print("✅ Mock engine running on port 9999")
                    return True
            except:
                time.sleep(0.5)

        print("❌ Mock engine failed to start")
        return False

    def start_web_services(self):
        """Start web API server"""
        print("\n[2/3] Starting web services...")

        server_path = self.root / "web" / "analog_api_server.py"
        if not server_path.exists():
            print("❌ Web server not found")
            return False

        proc = subprocess.Popen(
            [sys.executable, str(server_path)],
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT,
            creationflags=subprocess.CREATE_NEW_PROCESS_GROUP if sys.platform == 'win32' else 0
        )

        self.processes.append(("Web Services", proc))

        # Wait for web services to start
        for i in range(10):
            try:
                response = requests.get('http://localhost:8080/health', timeout=1)
                if response.status_code == 200:
                    print("✅ Web services running on port 8080")
                    return True
            except:
                time.sleep(0.5)

        print("❌ Web services failed to start")
        return False

    def test_endpoints(self):
        """Test all API endpoints"""
        print("\n[3/3] Testing endpoints...")

        tests = [
            ("Consensus status", "http://localhost:9999/api/status"),
            ("Prometheus metrics", "http://localhost:9999/metrics"),
            ("NetCat peers", "http://localhost:9999/api/netcat"),
            ("Web consensus status", "http://localhost:8080/api/consensus/status"),
            ("Network peers", "http://localhost:8080/api/network/peers"),
            ("System info", "http://localhost:8080/api/system/info"),
            ("POA comparison", "http://localhost:8080/api/comparison/poa"),
        ]

        for name, url in tests:
            try:
                response = requests.get(url, timeout=2)
                self.print_test(name, response.status_code == 200)
            except Exception as e:
                self.print_test(name, False, str(e)[:50])

    def test_contract_deployment(self):
        """Test contract deployment API"""
        print("\n📝 Testing contract deployment...")

        try:
            response = requests.post(
                'http://localhost:8080/api/contracts/deploy',
                json={
                    'name': 'Test Contract',
                    'contract_code': 'print("Hello from analog contract")'
                },
                timeout=2
            )

            if response.status_code == 200:
                data = response.json()
                if data.get('success'):
                    print(f"✅ Contract deployed: {data.get('contract_id')}")
                    self.passed += 1
                else:
                    print(f"❌ Contract deployment failed: {data.get('error')}")
                    self.failed += 1
            else:
                print(f"❌ HTTP {response.status_code}")
                self.failed += 1
        except Exception as e:
            print(f"❌ Contract deployment error: {e}")
            self.failed += 1

    def test_consensus_data(self):
        """Test consensus data accuracy"""
        print("\n📊 Testing consensus data...")

        try:
            response = requests.get('http://localhost:9999/api/status', timeout=2)
            data = response.json()

            # Check required fields
            required_fields = [
                'evolution_count',
                'consensus_count',
                'locked',
                'phase_variance',
                'target_hz',
                'uptime_seconds',
                'dimensions'
            ]

            for field in required_fields:
                if field in data:
                    self.print_test(f"Field: {field}", True)
                else:
                    self.print_test(f"Field: {field}", False, "Missing")

            # Verify values
            if data.get('target_hz') == 32768:
                self.print_test("Target Hz = 32,768", True)
            else:
                self.print_test("Target Hz = 32,768", False, f"Got {data.get('target_hz')}")

            if data.get('dimensions') == 8:
                self.print_test("Dimensions = 8", True)
            else:
                self.print_test("Dimensions = 8", False, f"Got {data.get('dimensions')}")

        except Exception as e:
            print(f"❌ Consensus data test error: {e}")
            self.failed += 1

    def run_all_tests(self):
        """Run complete integration test"""
        print("=" * 70)
        print("  ANALOG MAINNET INTEGRATION TEST")
        print("  Full stack test with mock consensus engine")
        print("=" * 70)

        try:
            # Start services
            if not self.start_mock_engine():
                return False

            if not self.start_web_services():
                return False

            # Run tests
            self.test_endpoints()
            self.test_contract_deployment()
            self.test_consensus_data()

            # Print summary
            print("\n" + "=" * 70)
            print("  TEST SUMMARY")
            print("=" * 70)
            print(f"\n✅ Passed: {self.passed}")
            print(f"❌ Failed: {self.failed}")

            if self.failed == 0:
                print("\n🎉 ALL INTEGRATION TESTS PASSED!")
                print("\n📊 Services Running:")
                print("  • Mock Consensus Engine: http://localhost:9999")
                print("  • Web Services: http://localhost:8080")
                print("  • Dashboard: http://localhost:8080/dashboard.html")
                print("\n⏳ Keeping services running for manual testing...")
                print("   Press Ctrl+C to stop")

                # Keep running for manual testing
                try:
                    while True:
                        time.sleep(1)
                except KeyboardInterrupt:
                    pass

                return True
            else:
                print(f"\n❌ {self.failed} test(s) failed")
                return False

        finally:
            self.cleanup()

def main():
    tester = IntegrationTester()
    success = tester.run_all_tests()
    sys.exit(0 if success else 1)

if __name__ == '__main__':
    main()
