#!/usr/bin/env python3
"""
Analog Mainnet Verification & Test Suite
Tests all components without requiring C compilation
"""

import sys
import json
from pathlib import Path
import importlib.util

class AnalogMainnetTester:
    def __init__(self):
        self.root = Path(__file__).parent
        self.passed = 0
        self.failed = 0
        self.warnings = 0

    def print_header(self, text):
        print("\n" + "=" * 70)
        print(f"  {text}")
        print("=" * 70)

    def print_test(self, name, status, message=""):
        symbols = {"PASS": "✅", "FAIL": "❌", "WARN": "⚠️", "INFO": "ℹ️"}
        symbol = symbols.get(status, "•")
        print(f"{symbol} {name}: {status}", end="")
        if message:
            print(f" - {message}")
        else:
            print()

        if status == "PASS":
            self.passed += 1
        elif status == "FAIL":
            self.failed += 1
        elif status == "WARN":
            self.warnings += 1

    def test_directory_structure(self):
        """Test that all required directories exist"""
        self.print_header("Testing Directory Structure")

        required_dirs = [
            "src",
            "web",
            "contracts",
            "config",
            "orchestration",
            "provisioning",
            "docs"
        ]

        for dir_name in required_dirs:
            dir_path = self.root / dir_name
            if dir_path.exists():
                self.print_test(f"Directory: {dir_name}", "PASS")
            else:
                self.print_test(f"Directory: {dir_name}", "FAIL", "Missing")

    def test_core_files(self):
        """Test that all core files exist"""
        self.print_header("Testing Core Files")

        required_files = [
            ("deploy.py", "Deployment script"),
            ("README.md", "Documentation"),
            ("QUICK_START.md", "Quick start guide"),
            ("DEPLOYMENT_COMPLETE.md", "Deployment summary"),
            ("src/hdgl_bridge_v40.c", "Core C engine"),
            ("src/hdgl_http_api.c", "HTTP API module"),
            ("src/hdgl_netcat.c", "NetCat module"),
            ("web/analog_api_server.py", "Web API server"),
            ("web/dashboard.html", "Dashboard"),
            ("contracts/analog_contract_cli.py", "Contract CLI"),
            ("contracts/example_contract.ac", "Example contract"),
            ("orchestration/orchestrate.py", "Orchestration tool"),
            ("docs/MIGRATION_GUIDE.md", "Migration guide")
        ]

        for file_path, description in required_files:
            full_path = self.root / file_path
            if full_path.exists():
                self.print_test(f"{description} ({file_path})", "PASS")
            else:
                self.print_test(f"{description} ({file_path})", "FAIL", "Missing")

    def test_python_modules(self):
        """Test that Python modules can be imported"""
        self.print_header("Testing Python Modules")

        modules = [
            ("web/analog_api_server.py", "Web API Server"),
            ("orchestration/orchestrate.py", "Orchestration"),
            ("contracts/analog_contract_cli.py", "Contract CLI")
        ]

        for module_path, name in modules:
            full_path = self.root / module_path
            if not full_path.exists():
                self.print_test(f"{name} import", "FAIL", "File not found")
                continue

            try:
                # Load module without executing
                spec = importlib.util.spec_from_file_location("test_module", full_path)
                module = importlib.util.module_from_spec(spec)
                self.print_test(f"{name} import", "PASS")
            except Exception as e:
                self.print_test(f"{name} import", "FAIL", str(e)[:50])

    def test_orchestration(self):
        """Test orchestration configuration generation"""
        self.print_header("Testing Orchestration")

        orchestrate_path = self.root / "orchestration" / "orchestrate.py"
        if not orchestrate_path.exists():
            self.print_test("Orchestration script", "FAIL", "Not found")
            return

        # Test config generation
        try:
            import subprocess
            result = subprocess.run(
                [sys.executable, str(orchestrate_path), "--config-only"],
                capture_output=True,
                text=True,
                timeout=5
            )

            if result.returncode == 0:
                # Parse JSON output
                try:
                    config = json.loads(result.stdout)
                    self.print_test("Config generation", "PASS")

                    # Verify config structure
                    if "consensus" in config:
                        self.print_test("Config: consensus section", "PASS")
                    else:
                        self.print_test("Config: consensus section", "FAIL")

                    if "network" in config:
                        self.print_test("Config: network section", "PASS")
                    else:
                        self.print_test("Config: network section", "FAIL")

                    if config.get("legacy", {}).get("poa_mode") == False:
                        self.print_test("Config: POA disabled", "PASS")
                    else:
                        self.print_test("Config: POA disabled", "FAIL", "POA still enabled")

                except json.JSONDecodeError:
                    self.print_test("Config JSON parsing", "FAIL", "Invalid JSON")
            else:
                self.print_test("Config generation", "FAIL", result.stderr[:50])

        except subprocess.TimeoutExpired:
            self.print_test("Config generation", "FAIL", "Timeout")
        except Exception as e:
            self.print_test("Config generation", "FAIL", str(e)[:50])

    def test_contract_cli(self):
        """Test contract CLI functionality"""
        self.print_header("Testing Contract CLI")

        cli_path = self.root / "contracts" / "analog_contract_cli.py"
        if not cli_path.exists():
            self.print_test("Contract CLI", "FAIL", "Not found")
            return

        # Test CLI help
        try:
            import subprocess
            result = subprocess.run(
                [sys.executable, str(cli_path)],
                capture_output=True,
                text=True,
                timeout=5
            )

            # Should show help and exit with code 1 (no command)
            if "deploy" in result.stdout.lower() and "list" in result.stdout.lower():
                self.print_test("CLI help text", "PASS")
            else:
                self.print_test("CLI help text", "WARN", "Help may be incomplete")

        except subprocess.TimeoutExpired:
            self.print_test("CLI help", "FAIL", "Timeout")
        except Exception as e:
            self.print_test("CLI help", "FAIL", str(e)[:50])

    def test_example_contract(self):
        """Test example contract syntax"""
        self.print_header("Testing Example Contract")

        contract_path = self.root / "contracts" / "example_contract.ac"
        if not contract_path.exists():
            self.print_test("Example contract", "FAIL", "Not found")
            return

        try:
            with open(contract_path, 'r') as f:
                contract_code = f.read()

            # Check for required elements
            if "class AnalogContract" in contract_code:
                self.print_test("Contract class definition", "PASS")
            else:
                self.print_test("Contract class definition", "FAIL")

            if "on_consensus_lock" in contract_code:
                self.print_test("Contract: on_consensus_lock", "PASS")
            else:
                self.print_test("Contract: on_consensus_lock", "FAIL")

            if "on_evolution_step" in contract_code:
                self.print_test("Contract: on_evolution_step", "PASS")
            else:
                self.print_test("Contract: on_evolution_step", "FAIL")

            # Test if it's valid Python
            try:
                compile(contract_code, contract_path, 'exec')
                self.print_test("Contract syntax", "PASS")
            except SyntaxError as e:
                self.print_test("Contract syntax", "FAIL", str(e)[:50])

        except Exception as e:
            self.print_test("Example contract", "FAIL", str(e)[:50])

    def test_c_source_files(self):
        """Test C source files exist and have key functions"""
        self.print_header("Testing C Source Files")

        c_files = [
            ("src/hdgl_bridge_v40.c", ["is_at_equilibrium", "main", "phase_variance"]),
            ("src/hdgl_http_api.c", ["http_api_thread", "handle_status"]),
            ("src/hdgl_netcat.c", ["netcat_server_thread", "phase_sync"])
        ]

        for file_path, required_functions in c_files:
            full_path = self.root / file_path
            if not full_path.exists():
                self.print_test(f"C file: {file_path}", "FAIL", "Not found")
                continue

            try:
                with open(full_path, 'r', encoding='utf-8', errors='ignore') as f:
                    content = f.read()

                self.print_test(f"C file: {file_path}", "PASS")

                # Check for key functions
                for func in required_functions:
                    if func in content:
                        self.print_test(f"  Function: {func}", "PASS")
                    else:
                        self.print_test(f"  Function: {func}", "WARN", "Not found")

            except Exception as e:
                self.print_test(f"C file: {file_path}", "FAIL", str(e)[:50])

    def test_dashboard(self):
        """Test dashboard HTML"""
        self.print_header("Testing Dashboard")

        dashboard_path = self.root / "web" / "dashboard.html"
        if not dashboard_path.exists():
            self.print_test("Dashboard HTML", "FAIL", "Not found")
            return

        try:
            with open(dashboard_path, 'r', encoding='utf-8') as f:
                html = f.read()

            # Check for key elements
            checks = [
                ("<!DOCTYPE html>", "HTML5 doctype"),
                ("Analog Mainnet", "Title"),
                ("evolutionCount", "Evolution count display"),
                ("phaseVariance", "Phase variance display"),
                ("consensusLock", "Consensus lock display"),
                ("deployForm", "Contract deployment form"),
                ("updateConsensusData", "Live update function")
            ]

            for search_str, description in checks:
                if search_str in html:
                    self.print_test(f"Dashboard: {description}", "PASS")
                else:
                    self.print_test(f"Dashboard: {description}", "WARN", "Not found")

        except Exception as e:
            self.print_test("Dashboard HTML", "FAIL", str(e)[:50])

    def test_documentation(self):
        """Test documentation completeness"""
        self.print_header("Testing Documentation")

        docs = [
            ("README.md", ["Quick Start", "API Reference", "Deployment"]),
            ("QUICK_START.md", ["Deploy Everything", "Endpoints", "Contract Commands"]),
            ("docs/MIGRATION_GUIDE.md", ["POA", "Analog", "Migration"])
        ]

        for doc_path, required_sections in docs:
            full_path = self.root / doc_path
            if not full_path.exists():
                self.print_test(f"Doc: {doc_path}", "FAIL", "Not found")
                continue

            try:
                with open(full_path, 'r', encoding='utf-8') as f:
                    content = f.read()

                self.print_test(f"Doc: {doc_path}", "PASS")

                # Check for required sections
                for section in required_sections:
                    if section in content:
                        self.print_test(f"  Section: {section}", "PASS")
                    else:
                        self.print_test(f"  Section: {section}", "WARN", "Not found")

            except Exception as e:
                self.print_test(f"Doc: {doc_path}", "FAIL", str(e)[:50])

    def test_web_api_server(self):
        """Test web API server structure"""
        self.print_header("Testing Web API Server")

        server_path = self.root / "web" / "analog_api_server.py"
        if not server_path.exists():
            self.print_test("Web API server", "FAIL", "Not found")
            return

        try:
            with open(server_path, 'r') as f:
                code = f.read()

            # Check for required routes
            routes = [
                ("@app.route('/')", "Root route"),
                ("/api/consensus/status", "Consensus status"),
                ("/api/contracts/deploy", "Contract deployment"),
                ("/api/network/peers", "Network peers"),
                ("/api/system/info", "System info"),
                ("/api/comparison/poa", "POA comparison")
            ]

            for route, description in routes:
                if route in code:
                    self.print_test(f"Route: {description}", "PASS")
                else:
                    self.print_test(f"Route: {description}", "FAIL", "Not found")

            # Check Flask imports
            if "from flask import Flask" in code:
                self.print_test("Flask import", "PASS")
            else:
                self.print_test("Flask import", "FAIL")

        except Exception as e:
            self.print_test("Web API server", "FAIL", str(e)[:50])

    def test_dependencies(self):
        """Test Python dependencies"""
        self.print_header("Testing Dependencies")

        dependencies = [
            ("flask", "Flask web framework"),
            ("requests", "HTTP requests library")
        ]

        for module, description in dependencies:
            try:
                __import__(module)
                self.print_test(f"{description} ({module})", "PASS")
            except ImportError:
                self.print_test(f"{description} ({module})", "WARN", "Not installed (pip install {module})")

    def print_summary(self):
        """Print test summary"""
        self.print_header("Test Summary")

        total = self.passed + self.failed + self.warnings

        print(f"\n✅ Passed:   {self.passed}/{total}")
        print(f"❌ Failed:   {self.failed}/{total}")
        print(f"⚠️  Warnings: {self.warnings}/{total}")

        if self.failed == 0:
            print("\n🎉 ALL CRITICAL TESTS PASSED!")
            if self.warnings > 0:
                print(f"⚠️  {self.warnings} warnings (non-critical)")
            return True
        else:
            print(f"\n❌ {self.failed} critical test(s) failed")
            return False

    def run_all_tests(self):
        """Run all tests"""
        print("\n" + "=" * 70)
        print("  ANALOG MAINNET VERIFICATION & TEST SUITE")
        print("  Testing all components without C compilation")
        print("=" * 70)

        self.test_directory_structure()
        self.test_core_files()
        self.test_python_modules()
        self.test_orchestration()
        self.test_contract_cli()
        self.test_example_contract()
        self.test_c_source_files()
        self.test_dashboard()
        self.test_web_api_server()
        self.test_documentation()
        self.test_dependencies()

        return self.print_summary()

def main():
    tester = AnalogMainnetTester()
    success = tester.run_all_tests()
    sys.exit(0 if success else 1)

if __name__ == '__main__':
    main()
