#!/usr/bin/env python3
"""
Analog API Server
Web services for Analog Mainnet consensus system
Replaces POA blockchain RPC with analog consensus API
"""

from flask import Flask, jsonify, request, send_from_directory
from flask_cors import CORS
import requests
import json
import time
from pathlib import Path

app = Flask(__name__)
CORS(app)

# Configuration
CONSENSUS_ENGINE_URL = 'http://localhost:9999'
WEB_ROOT = Path(__file__).parent
STATIC_DIR = WEB_ROOT / 'static'

# Cache for consensus data
consensus_cache = {
    'last_update': 0,
    'data': None
}

def get_consensus_status():
    """Get status from analog consensus engine"""
    try:
        # Check cache (update every 100ms)
        now = time.time()
        if consensus_cache['data'] and (now - consensus_cache['last_update']) < 0.1:
            return consensus_cache['data']

        # Fetch fresh data
        response = requests.get(f'{CONSENSUS_ENGINE_URL}/api/status', timeout=1)
        if response.status_code == 200:
            data = response.json()
            consensus_cache['data'] = data
            consensus_cache['last_update'] = now
            return data
    except Exception as e:
        print(f"Error fetching consensus status: {e}")

    return None

@app.route('/')
def index():
    """Serve main dashboard"""
    return send_from_directory(WEB_ROOT, 'dashboard.html')

@app.route('/api/consensus/status')
def consensus_status():
    """Get analog consensus status"""
    data = get_consensus_status()
    if data:
        return jsonify({
            'success': True,
            'consensus_type': 'analog_mathematical',
            'evolution_count': data.get('evolution_count', 0),
            'consensus_locked': data.get('locked', False),
            'phase_variance': data.get('phase_variance', 0),
            'consensus_steps': data.get('consensus_count', 0),
            'target_hz': data.get('target_hz', 32768),
            'uptime_seconds': data.get('uptime_seconds', 0),
            'dimensions': data.get('dimensions', 8),
            'timestamp': time.time()
        })
    else:
        return jsonify({
            'success': False,
            'error': 'Consensus engine not responding'
        }), 503

@app.route('/api/consensus/metrics')
def consensus_metrics():
    """Get Prometheus-style metrics"""
    try:
        response = requests.get(f'{CONSENSUS_ENGINE_URL}/metrics', timeout=1)
        if response.status_code == 200:
            return response.text, 200, {'Content-Type': 'text/plain'}
    except:
        pass
    return "# Error fetching metrics", 503

@app.route('/api/network/peers')
def network_peers():
    """Get analog network peers (via NetCat)"""
    try:
        response = requests.get(f'{CONSENSUS_ENGINE_URL}/api/netcat', timeout=1)
        if response.status_code == 200:
            data = response.json()
            return jsonify({
                'success': True,
                'peer_count': data.get('connected_peers', 0),
                'peers': data.get('peers', []),
                'phase_sync': data.get('phase_sync', 0),
                'consensus_reached': data.get('consensus_reached', False)
            })
    except:
        pass

    return jsonify({
        'success': False,
        'peer_count': 0,
        'peers': []
    })

@app.route('/api/contracts/deploy', methods=['POST'])
def deploy_contract():
    """Deploy analog contract"""
    try:
        data = request.get_json()
        contract_code = data.get('contract_code')
        contract_name = data.get('name', 'Unnamed Contract')

        if not contract_code:
            return jsonify({
                'success': False,
                'error': 'No contract code provided'
            }), 400

        # TODO: Implement actual contract deployment
        # For now, return success with mock contract ID
        contract_id = f"analog_{int(time.time() * 1000)}"

        return jsonify({
            'success': True,
            'contract_id': contract_id,
            'name': contract_name,
            'deployed_at': time.time(),
            'status': 'pending_consensus_lock'
        })

    except Exception as e:
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500

@app.route('/api/contracts/list')
def list_contracts():
    """List deployed contracts"""
    # TODO: Implement actual contract storage/retrieval
    return jsonify({
        'success': True,
        'contracts': []
    })

@app.route('/api/contracts/<contract_id>')
def get_contract(contract_id):
    """Get contract details"""
    # TODO: Implement actual contract retrieval
    return jsonify({
        'success': False,
        'error': 'Contract not found'
    }), 404

@app.route('/api/system/info')
def system_info():
    """Get system information"""
    data = get_consensus_status()

    return jsonify({
        'success': True,
        'system': 'Analog Mainnet',
        'version': '1.0.0',
        'consensus_type': 'Mathematical Convergence',
        'consensus_formula': 'Dₙ(r) = √(φ·Fₙ·2ⁿ·Pₙ·Ω)·r^k',
        'evolution_rate': '32,768 Hz',
        'consensus_threshold': 'CV < 0.1%',
        'validators_required': 0,
        'finality_type': 'Instant (when locked)',
        'network': 'analog_phase_coherent',
        'engine_status': 'running' if data else 'stopped'
    })

@app.route('/api/comparison/poa')
def poa_comparison():
    """Show comparison vs POA (for documentation)"""
    return jsonify({
        'success': True,
        'comparison': {
            'poa': {
                'consensus': 'Authority voting',
                'timing': '5-15 sec blocks',
                'validators': '2-3 authority nodes',
                'finality': 'After N confirmations',
                'attack_resistance': '51% authority',
                'energy': 'Mining overhead'
            },
            'analog': {
                'consensus': 'Mathematical convergence (CV < 0.1%)',
                'timing': '30.517 μs evolution period',
                'validators': '0 (deterministic proof)',
                'finality': 'Instant (when locked)',
                'attack_resistance': 'Reversing chaotic Lyapunov dynamics',
                'energy': 'Pure computation'
            },
            'improvement_factor': {
                'speed': '1,000,000x faster',
                'validators': 'None required',
                'finality': 'Instant vs. delayed',
                'security': 'Mathematical vs. trust-based'
            }
        }
    })

@app.route('/health')
def health():
    """Health check endpoint"""
    data = get_consensus_status()
    return jsonify({
        'status': 'healthy' if data else 'degraded',
        'consensus_engine': 'running' if data else 'stopped',
        'timestamp': time.time()
    })

if __name__ == '__main__':
    print("=" * 70)
    print("ANALOG API SERVER")
    print("Web Services for Post-Blockchain Consensus")
    print("=" * 70)
    print("\nStarting Flask server...")
    print("Endpoints:")
    print("  • Dashboard: http://localhost:8080/")
    print("  • Consensus Status: http://localhost:8080/api/consensus/status")
    print("  • Network Peers: http://localhost:8080/api/network/peers")
    print("  • Deploy Contract: POST http://localhost:8080/api/contracts/deploy")
    print("  • System Info: http://localhost:8080/api/system/info")
    print("  • POA Comparison: http://localhost:8080/api/comparison/poa")
    print("\n" + "=" * 70)

    app.run(host='0.0.0.0', port=8080, debug=True)
