#!/usr/bin/env python3
"""
Calculate required GAMMA for convergence based on measured forcing rates
"""
import math

# ============================================================================
# MEASURED DATA
# ============================================================================
gamma_actual = 0.00010386  # /sec - actual damping rate measured
time_to_steady_state = 213.62  # seconds - when variance stabilized
steady_state_variance = 7.27  # Mean variance at steady state

print("=" * 70)
print("STEADY STATE ANALYSIS")
print("=" * 70)
print(f"System reached steady state at t = {time_to_steady_state:.1f} sec")
print(f"Steady state variance: σ² = {steady_state_variance:.2f}")
print(f"Net damping rate: γ_net = {gamma_actual:.8f} /sec")

# ============================================================================
# FORCING RATE CALCULATION
# ============================================================================
# At steady state: forcing_rate = damping_rate
# The system oscillates because:
#   dσ²/dt = forcing_terms - GAMMA * σ² = 0
#
# Current equation: dx/dt = phase*exp(iφ) + coupling - GAMMA*x
#
# GAMMA = 0.02
# But effective damping is only 0.00010386 /sec
# This means forcing terms dominate!

GAMMA_current = 0.02
sample_rate = 32768  # Hz

# The -GAMMA*x term damps at rate GAMMA per RK4 step
# Over 1 second (32768 steps), net damping = (1 - (1-GAMMA*dt)^steps)
dt = 1.0 / sample_rate
steps_per_second = sample_rate

# Effective damping per second from -GAMMA*x term
gamma_from_damping_term = -math.log((1 - GAMMA_current * dt) ** steps_per_second)
print(f"\nCurrent GAMMA = {GAMMA_current}")
print(f"Damping from -GAMMA*x term: {gamma_from_damping_term:.6f} /sec")

# Forcing rate = damping_rate - net_observed_damping
forcing_rate = gamma_from_damping_term - gamma_actual
print(f"Forcing rate (coupling + phase): {forcing_rate:.6f} /sec")

# ============================================================================
# REQUIRED GAMMA FOR CONVERGENCE
# ============================================================================
# For 1-minute convergence (99% reduction):
target_convergence_time = 60  # seconds
target_net_damping = -math.log(0.01) / target_convergence_time  # γ for 99% reduction
print(f"\n{'='*70}")
print(f"TARGET: Converge in {target_convergence_time} seconds")
print(f"{'='*70}")
print(f"Required net damping: γ_net = {target_net_damping:.6f} /sec")

# Required GAMMA: must overcome forcing + provide net damping
# Net damping = (GAMMA * sample_rate) - forcing_rate
# So: GAMMA * sample_rate = forcing_rate + target_net_damping
# Therefore: GAMMA = (forcing_rate + target_net_damping) / sample_rate
required_gamma_total = forcing_rate + target_net_damping
required_GAMMA = required_gamma_total / sample_rate
print(f"Required total damping: {required_gamma_total:.6f} /sec")
print(f"  = forcing ({forcing_rate:.6f}) + net damping ({target_net_damping:.6f})")

print(f"\n🎯 REQUIRED GAMMA = {required_GAMMA:.6f}")
print(f"   Current: {GAMMA_current}")
print(f"   Increase by factor: {required_GAMMA / GAMMA_current:.2f}×")

# Sanity check: GAMMA must be > forcing_rate / sample_rate to overcome divergence
min_GAMMA_to_stabilize = forcing_rate / sample_rate
print(f"\n   Minimum GAMMA to stabilize: {min_GAMMA_to_stabilize:.8f}")
print(f"   Required GAMMA for 60s convergence: {required_GAMMA:.8f}")

# ============================================================================
# VERIFY WITH EXPONENTIAL DECAY
# ============================================================================
print(f"\n{'='*70}")
print(f"VERIFICATION")
print(f"{'='*70}")

# With new GAMMA, effective damping per second:
new_gamma_damping = -math.log((1 - required_GAMMA * dt) ** steps_per_second)
new_net_damping = new_gamma_damping - forcing_rate
print(f"With GAMMA = {required_GAMMA:.6f}:")
print(f"  Damping from -GAMMA*x: {new_gamma_damping:.6f} /sec")
print(f"  Forcing rate:          {forcing_rate:.6f} /sec")
print(f"  Net damping:           {new_net_damping:.6f} /sec")

# Time to converge
actual_convergence_time = -math.log(0.01) / new_net_damping
print(f"  Convergence time:      {actual_convergence_time:.1f} seconds")

# ============================================================================
# ALTERNATIVE: MODIFY FORCING TERMS
# ============================================================================
print(f"\n{'='*70}")
print(f"ALTERNATIVE APPROACHES")
print(f"{'='*70}")
print(f"Instead of increasing GAMMA, we could:")
print(f"1. Reduce K_COUPLING (currently creates +divergence)")
print(f"2. Reduce phase_term magnitude (currently creates chaos)")
print(f"3. Add adaptive damping to k1,k2,k3,k4 calculations")
print(f"4. Apply feedback to derivatives, not just position")

# Check current coupling constant
print(f"\nCurrent system parameters:")
print(f"  K_COUPLING = 0.1 (creates +{forcing_rate/2:.6f} /sec divergence)")
print(f"  phase_term = freq + 0.1*Dn_mod (creates +{forcing_rate/2:.6f} /sec chaos)")
print(f"  GAMMA = {GAMMA_current} (provides {gamma_from_damping_term:.6f} /sec damping)")
print(f"\nNet effect: Forcing ({forcing_rate:.6f}) >> Damping ({gamma_actual:.6f})")
print(f"Result: System oscillates at steady state σ² ≈ {steady_state_variance:.1f}")
