#!/usr/bin/env python3
"""
Correct analysis: The -GAMMA*x term operates at each RK4 timestep
"""
import math

# System parameters
GAMMA_current = 0.02
dt = 1.0 / 32768  # timestep
sample_rate = 32768

# Measured data
gamma_net_observed = 0.00010386  # /sec (nearly zero - steady state)
steady_state_variance = 7.27

print("=" * 70)
print("CORRECTED ANALYSIS: Per-timestep vs Per-second")
print("=" * 70)

# ============================================================================
# The equation is: dx/dt = phase_term*exp(iφ) + coupling - GAMMA*x
#
# In RK4 discrete form:
#   k = f(x) = phase_term*exp(iφ) + coupling - GAMMA*x
#   x_new = x + (dt/6)*(k1 + 2*k2 + 2*k3 + k4)
#
# The -GAMMA*x term contributes to k, which is then multiplied by dt
# So the effective damping per step is approximately: GAMMA * dt
# ============================================================================

print("\nPer-step damping:")
damping_per_step = GAMMA_current * dt
print(f"  -GAMMA*x term: {GAMMA_current} (constant)")
print(f"  Timestep: dt = {dt:.10f} sec")
print(f"  Damping per step: GAMMA*dt = {damping_per_step:.10f}")

# After N steps, with damping d per step:
# x(N) = x(0) * (1 - d)^N ≈ x(0) * exp(-d*N)  (for small d)
# So: γ_effective = d * N / T = d * N / (N*dt) = d / dt = GAMMA

print(f"\nOver 1 second ({sample_rate} steps):")
print(f"  Theoretical damping from -GAMMA*x: γ = GAMMA = {GAMMA_current:.6f} /sec")

# But we observed only 0.000104 /sec net damping!
# This means the forcing terms are adding +0.02 - 0.0001 ≈ +0.01990 /sec

forcing_rate_per_sec = GAMMA_current - gamma_net_observed
print(f"  Observed net damping: {gamma_net_observed:.6f} /sec")
print(f"  Therefore forcing rate: {forcing_rate_per_sec:.6f} /sec")

# ============================================================================
# REQUIRED GAMMA FOR CONVERGENCE
# ============================================================================
print(f"\n{'='*70}")
print("REQUIRED GAMMA FOR 60-SECOND CONVERGENCE")
print("=" * 70)

target_time = 60
target_net_damping = -math.log(0.01) / target_time  # 99% reduction
print(f"Target net damping: {target_net_damping:.6f} /sec")

# Net damping = GAMMA - forcing_rate
# So: GAMMA = forcing_rate + target_net_damping
required_GAMMA = forcing_rate_per_sec + target_net_damping
print(f"Required GAMMA: {forcing_rate_per_sec:.6f} + {target_net_damping:.6f}")
print(f"              = {required_GAMMA:.6f}")

print(f"\n🎯 CHANGE GAMMA FROM {GAMMA_current} TO {required_GAMMA:.6f}")
print(f"   Increase by factor: {required_GAMMA / GAMMA_current:.2f}×")

# ============================================================================
# VERIFICATION
# ============================================================================
print(f"\n{'='*70}")
print("VERIFICATION")
print("=" * 70)

new_damping = required_GAMMA
new_net = new_damping - forcing_rate_per_sec
convergence_time = -math.log(0.01) / new_net

print(f"With GAMMA = {required_GAMMA:.6f}:")
print(f"  Damping from -GAMMA*x: {new_damping:.6f} /sec")
print(f"  Forcing from phase+coupling: {forcing_rate_per_sec:.6f} /sec")
print(f"  Net damping: {new_net:.6f} /sec")
print(f"  Convergence time: {convergence_time:.1f} seconds ✓")

# ============================================================================
# IMPLEMENTATION
# ============================================================================
print(f"\n{'='*70}")
print("IMPLEMENTATION")
print("=" * 70)
print(f"Edit: analog-mainnet/src/hdgl_bridge_v40.c")
print(f"Line 55: #define GAMMA {required_GAMMA:.6f}L")
print(f"         (was: #define GAMMA {GAMMA_current}L)")
