#!/usr/bin/env python3
"""
Analog Mainnet Deployment Script
Fully replaces POA with mathematical consensus
"""

import os
import sys
import subprocess
import argparse
import json
import time
from pathlib import Path

class AnalogMainnetDeployer:
    def __init__(self, mode='local', headless=False, dashboard=True, web_ui=True):
        self.mode = mode
        self.headless = headless
        self.dashboard = dashboard
        self.web_ui = web_ui
        self.root = Path(__file__).parent
        self.processes = []

    def deploy_full_stack(self):
        """Deploy complete analog mainnet"""
        print("=" * 70)
        print("ANALOG MAINNET DEPLOYMENT")
        print("Post-Blockchain Consensus System")
        print("=" * 70)
        print(f"\nMode: {self.mode}")
        print(f"Headless: {self.headless}")
        print(f"Dashboard: {self.dashboard}")
        print(f"Web UI: {self.web_ui}")
        print()

        # Step 1: Compile C engine
        print("[1/5] Compiling analog consensus engine...")
        if not self.compile_engine():
            print("❌ Compilation failed")
            return False
        print("✅ Engine compiled\n")

        # Step 2: Generate runtime configuration
        print("[2/5] Generating runtime configuration...")
        if not self.generate_config():
            print("❌ Config generation failed")
            return False
        print("✅ Configuration ready\n")

        # Step 3: Start consensus engine
        print("[3/5] Starting analog consensus engine (32,768 Hz)...")
        if not self.start_consensus_engine():
            print("❌ Engine start failed")
            return False
        print("✅ Consensus engine running\n")

        # Step 4: Start web services
        if self.web_ui:
            print("[4/5] Starting web services...")
            if not self.start_web_services():
                print("❌ Web services failed")
                return False
            print("✅ Web services running\n")
        else:
            print("[4/5] Web services disabled (skipped)\n")

        # Step 5: Deploy dashboard
        if self.dashboard:
            print("[5/5] Deploying dashboard...")
            if not self.deploy_dashboard():
                print("❌ Dashboard deployment failed")
                return False
            print("✅ Dashboard deployed\n")
        else:
            print("[5/5] Dashboard disabled (skipped)\n")

        self.print_status()
        return True

    def compile_engine(self):
        """Compile C analog engine"""
        src_dir = self.root / 'src'
        if not src_dir.exists():
            print(f"❌ Source directory not found: {src_dir}")
            return False

        makefile = src_dir / 'Makefile'
        if makefile.exists():
            # Use Makefile
            result = subprocess.run(
                ['make', '-C', str(src_dir)],
                capture_output=True,
                text=True
            )
        else:
            # Direct GCC compilation
            result = subprocess.run([
                'gcc',
                '-o', str(src_dir / 'hdgl_bridge'),
                str(src_dir / 'hdgl_bridge_v40.c'),
                str(src_dir / 'hdgl_http_api.c'),
                str(src_dir / 'hdgl_netcat.c'),
                '-lm', '-lpthread', '-lwinmm', '-lws2_32'
            ], capture_output=True, text=True)

        if result.returncode != 0:
            print(f"Compilation error:\n{result.stderr}")
            return False
        return True

    def generate_config(self):
        """Generate runtime configuration from Base4096 stream"""
        # Check for orchestration tools
        orchestrate = self.root / 'orchestration' / 'orchestrate.py'
        if not orchestrate.exists():
            print(f"❌ Orchestrator not found: {orchestrate}")
            return False

        # Generate config
        result = subprocess.run([
            sys.executable,
            str(orchestrate),
            '--config-only',
            '--output', str(self.root / 'config' / 'runtime_config.json')
        ], capture_output=True, text=True)

        if result.returncode != 0:
            print(f"Config generation error:\n{result.stderr}")
            return False
        return True

    def start_consensus_engine(self):
        """Start the analog consensus engine"""
        engine = self.root / 'src' / 'hdgl_bridge.exe'
        if not engine.exists():
            engine = self.root / 'src' / 'hdgl_bridge'

        if not engine.exists():
            print(f"❌ Engine binary not found: {engine}")
            return False

        # Set environment
        env = os.environ.copy()
        env['HDGL_CONFIG'] = str(self.root / 'config' / 'runtime_config.json')

        # Build command
        cmd = [str(engine)]
        if self.headless:
            cmd.append('--headless')

        # Start engine
        log_file = self.root / 'logs' / 'consensus_engine.log'
        log_file.parent.mkdir(exist_ok=True)

        with open(log_file, 'w') as f:
            proc = subprocess.Popen(
                cmd,
                env=env,
                stdout=f,
                stderr=subprocess.STDOUT,
                cwd=str(self.root / 'src')
            )

        self.processes.append(('Consensus Engine', proc))

        # Wait for engine to start
        time.sleep(2)
        if proc.poll() is not None:
            print("❌ Engine process died")
            return False

        # Verify API is responding
        import urllib.request
        try:
            urllib.request.urlopen('http://localhost:9999/api/status', timeout=5)
            return True
        except:
            print("⚠️  Engine started but API not responding yet")
            return True  # Still consider success

    def start_web_services(self):
        """Start web services API server"""
        web_server = self.root / 'web' / 'analog_api_server.py'
        if not web_server.exists():
            print(f"❌ Web server not found: {web_server}")
            return False

        log_file = self.root / 'logs' / 'web_services.log'
        with open(log_file, 'w') as f:
            proc = subprocess.Popen(
                [sys.executable, str(web_server)],
                stdout=f,
                stderr=subprocess.STDOUT
            )

        self.processes.append(('Web Services', proc))
        time.sleep(1)

        return proc.poll() is None

    def deploy_dashboard(self):
        """Deploy web dashboard"""
        dashboard = self.root / 'web' / 'dashboard.html'
        if not dashboard.exists():
            print(f"⚠️  Dashboard not found: {dashboard}")
            return True  # Non-critical

        print(f"   Dashboard available at: http://localhost:8080")
        return True

    def print_status(self):
        """Print deployment status"""
        print("=" * 70)
        print("✅ ANALOG MAINNET DEPLOYED SUCCESSFULLY")
        print("=" * 70)
        print("\n📊 Services Running:")
        for name, proc in self.processes:
            status = "🟢 Running" if proc.poll() is None else "🔴 Stopped"
            print(f"   {name}: {status}")

        print("\n🌐 Endpoints:")
        print(f"   • Consensus API: http://localhost:9999/api/status")
        print(f"   • Metrics: http://localhost:9999/metrics")
        print(f"   • NetCat Status: http://localhost:9999/api/netcat")
        if self.web_ui:
            print(f"   • Web UI: http://localhost:8080")
        if self.dashboard:
            print(f"   • Dashboard: http://localhost:8080/dashboard.html")

        print("\n📈 Consensus Info:")
        print(f"   • Evolution Rate: 32,768 Hz (exact)")
        print(f"   • Consensus Model: Mathematical convergence (CV < 0.1%)")
        print(f"   • Validators: 0 (deterministic proof)")
        print(f"   • Finality: Instant (when locked)")

        print("\n🔧 Management:")
        print(f"   • Check status: curl http://localhost:9999/api/status")
        print(f"   • View logs: type logs\\consensus_engine.log")
        print(f"   • Deploy contract: python contracts\\deploy_analog_contract.py <file>")
        print(f"   • Stop all: Press Ctrl+C")

        print("\n" + "=" * 70)

    def shutdown(self):
        """Gracefully shutdown all processes"""
        print("\n🛑 Shutting down Analog Mainnet...")
        for name, proc in self.processes:
            if proc.poll() is None:
                print(f"   Stopping {name}...")
                proc.terminate()
                try:
                    proc.wait(timeout=5)
                except subprocess.TimeoutExpired:
                    proc.kill()
        print("✅ Shutdown complete")

def main():
    parser = argparse.ArgumentParser(description='Deploy Analog Mainnet')
    parser.add_argument('--mode', choices=['local', 'testnet', 'mainnet'],
                       default='local', help='Deployment mode')
    parser.add_argument('--full', action='store_true',
                       help='Deploy full stack (default)')
    parser.add_argument('--headless', action='store_true',
                       help='Run consensus engine in headless mode')
    parser.add_argument('--no-dashboard', action='store_true',
                       help='Disable dashboard')
    parser.add_argument('--no-web-ui', action='store_true',
                       help='Disable web UI')
    parser.add_argument('--debug', action='store_true',
                       help='Enable debug mode')

    args = parser.parse_args()

    deployer = AnalogMainnetDeployer(
        mode=args.mode,
        headless=args.headless,
        dashboard=not args.no_dashboard,
        web_ui=not args.no_web_ui
    )

    try:
        if deployer.deploy_full_stack():
            # Keep running
            print("\n⏳ Mainnet running. Press Ctrl+C to stop...")
            while True:
                time.sleep(1)
        else:
            print("\n❌ Deployment failed")
            sys.exit(1)
    except KeyboardInterrupt:
        deployer.shutdown()
    except Exception as e:
        print(f"\n❌ Error: {e}")
        deployer.shutdown()
        sys.exit(1)

if __name__ == '__main__':
    main()
