#!/bin/bash
# LXC Container Provisioning Script for Analog Mainnet
# Creates LXC containers for distributed consensus testing

set -e

echo "========================================================================"
echo "  ANALOG MAINNET LXC PROVISIONING"
echo "  Creating Production Consensus Containers"
echo "========================================================================"

# Configuration
BRIDGE_NAME="analog-br0"
SUBNET="10.100.0.0/24"
GATEWAY="10.100.0.1"

# Container definitions
declare -A CONTAINERS=(
    ["analog-primary"]="10.100.0.10"
    ["analog-peer1"]="10.100.0.11"
    ["analog-peer2"]="10.100.0.12"
    ["analog-peer3"]="10.100.0.13"
)

# Create network bridge if it doesn't exist
echo "[1/6] Setting up network bridge..."
if ! ip link show $BRIDGE_NAME > /dev/null 2>&1; then
    sudo ip link add name $BRIDGE_NAME type bridge
    sudo ip addr add $GATEWAY/24 dev $BRIDGE_NAME
    sudo ip link set $BRIDGE_NAME up
    echo "✅ Bridge $BRIDGE_NAME created"
else
    echo "✅ Bridge $BRIDGE_NAME already exists"
fi

# Enable IP forwarding
echo "[2/6] Enabling IP forwarding..."
sudo sysctl -w net.ipv4.ip_forward=1
echo "✅ IP forwarding enabled"

# Create base container
echo "[3/6] Creating base container template..."
if ! lxc-ls | grep -q "analog-base"; then
    sudo lxc-create -n analog-base -t alpine -- -r 3.18

    # Configure base container
    sudo bash -c "cat >> /var/lib/lxc/analog-base/config << EOF

# Network configuration
lxc.net.0.type = veth
lxc.net.0.flags = up
lxc.net.0.link = $BRIDGE_NAME
lxc.net.0.name = eth0

# Resource limits
lxc.cgroup2.memory.max = 1G
lxc.cgroup2.cpu.max = 200000 100000

# Autostart
lxc.start.auto = 0
EOF"

    echo "✅ Base container created"
else
    echo "✅ Base container already exists"
fi

# Install dependencies in base container
echo "[4/6] Installing dependencies in base container..."
sudo lxc-start -n analog-base -d
sleep 5

sudo lxc-attach -n analog-base -- apk add --no-cache \
    gcc g++ make musl-dev linux-headers curl ca-certificates

sudo lxc-stop -n analog-base
echo "✅ Dependencies installed"

# Clone containers for each node
echo "[5/6] Creating node containers..."
for container in "${!CONTAINERS[@]}"; do
    ip_addr="${CONTAINERS[$container]}"

    if lxc-ls | grep -q "^${container}$"; then
        echo "⚠️  Container $container already exists, skipping..."
        continue
    fi

    echo "  Creating $container at $ip_addr..."
    sudo lxc-copy -n analog-base -N $container

    # Configure network
    sudo bash -c "cat >> /var/lib/lxc/$container/config << EOF

# Static IP configuration
lxc.net.0.ipv4.address = $ip_addr/24
lxc.net.0.ipv4.gateway = $GATEWAY

# Hostname
lxc.uts.name = $container

# Autostart
lxc.start.auto = 1
lxc.start.delay = 2
EOF"

    echo "  ✅ $container created"
done

echo "✅ All containers created"

# Deploy analog consensus engine to containers
echo "[6/6] Deploying consensus engine to containers..."
for container in "${!CONTAINERS[@]}"; do
    echo "  Deploying to $container..."

    # Copy source files
    sudo mkdir -p /var/lib/lxc/$container/rootfs/app
    sudo cp src/hdgl_bridge_v40.c /var/lib/lxc/$container/rootfs/app/
    sudo cp src/hdgl_http_api.c /var/lib/lxc/$container/rootfs/app/
    sudo cp src/hdgl_netcat.c /var/lib/lxc/$container/rootfs/app/
    sudo cp -r config /var/lib/lxc/$container/rootfs/app/

    # Start container
    sudo lxc-start -n $container -d
    sleep 3

    # Compile inside container
    sudo lxc-attach -n $container -- bash -c "
        cd /app
        gcc -o hdgl_bridge hdgl_bridge_v40.c hdgl_http_api.c hdgl_netcat.c \
            -lm -lpthread -O3 -DPRODUCTION=1
        chmod +x hdgl_bridge
    "

    # Set node-specific configuration
    local role="peer"
    if [ "$container" = "analog-primary" ]; then
        role="primary"
    fi

    sudo lxc-attach -n $container -- bash -c "
        export HDGL_NODE_ID=$container
        export HDGL_ROLE=$role
        export HDGL_CONFIG=/app/config/runtime_config.json
    "

    echo "  ✅ $container deployed"
done

echo "✅ Deployment complete"

# Start all containers
echo ""
echo "Starting all containers..."
for container in "${!CONTAINERS[@]}"; do
    if ! lxc-info -n $container | grep -q "RUNNING"; then
        sudo lxc-start -n $container -d
        echo "  ✅ Started $container"
    fi
done

# Display container status
echo ""
echo "========================================================================"
echo "  CONTAINER STATUS"
echo "========================================================================"
for container in "${!CONTAINERS[@]}"; do
    ip_addr="${CONTAINERS[$container]}"
    status=$(lxc-info -n $container | grep State | awk '{print $2}')
    echo "  $container: $status @ $ip_addr"
done

echo ""
echo "========================================================================"
echo "  ENDPOINTS"
echo "========================================================================"
echo "  Primary Node:  http://10.100.0.10:9999/api/status"
echo "  Peer 1:        http://10.100.0.11:9999/api/status"
echo "  Peer 2:        http://10.100.0.12:9999/api/status"
echo "  Peer 3:        http://10.100.0.13:9999/api/status"
echo ""
echo "  Metrics:       http://10.100.0.10:9999/metrics"
echo ""
echo "========================================================================"
echo "✅ LXC PROVISIONING COMPLETE"
echo "========================================================================"
