/*
 * OpenCL GPU Kernel for Analog Codec V4.3
 * Runs 8-dimensional evolution on AMD RX 480
 * Massively parallel: Each work item evolves independently
 */

// State per dimension
typedef struct {
    float amplitude;      // Current amplitude
    float phase;          // Phase angle
    float frequency;      // Base frequency
    float dn;             // Dimensional separation
    float res_weight;     // Resonance weight
} DimState;

// Global state
typedef struct {
    DimState dims[8];     // 8 dimensions
    float omega;          // Ω coupling strength
    float k_coupling;     // Coupling coefficient
    float gamma_damping;  // Damping coefficient
    uint evolution;       // Evolution counter
} CodecState;

// Constants
#define PHI 1.618033988749895f
#define K_COUPLING 0.15f
#define GAMMA_DAMPING 0.01f
#define OMEGA_CAP 1000.0f
#define DT 0.001f  // Time step

// Fibonacci numbers for Dn calculation
__constant uint FIB[13] = {1, 1, 2, 3, 5, 8, 13, 21, 34, 55, 89, 144, 233};

// Prime numbers for Dn calculation
__constant uint PRIMES[8] = {2, 3, 5, 7, 11, 13, 17, 19};

// Calculate Dn for a dimension
float calculate_dn(float omega, int dim_index, float r) {
    float base = sqrt(PHI * FIB[dim_index + 2] * (1 << (dim_index + 1)) * PRIMES[dim_index] * omega);
    float k = 0.5f + 0.1f * dim_index;
    return base * pow(r, k);
}

// Coupling kernel
float coupling_kernel(float dn_i, float dn_j, float res_w_j) {
    float dn_correlation = fabs(dn_j - dn_i);
    float dn_coupling = exp(-dn_correlation / 50.0f);
    return K_COUPLING * res_w_j * dn_coupling;
}

// Evolve one dimension
float evolve_dimension(
    __global CodecState* state,
    int i,
    float dt
) {
    float amp_i = state->dims[i].amplitude;
    float phase_i = state->dims[i].phase;
    float freq_i = state->dims[i].frequency;
    float dn_i = state->dims[i].dn;

    // Self oscillation
    float self_term = -state->gamma_damping * amp_i + amp_i * cos(phase_i);

    // Coupling from all other dimensions
    float coupling_sum = 0.0f;
    for (int j = 0; j < 8; j++) {
        if (i != j) {
            float amp_j = state->dims[j].amplitude;
            float phase_j = state->dims[j].phase;
            float dn_j = state->dims[j].dn;
            float res_w_j = state->dims[j].res_weight;

            float K_eff = coupling_kernel(dn_i, dn_j, res_w_j);
            coupling_sum += K_eff * amp_j * sin(phase_j - phase_i);
        }
    }

    // Combined dynamics
    float dA_dt = self_term + coupling_sum;
    float new_amp = amp_i + dA_dt * dt;

    // Phase advance
    float new_phase = phase_i + 2.0f * M_PI_F * freq_i * dt;
    if (new_phase > 2.0f * M_PI_F) new_phase -= 2.0f * M_PI_F;

    return new_amp;
}

// Main evolution kernel - each work item does one evolution step
__kernel void evolve_codec(
    __global CodecState* states,
    uint num_evolutions,
    uint stride
) {
    int gid = get_global_id(0);
    __global CodecState* state = &states[gid];

    // Run multiple evolution steps per work item
    for (uint e = 0; e < num_evolutions; e++) {
        // Update Ω (with saturation cap)
        state->omega += 0.0115f;
        if (state->omega > OMEGA_CAP) {
            state->omega = OMEGA_CAP;
        }

        // Update Dn values for all dimensions
        for (int i = 0; i < 8; i++) {
            float r = 0.125f * i;
            state->dims[i].dn = calculate_dn(state->omega, i, r);
        }

        // Evolve all dimensions (store in temp to avoid race conditions)
        float new_amps[8];
        for (int i = 0; i < 8; i++) {
            new_amps[i] = evolve_dimension(state, i, DT);
        }

        // Update amplitudes
        for (int i = 0; i < 8; i++) {
            state->dims[i].amplitude = new_amps[i];
        }

        // Update phases
        for (int i = 0; i < 8; i++) {
            state->dims[i].phase += 2.0f * M_PI_F * state->dims[i].frequency * DT;
            if (state->dims[i].phase > 2.0f * M_PI_F) {
                state->dims[i].phase -= 2.0f * M_PI_F;
            }
        }

        state->evolution++;
    }
}

// Snapshot kernel - extract dimensional data for visualization
__kernel void snapshot_state(
    __global CodecState* states,
    __global float* amplitudes,   // Output: 8 amplitudes per state
    __global float* dn_values,    // Output: 8 Dn values per state
    __global float* omegas        // Output: 1 omega per state
) {
    int gid = get_global_id(0);
    __global CodecState* state = &states[gid];

    // Copy dimensional data
    for (int i = 0; i < 8; i++) {
        amplitudes[gid * 8 + i] = state->dims[i].amplitude;
        dn_values[gid * 8 + i] = state->dims[i].dn;
    }
    omegas[gid] = state->omega;
}
