#!/bin/bash
# HDGL Weekly Snapshot Consolidation Script
# Combines daily snapshots into weekly archives and enforces 1GB total cap
# BULLETPROOF MODE: Never fails, always recovers

set +e

PERMANENT_SNAPSHOT_DIR=${1:-/app/data/snapshots/permanent}
WEEKLY_ARCHIVE_DIR="${PERMANENT_SNAPSHOT_DIR}/weekly"
MAX_TOTAL_MB=1000  # 1GB cap for ALL permanent snapshots

# BULLETPROOF: Log to file with fallback
LOG_FILE="/var/log/hdgl/weekly-consolidation.log"
mkdir -p "$(dirname "$LOG_FILE")" 2>/dev/null || LOG_FILE="/tmp/weekly-consolidation.log"
exec 1> >(tee -a "$LOG_FILE" 2>/dev/null || cat)
exec 2>&1

echo "=== HDGL Weekly Snapshot Consolidation ==="
echo "Time: $(date)"
echo "Permanent Snapshot Dir: $PERMANENT_SNAPSHOT_DIR"
echo "Weekly Archive Dir: $WEEKLY_ARCHIVE_DIR"
echo "Max total: ${MAX_TOTAL_MB}MB"
echo ""

# BULLETPROOF: Create directories with fallback
mkdir -p "$WEEKLY_ARCHIVE_DIR" 2>/dev/null || {
    echo "⚠️  Failed to create weekly dir, using temp"
    WEEKLY_ARCHIVE_DIR="/tmp/weekly-snapshots"
    mkdir -p "$WEEKLY_ARCHIVE_DIR"
}

# Function to get directory size in MB
get_size_mb() {
    local size=$(du -sm "$1" 2>/dev/null | cut -f1)
    if [ -z "$size" ]; then
        echo "0"
    else
        echo "$size"
    fi
}

# Function to get file size in MB
get_file_size_mb() {
    if [ ! -f "$1" ]; then
        echo "0"
        return
    fi
    local size_bytes=$(stat -f%z "$1" 2>/dev/null || stat -c%s "$1" 2>/dev/null || echo 0)
    echo $((size_bytes / 1024 / 1024))
}

# Get current week identifier (ISO week: YYYY-WW)
CURRENT_WEEK=$(date +%Y-W%V)
WEEKLY_ARCHIVE="$WEEKLY_ARCHIVE_DIR/weekly_snapshot_${CURRENT_WEEK}.tar.zst"

echo "📦 Consolidating daily snapshots into weekly archive..."
echo "Current week: $CURRENT_WEEK"
echo ""

# Find daily snapshots from the past 7 days (exclude weekly archives)
DAILY_SNAPSHOTS=$(find "$PERMANENT_SNAPSHOT_DIR" -maxdepth 1 -name "permanent_snapshot_*.tar*" -type f -mtime -7 2>/dev/null | sort)

if [ -z "$DAILY_SNAPSHOTS" ]; then
    echo "ℹ️  No daily snapshots found to consolidate"
else
    SNAPSHOT_COUNT=$(echo "$DAILY_SNAPSHOTS" | wc -l | tr -d ' ')
    echo "Found $SNAPSHOT_COUNT daily snapshot(s) to consolidate"

    # Only create weekly archive if we have at least 3 daily snapshots
    if [ "$SNAPSHOT_COUNT" -ge 3 ] && [ ! -f "$WEEKLY_ARCHIVE" ]; then
        echo "Creating weekly archive: $(basename "$WEEKLY_ARCHIVE")"

        ARCHIVE_CREATED=0

        # BULLETPROOF: Try multiple compression strategies
        # Strategy 1: Ultra compression (best space savings)
        if echo "$DAILY_SNAPSHOTS" | tar -cf - -T - 2>/dev/null | zstd -19 -T0 --ultra -o "$WEEKLY_ARCHIVE" 2>/dev/null; then
            ARCHIVE_CREATED=1
            echo "  ✅ Compressed with zstd ultra"
        # Strategy 2: Normal compression (fallback)
        elif echo "$DAILY_SNAPSHOTS" | tar -cf - -T - 2>/dev/null | zstd -9 -o "$WEEKLY_ARCHIVE" 2>/dev/null; then
            ARCHIVE_CREATED=1
            echo "  ✅ Compressed with zstd normal"
        # Strategy 3: Just tar (last resort)
        elif echo "$DAILY_SNAPSHOTS" | tar -cf "${WEEKLY_ARCHIVE%.zst}.tar" -T - 2>/dev/null; then
            WEEKLY_ARCHIVE="${WEEKLY_ARCHIVE%.zst}.tar"
            ARCHIVE_CREATED=1
            echo "  ✅ Created uncompressed archive"
        else
            echo "  ⚠️  All compression methods failed"
        fi

        if [ "$ARCHIVE_CREATED" -eq 1 ] && [ -f "$WEEKLY_ARCHIVE" ]; then
            ARCHIVE_SIZE_MB=$(get_file_size_mb "$WEEKLY_ARCHIVE")
            echo "  Created: $(basename "$WEEKLY_ARCHIVE") (${ARCHIVE_SIZE_MB}MB)"

            # Make it immutable
            chattr +i "$WEEKLY_ARCHIVE" 2>/dev/null || chmod 444 "$WEEKLY_ARCHIVE" 2>/dev/null || true
            echo "  🔒 Protected: immutable"

            # Now delete the daily snapshots that were consolidated
            echo "  🗑️  Removing consolidated daily snapshots..."
            echo "$DAILY_SNAPSHOTS" | while read -r snapshot; do
                # Remove immutable flag first
                chattr -i "$snapshot" 2>/dev/null || true
                if rm -f "$snapshot" 2>/dev/null; then
                    echo "    Removed: $(basename "$snapshot")"
                else
                    echo "    ⚠️  Failed to remove: $(basename "$snapshot")"
                fi
            done

            echo "  ✅ Weekly consolidation complete"
        fi
    elif [ -f "$WEEKLY_ARCHIVE" ]; then
        echo "ℹ️  Weekly archive for $CURRENT_WEEK already exists"
    else
        echo "ℹ️  Not enough daily snapshots yet ($SNAPSHOT_COUNT < 3)"
    fi
fi

echo ""
echo "📊 Enforcing 1GB total cap on permanent snapshots..."

# Get current total size of ALL permanent snapshots (daily + weekly)
TOTAL_SIZE=$(get_size_mb "$PERMANENT_SNAPSHOT_DIR")
echo "Current total size: ${TOTAL_SIZE}MB / ${MAX_TOTAL_MB}MB"

# If over cap, start removing oldest content
if [ "$TOTAL_SIZE" -gt "$MAX_TOTAL_MB" ]; then
    echo "⚠️  Over cap! Starting cleanup..."

    CLEANUP_ATTEMPTS=0
    MAX_CLEANUP_ATTEMPTS=50

    while [ "$TOTAL_SIZE" -gt "$MAX_TOTAL_MB" ] && [ "$CLEANUP_ATTEMPTS" -lt "$MAX_CLEANUP_ATTEMPTS" ]; do
        # Priority 1: Remove oldest daily snapshots first (keep weekly archives)
        OLDEST_DAILY=$(find "$PERMANENT_SNAPSHOT_DIR" -maxdepth 1 -name "permanent_snapshot_*.tar*" -type f 2>/dev/null | sort | head -n1)

        if [ -n "$OLDEST_DAILY" ]; then
            echo "  Removing oldest daily: $(basename "$OLDEST_DAILY")"
            chattr -i "$OLDEST_DAILY" 2>/dev/null || true
            rm -f "$OLDEST_DAILY" 2>/dev/null || echo "    ⚠️  Failed to remove"
        else
            # Priority 2: If no daily snapshots left, remove oldest weekly archive
            OLDEST_WEEKLY=$(find "$WEEKLY_ARCHIVE_DIR" -name "weekly_snapshot_*.tar*" -type f 2>/dev/null | sort | head -n1)

            if [ -n "$OLDEST_WEEKLY" ]; then
                echo "  Removing oldest weekly: $(basename "$OLDEST_WEEKLY")"
                chattr -i "$OLDEST_WEEKLY" 2>/dev/null || true
                rm -f "$OLDEST_WEEKLY" 2>/dev/null || echo "    ⚠️  Failed to remove"
            else
                echo "  ⚠️  No more files to remove"
                break
            fi
        fi

        # Recalculate size
        TOTAL_SIZE=$(get_size_mb "$PERMANENT_SNAPSHOT_DIR")
        CLEANUP_ATTEMPTS=$((CLEANUP_ATTEMPTS + 1))
    done

    echo "  Cleanup complete: ${TOTAL_SIZE}MB / ${MAX_TOTAL_MB}MB"
fi

# Final summary
echo ""
echo "=== Weekly Consolidation Summary ==="

# Count files
DAILY_COUNT=$(find "$PERMANENT_SNAPSHOT_DIR" -maxdepth 1 -name "permanent_snapshot_*.tar*" -type f 2>/dev/null | wc -l | tr -d ' ')
WEEKLY_COUNT=$(find "$WEEKLY_ARCHIVE_DIR" -name "weekly_snapshot_*.tar*" -type f 2>/dev/null | wc -l | tr -d ' ')
DAILY_SIZE=$(get_size_mb "$PERMANENT_SNAPSHOT_DIR")

# Calculate weekly archive size separately
WEEKLY_SIZE=0
if [ -d "$WEEKLY_ARCHIVE_DIR" ]; then
    WEEKLY_SIZE=$(get_size_mb "$WEEKLY_ARCHIVE_DIR")
fi

echo "Daily snapshots: $DAILY_COUNT files"
echo "Weekly archives: $WEEKLY_COUNT files (${WEEKLY_SIZE}MB)"
echo "Total size: ${DAILY_SIZE}MB / ${MAX_TOTAL_MB}MB"

if [ "$DAILY_SIZE" -le "$MAX_TOTAL_MB" ]; then
    echo "✅ Within 1GB cap"
else
    echo "⚠️  Still over cap (will retry next week)"
fi

echo ""
echo "=== Consolidation Complete ==="

# BULLETPROOF: Always exit success
exit 0
