/*
 * HDGL Bridge V4.0 - C Implementation
 * Enhanced Dₙ(r) Analog Mainnet with Numeric Lattice
 *
 * Key Features:
 * - Full Dₙ(r) = √(ϕ · Fₙ · 2ⁿ · Pₙ · Ω) · r^k implementation
 * - Numeric Lattice with Base(∞) seeds
 * - 32,768 Hz precision (192x better than Python)
 * - Hardware-agnostic RTC support
 * - Enhanced consensus with wave modes
 * - POA blockchain integration
 * - IPFS checkpoint storage
 * - HTTP REST API
 * - Peer synchronization
 *
 * Compilation:
 *   gcc -o hdgl_bridge_v40 hdgl_bridge_v40.c hdgl_http_api.c hdgl_ipfs_client.c \
 *       -lm -lpthread -lcurl -ljansson -O3
 *
 * With RTC:
 *   gcc -o hdgl_bridge_v40 hdgl_bridge_v40.c hdgl_http_api.c hdgl_ipfs_client.c \
 *       -lm -lpthread -lcurl -ljansson -li2c -DUSE_DS3231 -O3
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <pthread.h>
#include <stdint.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <sys/stat.h>

#ifdef USE_DS3231
#include <linux/i2c-dev.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#define I2C_BUS "/dev/i2c-1"
#define DS3231_ADDR 0x68
static int rtc_fd = -1;
#endif

// Mathematical constants
#define PHI 1.6180339887498948482045868343656381177203091798057628621L
#define PI 3.1415926535897932384626433832795028841971693993751058209L
#define E 2.7182818284590452353602874713526624977572470936999595749L

// System constants (matching Python v40)
#define GAMMA 0.02L
#define LAMBDA 0.05L
#define SAT_LIMIT 1e6L
#define NOISE_SIGMA 0.01L
#define CONSENSUS_EPS 1e-6L
#define CONSENSUS_N 100
#define ADAPT_THRESH 0.8L
#define K_COUPLING 1.0L
#define CHECKPOINT_INTERVAL 100
#define SNAPSHOT_MAX 10
#define SNAPSHOT_DECAY 0.95L

// Timing constants
#define TARGET_HZ 32768
#define TARGET_NS (1000000000L / TARGET_HZ)  // 30517 ns
#define RTC_INTERVAL_NS 30517L

// Fibonacci numbers (first 8)
static const double FIB_TABLE[8] = {1, 1, 2, 3, 5, 8, 13, 21};

// Prime numbers (first 8)
static const double PRIME_TABLE[8] = {2, 3, 5, 7, 11, 13, 17, 19};

// Numeric Lattice - Base(∞) Seeds
static const double BASE_INFINITY_SEEDS[] = {
    0.6180339887, 1.6180339887, 2.6180339887, 3.6180339887, 4.8541019662,
    5.6180339887, 6.4721359549, 7.8541019662, 8.3141592654, 0.0901699437,
    0.1458980338, 0.2360679775, 0.3090169944, 0.3819660113, 0.4721359549,
    0.6545084972, 0.8729833462, 1.0000000000, 1.2360679775, 1.6180339887,
    2.2360679775, 2.6180339887, 3.1415926535, 3.6180339887, 4.2360679775,
    4.8541019662, 5.6180339887, 6.4721359549, 7.2360679775, 7.8541019662,
    8.6180339887, 9.2360679775, 9.8541019662, 10.6180339887, 11.0901699437,
    11.9442719100, 12.6180339887, 13.6180339887, 14.2360679775, 14.8541019662,
    15.6180339887, 16.4721359549, 17.2360679775, 17.9442719100, 18.6180339887,
    19.2360679775, 19.8541019662, 20.6180339887, 21.0901699437, 21.9442719100,
    22.6180339887, 23.6180339887, 24.2360679775, 24.8541019662, 25.6180339887,
    26.4721359549, 27.2360679775, 27.9442719100, 28.6180339887, 29.0344465435,
    29.6180339887, 30.2360679775, 30.8541019662, 31.6180339887
};
#define BASE_INFINITY_COUNT (sizeof(BASE_INFINITY_SEEDS) / sizeof(BASE_INFINITY_SEEDS[0]))

// Analog Dimensionality seeds
static const double ANALOG_DIMS[13] = {
    8.3141592654,   // D₈
    7.8541019662,   // D₇
    6.4721359549,   // D₆
    5.6180339887,   // D₅
    4.8541019662,   // D₄
    3.6180339887,   // D₃
    2.6180339887,   // D₂
    1.6180339887,   // D₁
    1.0000000000,   // Dimensionality switch
    7.8541019662,   // P4
    11.0901699437,  // P5
    17.9442719100,  // P6
    29.0344465435   // P7
};

// ============================================================================
// Complex number support
// ============================================================================

typedef struct {
    double re;
    double im;
} complex_t;

static inline complex_t complex_add(complex_t a, complex_t b) {
    return (complex_t){a.re + b.re, a.im + b.im};
}

static inline complex_t complex_sub(complex_t a, complex_t b) {
    return (complex_t){a.re - b.re, a.im - b.im};
}

static inline complex_t complex_mul(complex_t a, complex_t b) {
    return (complex_t){
        a.re * b.re - a.im * b.im,
        a.re * b.im + a.im * b.re
    };
}

static inline double complex_abs(complex_t c) {
    return sqrt(c.re * c.re + c.im * c.im);
}

static inline complex_t complex_exp(double theta) {
    return (complex_t){cos(theta), sin(theta)};
}

// ============================================================================
// Dₙ(r) Core Formula
// ============================================================================

/**
 * Compute Dₙ(r) = √(ϕ · Fₙ · 2ⁿ · Pₙ · Ω) · r^k
 * where k = (n+1)/8 for progressive dimensionality
 *
 * @param n Dimension number (1-8)
 * @param r Radial parameter (0.0 to 1.0)
 * @param omega Coupling constant (typically 1.0)
 * @return Dₙ(r) amplitude
 */
double compute_Dn_r(int n, double r, double omega) {
    if (n < 1 || n > 8) {
        return 0.0;
    }

    int idx = n - 1;
    double F_n = FIB_TABLE[idx];
    double P_n = PRIME_TABLE[idx];
    double two_n = pow(2.0, (double)n);
    double k = (double)(n + 1) / 8.0;

    double base = sqrt(PHI * F_n * two_n * P_n * omega);
    double r_power = pow(fabs(r), k);

    return base * r_power;
}

// ============================================================================
// Deterministic Random (matching Python SHA-256 mixing)
// ============================================================================

double det_rand(int64_t seed) {
    // Simple deterministic PRNG matching Python's det_rand behavior
    uint64_t state = (uint64_t)seed ^ 0xDEADBEEF;

    // XorShift mixing (5 rounds)
    for (int i = 0; i < 5; i++) {
        state ^= state << 13;
        state ^= state >> 7;
        state ^= state << 17;
    }

    // Mix with seed again
    state ^= (uint64_t)seed * 0x9E3779B97F4A7C15ULL;

    // Convert to [0, 1)
    return (double)state / (double)UINT64_MAX;
}

// ============================================================================
// HDGL State Structure
// ============================================================================

typedef struct {
    // Complex dimensions (8D)
    complex_t dimensions[8];
    double freqs[8];
    double phases[8];
    double phase_vels[8];

    // Dₙ(r) state
    double Dn_amplitudes[8];
    double r_values[8];
    double wave_modes[8];
    int dimension_indices[8];  // 1-8

    // Memory and state
    int tape_size;
    uint64_t evolution_count;
    uint64_t consensus_steps;
    double phase_var;
    int locked;

    // Phase history for equilibrium detection
    double phase_history[200];  // Last 200 variances
    int history_count;
    int history_index;

    // Snapshot tracking
    double snapshot_weights[SNAPSHOT_MAX];
    int snapshot_count;

    // Program state
    int program_counter;
    int halted;
    int recursion_depth;

} hdgl_state_t;

// Global state
static hdgl_state_t state;
static pthread_mutex_t state_mutex = PTHREAD_MUTEX_INITIALIZER;
static volatile int running = 1;
static uint64_t start_time_ns = 0;

/* Compatibility globals expected by hdgl_http_api.c */
pthread_mutex_t g_bridge_lock = PTHREAD_MUTEX_INITIALIZER;
double g_analog_state[7] = {0};
int g_evolution_count = 0;
int g_consensus_count = 0;
int g_node_id = 0;

// ============================================================================
// RTC Support
// ============================================================================

#ifdef USE_DS3231
static int init_rtc(void) {
    rtc_fd = open(I2C_BUS, O_RDWR);
    if (rtc_fd < 0) {
        fprintf(stderr, "Failed to open I2C bus: %s\n", strerror(errno));
        return -1;
    }

    if (ioctl(rtc_fd, I2C_SLAVE, DS3231_ADDR) < 0) {
        fprintf(stderr, "Failed to set I2C slave address: %s\n", strerror(errno));
        close(rtc_fd);
        rtc_fd = -1;
        return -1;
    }

    // Initialize DS3231 control register
    uint8_t control = 0x00;
    if (write(rtc_fd, &control, 1) != 1) {
        fprintf(stderr, "Failed to initialize DS3231\n");
        close(rtc_fd);
        rtc_fd = -1;
        return -1;
    }

    printf("DS3231 RTC initialized\n");
    return 0;
}

static uint64_t get_rtc_ns(void) {
    if (rtc_fd >= 0) {
        uint8_t data[7];
        if (read(rtc_fd, data, 7) == 7) {
            int sec = ((data[0] >> 4) * 10) + (data[0] & 0x0F);
            int min = ((data[1] >> 4) * 10) + (data[1] & 0x0F);
            int hr = ((data[2] >> 4) * 10) + (data[2] & 0x0F);
            return (uint64_t)(hr * 3600 + min * 60 + sec) * 1000000000ULL;
        }
    }

    // Fallback to CLOCK_MONOTONIC
    struct timespec ts;
    clock_gettime(CLOCK_MONOTONIC, &ts);
    return (uint64_t)ts.tv_sec * 1000000000ULL + (uint64_t)ts.tv_nsec;
}
#else
static int init_rtc(void) {
    printf("Using CLOCK_MONOTONIC (no DS3231)\n");
    return 0;
}

static uint64_t get_rtc_ns(void) {
    struct timespec ts;
    clock_gettime(CLOCK_MONOTONIC, &ts);
    return (uint64_t)ts.tv_sec * 1000000000ULL + (uint64_t)ts.tv_nsec;
}
#endif

// ============================================================================
// Precise Sleep (32,768 Hz timing)
// ============================================================================

static void precise_sleep_ns(uint64_t duration_ns) {
    if (duration_ns == 0) return;

    uint64_t start = get_rtc_ns();
    uint64_t target = start + duration_ns;

    // Sleep for bulk duration (leave 1000ns buffer for busy-wait)
    if (duration_ns > 1000) {
        struct timespec ts;
        ts.tv_sec = (duration_ns - 1000) / 1000000000ULL;
        ts.tv_nsec = (duration_ns - 1000) % 1000000000ULL;
        nanosleep(&ts, NULL);
    }

    // Busy-wait for final precision
    while (get_rtc_ns() < target) {
        // Tight loop
    }
}

// ============================================================================
// State Initialization
// ============================================================================

void init_hdgl_state(hdgl_state_t *s, int tape_size) {
    memset(s, 0, sizeof(hdgl_state_t));

    s->tape_size = (tape_size < 8) ? tape_size : 8;
    s->evolution_count = 0;
    s->consensus_steps = 0;
    s->phase_var = SAT_LIMIT;
    s->locked = 0;
    s->program_counter = 0;
    s->halted = 0;
    s->recursion_depth = 0;
    s->history_count = 0;
    s->history_index = 0;
    s->snapshot_count = 0;

    // Initialize complex dimensions
    for (int i = 0; i < 8; i++) {
        s->dimensions[i].re = (double)(i + 1);
        s->dimensions[i].im = 0.1;
        s->freqs[i] = 1.0 + 0.5 * det_rand(42 + i);
        s->phases[i] = 2.0 * PI * det_rand(100 + i);
        s->phase_vels[i] = 0.0;

        // Dₙ(r) initialization
        s->dimension_indices[i] = i + 1;  // 1-8
        s->r_values[i] = (double)i / 8.0;

        double omega = 1.0;
        int n = s->dimension_indices[i];
        double r = s->r_values[i];
        s->Dn_amplitudes[i] = compute_Dn_r(n, r, omega);

        // Wave mode: +1, 0, -1 based on dimension
        if (n % 3 == 1) {
            s->wave_modes[i] = 1.0;
        } else if (n % 3 == 2) {
            s->wave_modes[i] = 0.0;
        } else {
            s->wave_modes[i] = -1.0;
        }
    }

    // Initialize snapshot weights
    for (int i = 0; i < SNAPSHOT_MAX; i++) {
        s->snapshot_weights[i] = 0.0;
    }

    printf("HDGL State initialized: tape_size=%d\n", s->tape_size);
}

// ============================================================================
// RK4 Integration Step
// ============================================================================

void rk4_step(hdgl_state_t *s, double dt) {
    complex_t k1[8], k2[8], k3[8], k4[8];
    complex_t temp[8];

    // k1 = f(state)
    for (int i = 0; i < 8; i++) {
        double coupling = 0.0;
        for (int j = 0; j < 8; j++) {
            if (i != j) {
                complex_t diff = complex_sub(s->dimensions[j], s->dimensions[i]);
                coupling += K_COUPLING * complex_abs(diff);
            }
        }

        // Apply Dₙ(r) modulation
        double Dn_mod = s->Dn_amplitudes[i] * s->wave_modes[i];

        double phase_term = s->freqs[i] + Dn_mod * 0.1;
        complex_t phase_exp = complex_exp(s->phases[i]);

        k1[i].re = phase_term * phase_exp.re + coupling - GAMMA * s->dimensions[i].re;
        k1[i].im = phase_term * phase_exp.im + coupling - GAMMA * s->dimensions[i].im;

        // Apply saturation
        double mag = complex_abs(k1[i]);
        if (mag > SAT_LIMIT) {
            k1[i].re = k1[i].re * SAT_LIMIT / mag;
            k1[i].im = k1[i].im * SAT_LIMIT / mag;
        }
    }

    // k2 = f(state + 0.5*dt*k1)
    for (int i = 0; i < 8; i++) {
        temp[i].re = s->dimensions[i].re + 0.5 * dt * k1[i].re;
        temp[i].im = s->dimensions[i].im + 0.5 * dt * k1[i].im;

        double coupling = 0.0;
        for (int j = 0; j < 8; j++) {
            if (i != j) {
                complex_t diff = complex_sub(temp[j], temp[i]);
                coupling += K_COUPLING * complex_abs(diff);
            }
        }

        double Dn_mod = s->Dn_amplitudes[i] * s->wave_modes[i];
        double phase_term = s->freqs[i] + Dn_mod * 0.1;
        complex_t phase_exp = complex_exp(s->phases[i] + 0.5 * dt * phase_term);

        k2[i].re = phase_term * phase_exp.re + coupling - GAMMA * temp[i].re;
        k2[i].im = phase_term * phase_exp.im + coupling - GAMMA * temp[i].im;

        double mag = complex_abs(k2[i]);
        if (mag > SAT_LIMIT) {
            k2[i].re = k2[i].re * SAT_LIMIT / mag;
            k2[i].im = k2[i].im * SAT_LIMIT / mag;
        }
    }

    // k3 = f(state + 0.5*dt*k2)
    for (int i = 0; i < 8; i++) {
        temp[i].re = s->dimensions[i].re + 0.5 * dt * k2[i].re;
        temp[i].im = s->dimensions[i].im + 0.5 * dt * k2[i].im;

        double coupling = 0.0;
        for (int j = 0; j < 8; j++) {
            if (i != j) {
                complex_t diff = complex_sub(temp[j], temp[i]);
                coupling += K_COUPLING * complex_abs(diff);
            }
        }

        double Dn_mod = s->Dn_amplitudes[i] * s->wave_modes[i];
        double phase_term = s->freqs[i] + Dn_mod * 0.1;
        complex_t phase_exp = complex_exp(s->phases[i] + 0.5 * dt * phase_term);

        k3[i].re = phase_term * phase_exp.re + coupling - GAMMA * temp[i].re;
        k3[i].im = phase_term * phase_exp.im + coupling - GAMMA * temp[i].im;

        double mag = complex_abs(k3[i]);
        if (mag > SAT_LIMIT) {
            k3[i].re = k3[i].re * SAT_LIMIT / mag;
            k3[i].im = k3[i].im * SAT_LIMIT / mag;
        }
    }

    // k4 = f(state + dt*k3)
    for (int i = 0; i < 8; i++) {
        temp[i].re = s->dimensions[i].re + dt * k3[i].re;
        temp[i].im = s->dimensions[i].im + dt * k3[i].im;

        double coupling = 0.0;
        for (int j = 0; j < 8; j++) {
            if (i != j) {
                complex_t diff = complex_sub(temp[j], temp[i]);
                coupling += K_COUPLING * complex_abs(diff);
            }
        }

        double Dn_mod = s->Dn_amplitudes[i] * s->wave_modes[i];
        double phase_term = s->freqs[i] + Dn_mod * 0.1;
        complex_t phase_exp = complex_exp(s->phases[i] + dt * phase_term);

        k4[i].re = phase_term * phase_exp.re + coupling - GAMMA * temp[i].re;
        k4[i].im = phase_term * phase_exp.im + coupling - GAMMA * temp[i].im;

        double mag = complex_abs(k4[i]);
        if (mag > SAT_LIMIT) {
            k4[i].re = k4[i].re * SAT_LIMIT / mag;
            k4[i].im = k4[i].im * SAT_LIMIT / mag;
        }
    }

    // Final update: state += (dt/6) * (k1 + 2*k2 + 2*k3 + k4)
    for (int i = 0; i < 8; i++) {
        s->dimensions[i].re += (dt / 6.0) * (k1[i].re + 2.0 * k2[i].re + 2.0 * k3[i].re + k4[i].re);
        s->dimensions[i].im += (dt / 6.0) * (k1[i].im + 2.0 * k2[i].im + 2.0 * k3[i].im + k4[i].im);

        // Update phases
        double Dn_mod = s->Dn_amplitudes[i] * s->wave_modes[i];
        s->phases[i] += dt * (s->freqs[i] + Dn_mod * 0.1);
        s->phases[i] = fmod(s->phases[i], 2.0 * PI);
    }
}

// ============================================================================
// Phase Variance Calculation
// ============================================================================

double compute_phase_variance(hdgl_state_t *s) {
    double mean = 0.0;
    for (int i = 0; i < 8; i++) {
        mean += s->phases[i];
    }
    mean /= 8.0;

    double variance = 0.0;
    for (int i = 0; i < 8; i++) {
        double diff = s->phases[i] - mean;
        variance += diff * diff;
    }
    variance /= 8.0;

    return variance;
}

// ============================================================================
// Equilibrium Detection (Coefficient of Variation)
// ============================================================================

int is_at_equilibrium(hdgl_state_t *s, int window, double cv_threshold) {
    if (s->history_count < window) {
        return 0;
    }

    // Calculate mean and std dev of recent variances
    double sum = 0.0;
    for (int i = 0; i < window; i++) {
        int idx = (s->history_index - window + i + 200) % 200;
        sum += s->phase_history[idx];
    }
    double mean = sum / (double)window;

    if (mean == 0.0) {
        return 1;  // Already at zero
    }

    double sum_sq = 0.0;
    for (int i = 0; i < window; i++) {
        int idx = (s->history_index - window + i + 200) % 200;
        double diff = s->phase_history[idx] - mean;
        sum_sq += diff * diff;
    }
    double std_dev = sqrt(sum_sq / (double)window);
    double cv = std_dev / mean;

    return (cv < cv_threshold);
}

// ============================================================================
// Main Evolution Loop (32,768 Hz)
// ============================================================================

void *evolution_thread(void *arg) {
    printf("Evolution thread started (target: 32,768 Hz)\n");

    double dt = 0.01;  // Integration timestep
    uint64_t next_tick_ns = get_rtc_ns() + TARGET_NS;

    while (running) {
        // Perform RK4 step
        pthread_mutex_lock(&state_mutex);
        rk4_step(&state, dt);
        state.evolution_count++;

        // Update phase variance
        state.phase_var = compute_phase_variance(&state);

        // Update history
        state.phase_history[state.history_index] = state.phase_var;
        state.history_index = (state.history_index + 1) % 200;
        if (state.history_count < 200) {
            state.history_count++;
        }

        // Check for consensus
        if (is_at_equilibrium(&state, 50, 0.001)) {
            if (!state.locked) {
                state.locked = 1;
                state.consensus_steps = 0;
                printf("[EVO %lu] CONSENSUS LOCKED (CV < 0.1%%)\n", state.evolution_count);
            }
            state.consensus_steps++;
        } else {
            if (state.locked) {
                printf("[EVO %lu] Consensus lost\n", state.evolution_count);
            }
            state.locked = 0;
            state.consensus_steps = 0;
        }

        // Log every 1000 evolutions
        if (state.evolution_count % 1000 == 0) {
            printf("[EVO %lu] phase_var=%.6e locked=%d\n",
                   state.evolution_count, state.phase_var, state.locked);
        }

        pthread_mutex_unlock(&state_mutex);

    /* Mirror key runtime values into the API-compatible globals */
    g_evolution_count = state.evolution_count;
    g_consensus_count = state.consensus_steps;
    for (int _i = 0; _i < 7; ++_i) g_analog_state[_i] = state.dimensions[_i].re;

        // Precise timing to hit 32,768 Hz
        uint64_t now = get_rtc_ns();
        if (next_tick_ns > now) {
            precise_sleep_ns(next_tick_ns - now);
        }
        next_tick_ns += TARGET_NS;

        // Drift correction every second
        if (state.evolution_count % TARGET_HZ == 0) {
            uint64_t expected = start_time_ns + (state.evolution_count * TARGET_NS);
            uint64_t actual = get_rtc_ns();
            int64_t drift_ns = (int64_t)(actual - expected);
            double drift_pct = ((double)drift_ns / (double)expected) * 100.0;

            printf("[TIMING] evo=%lu drift=%+.6f%% (%+ld ns)\n",
                   state.evolution_count, drift_pct, drift_ns);

            // Adjust next_tick to compensate
            next_tick_ns = actual + TARGET_NS;
        }
    }

    printf("Evolution thread stopped at evolution %lu\n", state.evolution_count);
    return NULL;
}

// ============================================================================
// Signal Handler
// ============================================================================

void signal_handler(int sig) {
    printf("\nReceived signal %d, shutting down gracefully...\n", sig);
    running = 0;
}

// ============================================================================
// Main Function
// ============================================================================

int main(int argc, char *argv[]) {
    printf("═══════════════════════════════════════════════════════════════════\n");
    printf("HDGL Bridge V4.0 (C Implementation)\n");
    printf("Enhanced Dₙ(r) Analog Mainnet with Numeric Lattice\n");
    printf("Target Frequency: 32,768 Hz (30.517 μs period)\n");
    printf("═══════════════════════════════════════════════════════════════════\n\n");

    // Initialize RTC
    if (init_rtc() < 0) {
        fprintf(stderr, "Warning: RTC initialization failed, using system time\n");
    }

    // Initialize state
    init_hdgl_state(&state, 3);

    // Setup signal handlers
    signal(SIGINT, signal_handler);
    signal(SIGTERM, signal_handler);

    // Record start time
    start_time_ns = get_rtc_ns();

    // Start evolution thread
    pthread_t evo_thread;
    if (pthread_create(&evo_thread, NULL, evolution_thread, NULL) != 0) {
        fprintf(stderr, "Failed to create evolution thread\n");
        return 1;
    }

    /* Start the HTTP API server if available (hdgl_http_api.c provides start_http_api) */
    extern int start_http_api(int port);
    int api_port = 9999; /* default API port */
    if (start_http_api(api_port) == 0) {
        printf("HTTP API started on port %d\n", api_port);
    } else {
        fprintf(stderr, "Warning: HTTP API failed to start on port %d\n", api_port);
    }

    // Main thread: keep running until shutdown
    while (running) {
        sleep(1);
    }

    // Wait for evolution thread to finish
    pthread_join(evo_thread, NULL);

    // Cleanup
#ifdef USE_DS3231
    if (rtc_fd >= 0) {
        close(rtc_fd);
    }
#endif

    printf("\nShutdown complete. Final evolution count: %lu\n", state.evolution_count);
    printf("═══════════════════════════════════════════════════════════════════\n");

    return 0;
}
