# 🔌 API REFERENCE - Crème de la Crème

**Complete HTTP API documentation for the perfect 10/10 system**

Base URL: `http://localhost:9998`

---

## 📋 TABLE OF CONTENTS

1. [Authentication](#authentication)
2. [Core Endpoints](#core-endpoints)
3. [Status & Health](#status--health)
4. [Dimensions & State](#dimensions--state)
5. [Consensus](#consensus)
6. [Snapshots & Backups](#snapshots--backups)
7. [Monitoring](#monitoring)
8. [Administration](#administration)
9. [WebSocket API](#websocket-api)
10. [Error Codes](#error-codes)

---

## 🔐 AUTHENTICATION

**Current version uses API key authentication (optional).**

### **With API Key**
```bash
curl -H "X-API-Key: your-api-key-here" \
  http://localhost:9998/api/status
```

### **Without API Key (Local Access)**
```bash
curl http://localhost:9998/api/status
```

**Configuration** (`config/creme.json`):
```json
{
  "api_key_required": false,
  "api_key": "your-secret-key-here"
}
```

---

## 🎯 CORE ENDPOINTS

### **GET /api/status**

Get overall system status.

**Request:**
```bash
curl http://localhost:9998/api/status
```

**Response:**
```json
{
  "status": "running",
  "version": "4.2",
  "architecture": "framework-native",
  "evolution_rate": 32768,
  "precision": "GMP 256-bit",
  "consensus_locked": true,
  "cv": 0.000823,
  "uptime_seconds": 86400,
  "memory_mb": 1.2,
  "binary_size_kb": 50,
  "contexts_size_kb": 120
}
```

**Fields:**
- `status`: "running", "starting", "stopping", "error"
- `version`: Software version
- `architecture`: Always "framework-native"
- `evolution_rate`: Current Hz (should be 32768)
- `precision`: "GMP 256-bit"
- `consensus_locked`: Boolean consensus state
- `cv`: Coefficient of variation (< 0.001 = locked)
- `uptime_seconds`: Time since start
- `memory_mb`: Current memory usage
- `binary_size_kb`: Binary file size
- `contexts_size_kb`: Total context size

---

### **GET /api/version**

Get version information.

**Request:**
```bash
curl http://localhost:9998/api/version
```

**Response:**
```json
{
  "version": "4.2",
  "build_date": "2025-10-31",
  "gmp_version": "6.2.1",
  "features": [
    "framework-native",
    "gmp-256bit",
    "fibonacci-lattice",
    "wu-wei-compression",
    "bulletproof-automation"
  ]
}
```

---

### **GET /api/health**

Health check endpoint (useful for load balancers).

**Request:**
```bash
curl http://localhost:9998/api/health
```

**Response (Healthy):**
```json
{
  "status": "healthy",
  "checks": {
    "evolution_rate": "OK",
    "consensus": "OK",
    "memory": "OK",
    "disk": "OK",
    "contexts": "OK"
  }
}
```

**Response (Unhealthy):**
```json
{
  "status": "unhealthy",
  "checks": {
    "evolution_rate": "OK",
    "consensus": "FAIL",
    "memory": "OK",
    "disk": "WARN",
    "contexts": "OK"
  },
  "errors": [
    "Consensus not locked (CV=0.15%)",
    "Disk usage >90%"
  ]
}
```

**HTTP Status Codes:**
- `200 OK` - Healthy
- `503 Service Unavailable` - Unhealthy

---

## 📊 DIMENSIONS & STATE

### **GET /api/dimensions**

Get current 8-dimensional state vector.

**Request:**
```bash
curl http://localhost:9998/api/dimensions
```

**Response:**
```json
{
  "dimensions": [8.24, 8.89, 7.91, 9.12, 7.68, 8.01, 9.87, 7.12],
  "mean": 8.23,
  "std_dev": 0.68,
  "cv": 0.000823,
  "min": 7.12,
  "max": 9.87,
  "range": 2.75,
  "timestamp": "2025-10-31T14:23:45Z",
  "evolution_count": 1048576
}
```

**Fields:**
- `dimensions`: Array of 8 float values (Dₙ(r) values)
- `mean`: Average of dimensions
- `std_dev`: Standard deviation
- `cv`: Coefficient of variation (std_dev / mean)
- `min`: Minimum dimension value
- `max`: Maximum dimension value
- `range`: max - min
- `timestamp`: ISO 8601 timestamp
- `evolution_count`: Total evolutions completed

---

### **GET /api/dimensions/history**

Get historical dimension data.

**Query Parameters:**
- `limit` (optional): Number of records (default: 100, max: 10000)
- `start` (optional): Start timestamp (ISO 8601)
- `end` (optional): End timestamp (ISO 8601)
- `interval` (optional): Sampling interval in seconds (default: 60)

**Request:**
```bash
curl "http://localhost:9998/api/dimensions/history?limit=10&interval=300"
```

**Response:**
```json
{
  "count": 10,
  "interval": 300,
  "data": [
    {
      "timestamp": "2025-10-31T14:00:00Z",
      "dimensions": [8.24, 8.89, 7.91, 9.12, 7.68, 8.01, 9.87, 7.12],
      "cv": 0.000823
    },
    {
      "timestamp": "2025-10-31T14:05:00Z",
      "dimensions": [8.26, 8.87, 7.93, 9.10, 7.70, 8.03, 9.85, 7.14],
      "cv": 0.000819
    }
  ]
}
```

---

### **GET /api/dimensions/live**

Server-Sent Events (SSE) stream of live dimension updates.

**Request:**
```bash
curl -N http://localhost:9998/api/dimensions/live
```

**Response (streaming):**
```
event: update
data: {"dimensions": [8.24, 8.89, 7.91, 9.12, 7.68, 8.01, 9.87, 7.12], "cv": 0.000823, "timestamp": "2025-10-31T14:23:45Z"}

event: update
data: {"dimensions": [8.26, 8.87, 7.93, 9.10, 7.70, 8.03, 9.85, 7.14], "cv": 0.000819, "timestamp": "2025-10-31T14:23:46Z"}
```

**Update Frequency:** Real-time (as fast as evolution rate allows, typically 1-10 Hz for API)

---

## 🔒 CONSENSUS

### **GET /api/consensus**

Get consensus information.

**Request:**
```bash
curl http://localhost:9998/api/consensus
```

**Response:**
```json
{
  "locked": true,
  "cv": 0.000823,
  "threshold": 0.001,
  "phase": "stable",
  "lock_duration_seconds": 3600,
  "last_lock": "2025-10-31T13:23:45Z",
  "hysteresis": {
    "upper_threshold": 0.0011,
    "lower_threshold": 0.0009,
    "current_cv": 0.000823
  }
}
```

**Fields:**
- `locked`: Boolean consensus state
- `cv`: Current CV value
- `threshold`: Lock threshold (0.001 = 0.1%)
- `phase`: "stable", "locking", "unlocking", "oscillating"
- `lock_duration_seconds`: Time since locked
- `last_lock`: Timestamp of last lock event
- `hysteresis`: Hysteresis thresholds for stability

---

### **GET /api/consensus/history**

Get consensus history over time.

**Query Parameters:**
- `limit` (optional): Number of records (default: 100)
- `start` (optional): Start timestamp
- `end` (optional): End timestamp

**Request:**
```bash
curl "http://localhost:9998/api/consensus/history?limit=20"
```

**Response:**
```json
{
  "count": 20,
  "data": [
    {
      "timestamp": "2025-10-31T14:00:00Z",
      "locked": true,
      "cv": 0.000823,
      "phase": "stable"
    },
    {
      "timestamp": "2025-10-31T14:01:00Z",
      "locked": true,
      "cv": 0.000819,
      "phase": "stable"
    }
  ],
  "statistics": {
    "lock_percentage": 98.5,
    "average_cv": 0.000821,
    "min_cv": 0.000815,
    "max_cv": 0.000890
  }
}
```

---

## 💾 SNAPSHOTS & BACKUPS

### **GET /api/snapshots**

List all available snapshots.

**Request:**
```bash
curl http://localhost:9998/api/snapshots
```

**Response:**
```json
{
  "count": 28,
  "total_size_mb": 2.4,
  "compression_ratio": 25.3,
  "snapshots": [
    {
      "id": "20251031_142345",
      "timestamp": "2025-10-31T14:23:45Z",
      "type": "automatic",
      "contexts": 6,
      "size_compressed_kb": 85,
      "size_uncompressed_kb": 2150,
      "compression_ratio": 25.3,
      "ipfs_hash": "QmXa7c9...",
      "verified": true
    }
  ]
}
```

---

### **POST /api/snapshots**

Create a new snapshot manually.

**Request:**
```bash
curl -X POST http://localhost:9998/api/snapshots \
  -H "Content-Type: application/json" \
  -d '{"type": "manual", "description": "Pre-upgrade snapshot"}'
```

**Response:**
```json
{
  "status": "created",
  "snapshot": {
    "id": "20251031_150000",
    "timestamp": "2025-10-31T15:00:00Z",
    "type": "manual",
    "description": "Pre-upgrade snapshot",
    "contexts": 6,
    "size_compressed_kb": 87,
    "ipfs_hash": "QmYb8d0..."
  }
}
```

---

### **POST /api/snapshots/{id}/restore**

Restore from a snapshot.

**Request:**
```bash
curl -X POST http://localhost:9998/api/snapshots/20251031_142345/restore
```

**Response:**
```json
{
  "status": "restoring",
  "snapshot_id": "20251031_142345",
  "estimated_time_seconds": 30,
  "message": "System will restart after restore completes"
}
```

**Note:** This will stop the service, restore contexts, and restart.

---

### **GET /api/backups**

List IPFS permanent backups.

**Request:**
```bash
curl http://localhost:9998/api/backups
```

**Response:**
```json
{
  "count": 7,
  "backups": [
    {
      "date": "2025-10-31",
      "ipfs_hash": "QmZc9e1...",
      "size_kb": 90,
      "pinned": true,
      "verified": true
    }
  ]
}
```

---

## 📈 MONITORING

### **GET /api/metrics**

Get comprehensive system metrics.

**Request:**
```bash
curl http://localhost:9998/api/metrics
```

**Response:**
```json
{
  "evolution": {
    "rate_hz": 32768,
    "count": 1048576,
    "uptime_seconds": 86400
  },
  "performance": {
    "binary_size_kb": 50,
    "memory_mb": 1.2,
    "startup_time_ms": 850,
    "contexts_size_kb": 120
  },
  "consensus": {
    "locked": true,
    "cv": 0.000823,
    "lock_percentage_24h": 98.5
  },
  "storage": {
    "snapshots_count": 28,
    "snapshots_size_mb": 2.4,
    "backups_count": 7,
    "compression_ratio": 25.3
  },
  "system": {
    "cpu_usage_percent": 2.5,
    "memory_usage_percent": 0.05,
    "disk_usage_percent": 15.3
  }
}
```

---

### **GET /api/metrics/performance**

Get detailed performance metrics.

**Request:**
```bash
curl http://localhost:9998/api/metrics/performance
```

**Response:**
```json
{
  "comparison_to_docker": {
    "size_reduction": "25000x",
    "startup_speedup": "148x",
    "memory_reduction": "100x"
  },
  "current": {
    "binary_size_kb": 50,
    "startup_time_ms": 850,
    "memory_mb": 1.2,
    "context_size_kb": 120
  },
  "docker_equivalent": {
    "binary_size_mb": 500,
    "startup_time_ms": 5000,
    "memory_mb": 120,
    "container_size_mb": 3000
  },
  "benchmarks": {
    "evolution_rate": 32768,
    "api_latency_ms": 5,
    "snapshot_time_seconds": 2,
    "restore_time_seconds": 30
  }
}
```

---

### **GET /api/logs**

Get recent log entries.

**Query Parameters:**
- `level` (optional): Filter by level (DEBUG, INFO, WARN, ERROR)
- `limit` (optional): Number of entries (default: 100)
- `since` (optional): Timestamp to start from

**Request:**
```bash
curl "http://localhost:9998/api/logs?level=ERROR&limit=10"
```

**Response:**
```json
{
  "count": 2,
  "logs": [
    {
      "timestamp": "2025-10-31T14:23:45Z",
      "level": "ERROR",
      "component": "consensus",
      "message": "CV exceeded threshold: 0.0015",
      "context": {
        "cv": 0.0015,
        "threshold": 0.001
      }
    }
  ]
}
```

---

## ⚙️ ADMINISTRATION

### **POST /api/admin/restart**

Restart the system gracefully.

**Request:**
```bash
curl -X POST http://localhost:9998/api/admin/restart \
  -H "X-API-Key: your-api-key"
```

**Response:**
```json
{
  "status": "restarting",
  "message": "System will restart in 5 seconds",
  "expected_downtime_seconds": 10
}
```

---

### **POST /api/admin/snapshot-now**

Trigger immediate snapshot.

**Request:**
```bash
curl -X POST http://localhost:9998/api/admin/snapshot-now
```

**Response:**
```json
{
  "status": "created",
  "snapshot_id": "20251031_152000",
  "size_kb": 87,
  "ipfs_hash": "QmZd0f2..."
}
```

---

### **POST /api/admin/consolidate**

Trigger snapshot consolidation.

**Request:**
```bash
curl -X POST http://localhost:9998/api/admin/consolidate
```

**Response:**
```json
{
  "status": "consolidating",
  "snapshots_processed": 28,
  "space_freed_mb": 15.2,
  "estimated_time_seconds": 120
}
```

---

### **GET /api/admin/config**

Get current configuration.

**Request:**
```bash
curl http://localhost:9998/api/admin/config
```

**Response:**
```json
{
  "evolution_rate": 32768,
  "api_port": 9998,
  "precision_bits": 256,
  "consensus_threshold": 0.001,
  "snapshot_interval": 21600,
  "snapshot_retention": 7,
  "log_level": "INFO"
}
```

---

### **PUT /api/admin/config**

Update configuration (requires restart).

**Request:**
```bash
curl -X PUT http://localhost:9998/api/admin/config \
  -H "Content-Type: application/json" \
  -H "X-API-Key: your-api-key" \
  -d '{"log_level": "DEBUG", "snapshot_interval": 3600}'
```

**Response:**
```json
{
  "status": "updated",
  "message": "Configuration updated. Restart required.",
  "changes": {
    "log_level": {"old": "INFO", "new": "DEBUG"},
    "snapshot_interval": {"old": 21600, "new": 3600}
  }
}
```

---

## 🔌 WEBSOCKET API

### **ws://localhost:9998/ws**

WebSocket connection for real-time updates.

**Connect:**
```javascript
const ws = new WebSocket('ws://localhost:9998/ws');

ws.onmessage = (event) => {
  const data = JSON.parse(event.data);
  console.log(data);
};
```

**Message Types:**

**1. Dimension Update**
```json
{
  "type": "dimensions",
  "timestamp": "2025-10-31T14:23:45Z",
  "data": {
    "dimensions": [8.24, 8.89, 7.91, 9.12, 7.68, 8.01, 9.87, 7.12],
    "cv": 0.000823
  }
}
```

**2. Consensus Change**
```json
{
  "type": "consensus",
  "timestamp": "2025-10-31T14:23:45Z",
  "data": {
    "locked": true,
    "cv": 0.000823,
    "phase": "stable"
  }
}
```

**3. Snapshot Created**
```json
{
  "type": "snapshot",
  "timestamp": "2025-10-31T14:23:45Z",
  "data": {
    "id": "20251031_142345",
    "size_kb": 87,
    "ipfs_hash": "QmXa7c9..."
  }
}
```

**4. Alert**
```json
{
  "type": "alert",
  "level": "warning",
  "timestamp": "2025-10-31T14:23:45Z",
  "message": "Disk usage >80%"
}
```

**Subscribe to specific types:**
```json
{
  "action": "subscribe",
  "types": ["dimensions", "consensus"]
}
```

---

## ❌ ERROR CODES

**HTTP Status Codes:**

| Code | Meaning | Example |
|------|---------|---------|
| 200 | Success | Request completed successfully |
| 201 | Created | Snapshot created |
| 400 | Bad Request | Invalid parameters |
| 401 | Unauthorized | Missing/invalid API key |
| 404 | Not Found | Snapshot ID not found |
| 429 | Rate Limited | Too many requests |
| 500 | Server Error | Internal error |
| 503 | Service Unavailable | System not healthy |

**Error Response Format:**
```json
{
  "error": {
    "code": "CONSENSUS_NOT_LOCKED",
    "message": "Consensus is not currently locked",
    "details": {
      "cv": 0.0015,
      "threshold": 0.001
    },
    "timestamp": "2025-10-31T14:23:45Z"
  }
}
```

**Common Error Codes:**

- `INVALID_PARAMETER` - Parameter validation failed
- `SNAPSHOT_NOT_FOUND` - Snapshot ID doesn't exist
- `CONSENSUS_NOT_LOCKED` - Operation requires locked consensus
- `SYSTEM_BUSY` - System is performing maintenance
- `DISK_FULL` - Insufficient disk space
- `CONTEXT_CORRUPTED` - Context integrity check failed
- `RATE_LIMIT_EXCEEDED` - Too many requests

---

## 📝 EXAMPLES

### **Python Client**

```python
import requests
import json

base_url = "http://localhost:9998"

# Get status
response = requests.get(f"{base_url}/api/status")
status = response.json()
print(f"System running: {status['status']}")
print(f"Consensus locked: {status['consensus_locked']}")

# Get dimensions
response = requests.get(f"{base_url}/api/dimensions")
dims = response.json()
print(f"Current dimensions: {dims['dimensions']}")
print(f"CV: {dims['cv']:.6f}")

# Create snapshot
response = requests.post(
    f"{base_url}/api/snapshots",
    json={"type": "manual", "description": "API test"}
)
snapshot = response.json()
print(f"Snapshot created: {snapshot['snapshot']['id']}")
```

### **JavaScript Client**

```javascript
const baseUrl = 'http://localhost:9998';

// Get status
fetch(`${baseUrl}/api/status`)
  .then(res => res.json())
  .then(data => {
    console.log('Status:', data.status);
    console.log('Consensus:', data.consensus_locked);
  });

// WebSocket for live updates
const ws = new WebSocket('ws://localhost:9998/ws');
ws.onmessage = (event) => {
  const msg = JSON.parse(event.data);
  if (msg.type === 'dimensions') {
    console.log('Dimensions:', msg.data.dimensions);
    console.log('CV:', msg.data.cv);
  }
};
```

### **cURL Examples**

```bash
# Health check
curl http://localhost:9998/api/health

# Get metrics
curl http://localhost:9998/api/metrics | jq .

# Stream live dimensions
curl -N http://localhost:9998/api/dimensions/live

# Create manual snapshot
curl -X POST http://localhost:9998/api/snapshots \
  -H "Content-Type: application/json" \
  -d '{"type": "manual", "description": "Manual backup"}'

# Get snapshot history
curl http://localhost:9998/api/snapshots | jq '.snapshots[] | {id, timestamp, size_compressed_kb}'

# Restart system (with API key)
curl -X POST http://localhost:9998/api/admin/restart \
  -H "X-API-Key: your-secret-key"
```

---

## 📚 ADDITIONAL RESOURCES

- **Full Documentation**: See `docs/ARCHITECTURE.md`
- **Implementation Guide**: See `docs/IMPLEMENTATION_GUIDE.md`
- **Deployment**: See `docs/DEPLOYMENT_CHECKLIST.md`
- **Quick Start**: See `docs/QUICKSTART.md`

---

**API Version**: 1.0
**Last Updated**: 2025-10-31
**Status**: Production Ready
