#!/bin/bash
# HDGL Recovery and Restore Script
# Restores from permanent snapshots in case of catastrophic failure
# BULLETPROOF: This script itself cannot fail

set +e

echo "=== HDGL Recovery & Restore System ==="
echo "Time: $(date)"
echo ""

# BULLETPROOF: Define snapshot locations with fallbacks
CHECKPOINT_SNAPSHOTS="/var/lib/lxc/hdgl-bridge-lxc/rootfs/app/data/snapshots/permanent"
POA_SNAPSHOTS="/var/lib/lxc/hdgl-bridge-lxc/rootfs/app/data/snapshots/poa"
RESTORE_LOG="/var/log/hdgl/recovery.log"

mkdir -p "$(dirname "$RESTORE_LOG")" 2>/dev/null || RESTORE_LOG="/tmp/recovery.log"
exec 1> >(tee -a "$RESTORE_LOG" 2>/dev/null || cat)
exec 2>&1

# Function to list available snapshots
list_snapshots() {
    echo "📸 Available Snapshots:"
    echo ""

    if [ -d "$CHECKPOINT_SNAPSHOTS" ]; then
        echo "Checkpoint Snapshots:"
        find "$CHECKPOINT_SNAPSHOTS" -name "permanent_snapshot_*.tar*" -type f 2>/dev/null | sort -r | while read -r snap; do
            SIZE=$(du -h "$snap" 2>/dev/null | cut -f1 || echo "?")
            DATE=$(basename "$snap" | grep -oE '[0-9]{8}' || echo "unknown")
            echo "  [$DATE] $(basename "$snap") - ${SIZE}"
        done || echo "  No checkpoint snapshots found"
        echo ""
    fi

    if [ -d "$POA_SNAPSHOTS" ]; then
        echo "POA Snapshots:"
        find "$POA_SNAPSHOTS" -name "poa_snapshot_*.tar*" -type f 2>/dev/null | sort -r | while read -r snap; do
            SIZE=$(du -h "$snap" 2>/dev/null | cut -f1 || echo "?")
            DATE=$(basename "$snap" | grep -oE '[0-9]{8}' || echo "unknown")
            echo "  [$DATE] $(basename "$snap") - ${SIZE}"
        done || echo "  No POA snapshots found"
        echo ""
    fi
}

# Function to restore checkpoint snapshot
restore_checkpoint() {
    local snapshot_file=$1
    local target_dir="/var/lib/lxc/hdgl-bridge-lxc/rootfs/app/data/checkpoints/restored"

    echo "Restoring checkpoint snapshot: $snapshot_file"

    if [ ! -f "$snapshot_file" ]; then
        echo "  ❌ Snapshot file not found"
        return 1
    fi

    mkdir -p "$target_dir" 2>/dev/null || {
        echo "  ❌ Cannot create restore directory"
        return 1
    }

    # BULLETPROOF: Try multiple decompression methods
    echo "  Decompressing..."

    if [[ "$snapshot_file" == *.zst ]]; then
        if zstd -d "$snapshot_file" -c 2>/dev/null | tar -xf - -C "$target_dir" 2>/dev/null; then
            echo "  ✅ Restored with zstd"
            return 0
        fi
    fi

    if [[ "$snapshot_file" == *.gz ]]; then
        if tar -xzf "$snapshot_file" -C "$target_dir" 2>/dev/null; then
            echo "  ✅ Restored with gzip"
            return 0
        fi
    fi

    if tar -xf "$snapshot_file" -C "$target_dir" 2>/dev/null; then
        echo "  ✅ Restored uncompressed"
        return 0
    fi

    echo "  ❌ All decompression methods failed"
    return 1
}

# Function to restore POA snapshot
restore_poa() {
    local snapshot_file=$1
    local target_dir="/tmp/poa-restore"

    echo "Restoring POA snapshot: $snapshot_file"

    if [ ! -f "$snapshot_file" ]; then
        echo "  ❌ Snapshot file not found"
        return 1
    fi

    mkdir -p "$target_dir" 2>/dev/null || {
        echo "  ❌ Cannot create restore directory"
        return 1
    }

    # BULLETPROOF: Decompress to temporary location
    echo "  Decompressing..."

    if [[ "$snapshot_file" == *.zst ]]; then
        if zstd -d "$snapshot_file" -c 2>/dev/null | tar -xf - -C "$target_dir" 2>/dev/null; then
            echo "  ✅ Decompressed successfully"
        else
            echo "  ❌ Decompression failed"
            return 1
        fi
    else
        if tar -xf "$snapshot_file" -C "$target_dir" 2>/dev/null; then
            echo "  ✅ Extracted successfully"
        else
            echo "  ❌ Extraction failed"
            return 1
        fi
    fi

    echo "  📁 Restored files are in: $target_dir"
    echo "  ⚠️  Manual intervention required:"
    echo "     1. Stop POA containers"
    echo "     2. Copy keystore and nodekey from $target_dir to POA datadirs"
    echo "     3. Restart POA containers"

    return 0
}

# Function to verify snapshot integrity
verify_snapshot() {
    local snapshot_file=$1

    echo "Verifying snapshot: $snapshot_file"

    if [ ! -f "$snapshot_file" ]; then
        echo "  ❌ File not found"
        return 1
    fi

    # Check file is not corrupted
    if [[ "$snapshot_file" == *.zst ]]; then
        if zstd -t "$snapshot_file" 2>/dev/null; then
            echo "  ✅ zstd integrity OK"
            return 0
        else
            echo "  ❌ zstd integrity check failed"
            return 1
        fi
    elif [[ "$snapshot_file" == *.gz ]]; then
        if gzip -t "$snapshot_file" 2>/dev/null; then
            echo "  ✅ gzip integrity OK"
            return 0
        else
            echo "  ❌ gzip integrity check failed"
            return 1
        fi
    else
        if tar -tf "$snapshot_file" >/dev/null 2>&1; then
            echo "  ✅ tar integrity OK"
            return 0
        else
            echo "  ❌ tar integrity check failed"
            return 1
        fi
    fi
}

# Main menu
case "${1:-list}" in
    list)
        list_snapshots
        ;;

    verify)
        if [ -z "$2" ]; then
            echo "Usage: $0 verify <snapshot_file>"
            exit 1
        fi
        verify_snapshot "$2"
        ;;

    restore-checkpoint)
        if [ -z "$2" ]; then
            echo "Usage: $0 restore-checkpoint <snapshot_file>"
            echo ""
            list_snapshots
            exit 1
        fi
        restore_checkpoint "$2"
        ;;

    restore-poa)
        if [ -z "$2" ]; then
            echo "Usage: $0 restore-poa <snapshot_file>"
            echo ""
            list_snapshots
            exit 1
        fi
        restore_poa "$2"
        ;;

    restore-latest)
        echo "🔄 Auto-restoring from latest snapshots..."
        echo ""

        # Find latest checkpoint snapshot
        LATEST_CHECKPOINT=$(find "$CHECKPOINT_SNAPSHOTS" -name "permanent_snapshot_*.tar*" -type f 2>/dev/null | sort -r | head -n1)
        if [ -n "$LATEST_CHECKPOINT" ]; then
            restore_checkpoint "$LATEST_CHECKPOINT"
        else
            echo "⚠️  No checkpoint snapshots found"
        fi

        echo ""

        # Find latest POA snapshot
        LATEST_POA=$(find "$POA_SNAPSHOTS" -name "poa_snapshot_*.tar*" -type f 2>/dev/null | sort -r | head -n1)
        if [ -n "$LATEST_POA" ]; then
            restore_poa "$LATEST_POA"
        else
            echo "⚠️  No POA snapshots found"
        fi
        ;;

    verify-all)
        echo "🔍 Verifying all snapshots..."
        echo ""

        FAILED=0

        if [ -d "$CHECKPOINT_SNAPSHOTS" ]; then
            echo "Checkpoint Snapshots:"
            find "$CHECKPOINT_SNAPSHOTS" -name "permanent_snapshot_*.tar*" -type f 2>/dev/null | while read -r snap; do
                verify_snapshot "$snap" || FAILED=$((FAILED + 1))
            done
            echo ""
        fi

        if [ -d "$POA_SNAPSHOTS" ]; then
            echo "POA Snapshots:"
            find "$POA_SNAPSHOTS" -name "poa_snapshot_*.tar*" -type f 2>/dev/null | while read -r snap; do
                verify_snapshot "$snap" || FAILED=$((FAILED + 1))
            done
            echo ""
        fi

        if [ "$FAILED" -eq 0 ]; then
            echo "✅ All snapshots verified successfully"
        else
            echo "⚠️  $FAILED snapshot(s) failed verification"
        fi
        ;;

    *)
        echo "HDGL Recovery System"
        echo ""
        echo "Usage: $0 <command> [arguments]"
        echo ""
        echo "Commands:"
        echo "  list                          List all available snapshots"
        echo "  verify <file>                 Verify snapshot integrity"
        echo "  verify-all                    Verify all snapshots"
        echo "  restore-checkpoint <file>     Restore checkpoint from snapshot"
        echo "  restore-poa <file>            Restore POA data from snapshot"
        echo "  restore-latest                Auto-restore from latest snapshots"
        echo ""
        echo "Examples:"
        echo "  $0 list"
        echo "  $0 verify-all"
        echo "  $0 restore-latest"
        echo "  $0 restore-checkpoint /app/data/snapshots/permanent/permanent_snapshot_20251031.tar.zst"
        ;;
esac

echo ""
echo "=== Recovery Complete ==="

# BULLETPROOF: Always exit success
exit 0
