# Wu-Wei Compression Test Build Script
# Windows PowerShell version

Write-Host "`n╔════════════════════════════════════════════════════════════╗" -ForegroundColor Cyan
Write-Host "║         WU-WEI COMPRESSION TEST BUILD                      ║" -ForegroundColor Cyan
Write-Host "╚════════════════════════════════════════════════════════════╝`n" -ForegroundColor Cyan

$SrcDir = "src"
$TestFile = "$SrcDir\test_wu_wei.c"
$CompressFile = "$SrcDir\wu_wei_compress.c"
$OutputExe = "test_wu_wei.exe"

# Check if source files exist
if (-not (Test-Path $TestFile)) {
    Write-Host "❌ Error: $TestFile not found" -ForegroundColor Red
    exit 1
}

if (-not (Test-Path $CompressFile)) {
    Write-Host "❌ Error: $CompressFile not found" -ForegroundColor Red
    exit 1
}

Write-Host "✓ Source files found" -ForegroundColor Green

# Try to find GCC
$GccPath = Get-Command gcc -ErrorAction SilentlyContinue

if ($null -eq $GccPath) {
    Write-Host "`n❌ GCC not found in PATH" -ForegroundColor Red
    Write-Host "`nTo build this test, you need:" -ForegroundColor Yellow
    Write-Host "  1. GCC compiler (MinGW-w64 or similar)" -ForegroundColor Yellow
    Write-Host "  2. zlib library (libz)" -ForegroundColor Yellow
    Write-Host "`nInstall options:" -ForegroundColor Yellow
    Write-Host "  • MinGW-w64: https://www.mingw-w64.org/" -ForegroundColor Cyan
    Write-Host "  • MSYS2: https://www.msys2.org/" -ForegroundColor Cyan
    Write-Host "  • WSL (Ubuntu): wsl --install" -ForegroundColor Cyan
    Write-Host "`nOnce installed, run:" -ForegroundColor Yellow
    Write-Host "  gcc -o $OutputExe $TestFile $CompressFile -lz -lm" -ForegroundColor White
    exit 1
}

Write-Host "✓ GCC found: $($GccPath.Source)" -ForegroundColor Green

# Try to compile
Write-Host "`nCompiling Wu-Wei compression test..." -ForegroundColor Cyan
Write-Host "Command: gcc -o $OutputExe $TestFile $CompressFile -lz -lm -O2" -ForegroundColor Gray

try {
    $ErrorActionPreference = "Stop"
    & gcc -o $OutputExe $TestFile $CompressFile -lz -lm -O2 2>&1 | Tee-Object -Variable CompileOutput

    if ($LASTEXITCODE -eq 0) {
        Write-Host "`n✅ Compilation successful!" -ForegroundColor Green

        if (Test-Path $OutputExe) {
            $FileSize = (Get-Item $OutputExe).Length
            Write-Host "   Binary size: $([math]::Round($FileSize/1KB, 2)) KB" -ForegroundColor Green

            Write-Host "`nRunning tests..." -ForegroundColor Cyan
            Write-Host "═══════════════════════════════════════════════════════════`n" -ForegroundColor Gray

            & ".\$OutputExe"

            Write-Host "`n═══════════════════════════════════════════════════════════" -ForegroundColor Gray
            Write-Host "✅ Tests completed successfully!" -ForegroundColor Green
        }
    } else {
        Write-Host "`n❌ Compilation failed!" -ForegroundColor Red
        Write-Host $CompileOutput -ForegroundColor Red

        if ($CompileOutput -match "zlib") {
            Write-Host "`nMissing zlib library. Install with:" -ForegroundColor Yellow
            Write-Host "  MSYS2: pacman -S mingw-w64-x86_64-zlib" -ForegroundColor Cyan
            Write-Host "  Or use WSL Ubuntu: sudo apt-get install zlib1g-dev" -ForegroundColor Cyan
        }
    }
} catch {
    Write-Host "`nBuild error: $_" -ForegroundColor Red
    exit 1
}
