#!/bin/bash
# Wu-Wei Compression Test Build Script
# Linux/macOS/WSL version

set -e

echo ""
echo "╔════════════════════════════════════════════════════════════╗"
echo "║         WU-WEI COMPRESSION TEST BUILD                      ║"
echo "╚════════════════════════════════════════════════════════════╝"
echo ""

SRC_DIR="src"
TEST_FILE="$SRC_DIR/test_wu_wei.c"
COMPRESS_FILE="$SRC_DIR/wu_wei_compress.c"
OUTPUT_EXE="test_wu_wei"

# Check if source files exist
if [ ! -f "$TEST_FILE" ]; then
    echo "❌ Error: $TEST_FILE not found"
    exit 1
fi

if [ ! -f "$COMPRESS_FILE" ]; then
    echo "❌ Error: $COMPRESS_FILE not found"
    exit 1
fi

echo "✓ Source files found"

# Check for GCC
if ! command -v gcc &> /dev/null; then
    echo ""
    echo "❌ GCC not found"
    echo ""
    echo "To build this test, you need:"
    echo "  1. GCC compiler"
    echo "  2. zlib library (libz)"
    echo ""
    echo "Install with:"
    echo "  Ubuntu/Debian: sudo apt-get install gcc zlib1g-dev"
    echo "  macOS: xcode-select --install (or brew install gcc)"
    echo "  Fedora: sudo dnf install gcc zlib-devel"
    echo ""
    exit 1
fi

echo "✓ GCC found: $(gcc --version | head -n1)"

# Check for zlib
if ! ldconfig -p 2>/dev/null | grep -q libz || ! pkg-config --exists zlib 2>/dev/null; then
    echo ""
    echo "⚠️  Warning: zlib might not be installed"
    echo "   If compilation fails, install with:"
    echo "   Ubuntu/Debian: sudo apt-get install zlib1g-dev"
    echo "   macOS: brew install zlib"
    echo ""
fi

# Compile
echo ""
echo "Compiling Wu-Wei compression test..."
echo "Command: gcc -o $OUTPUT_EXE $TEST_FILE $COMPRESS_FILE -lz -lm -O2"
echo ""

if gcc -o "$OUTPUT_EXE" "$TEST_FILE" "$COMPRESS_FILE" -lz -lm -O2 2>&1; then
    echo ""
    echo "✅ Compilation successful!"

    if [ -f "$OUTPUT_EXE" ]; then
        FILE_SIZE=$(stat -f%z "$OUTPUT_EXE" 2>/dev/null || stat -c%s "$OUTPUT_EXE" 2>/dev/null)
        FILE_SIZE_KB=$((FILE_SIZE / 1024))
        echo "   Binary size: ${FILE_SIZE_KB} KB"

        echo ""
        echo "Running tests..."
        echo "═══════════════════════════════════════════════════════════"
        echo ""

        ./"$OUTPUT_EXE"

        echo ""
        echo "═══════════════════════════════════════════════════════════"
        echo "✅ Tests completed successfully!"
        echo ""
    fi
else
    echo ""
    echo "❌ Compilation failed!"
    echo ""
    echo "If you see 'zlib not found' errors, install zlib-dev:"
    echo "  Ubuntu/Debian: sudo apt-get install zlib1g-dev"
    echo "  macOS: brew install zlib"
    echo "  Fedora: sudo dnf install zlib-devel"
    echo ""
    exit 1
fi
