/*
 * HDGL High-Precision Clock Module
 * Generates authentic 32,768 Hz analog lattice clock signal
 * 1:1 correspondence with analog mainnet evolution
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <math.h>
#include <time.h>
#include <string.h>

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

// Crystal oscillator frequency (watch crystal standard)
#define XTAL_FREQ 32768.0
#define DT (1.0 / XTAL_FREQ)  // 30.517578125 microseconds per tick
#define TICKS_PER_EVOLUTION (XTAL_FREQ * 5.0)  // 163,840 ticks per evolution (5 seconds)

// Phase accumulator for continuous waveform
typedef struct {
    double phase;           // Current phase (radians)
    double frequency;       // Current frequency (Hz)
    double phase_variance;  // From analog lattice
    double amplitude;       // Signal amplitude
    uint64_t tick_count;    // 32,768 Hz tick counter
    uint64_t evolution_count; // Evolution counter
    double time_offset;     // Synchronization offset
} ClockState;

// Initialize clock state
void clock_init(ClockState *clk) {
    clk->phase = 0.0;
    clk->frequency = XTAL_FREQ;
    clk->phase_variance = 0.0;
    clk->amplitude = 1.0;
    clk->tick_count = 0;
    clk->evolution_count = 0;
    clk->time_offset = 0.0;
}

// Update clock with analog lattice state
void clock_update_from_lattice(ClockState *clk, double phase_var, uint64_t evolution) {
    clk->phase_variance = phase_var;
    clk->evolution_count = evolution;

    // Modulate frequency based on phase variance (subtle coupling)
    // This creates 1:1 correspondence with analog lattice dynamics
    clk->frequency = XTAL_FREQ * (1.0 + phase_var * 1e-7);

    // Update amplitude based on lattice coupling
    clk->amplitude = 1.0 + 0.1 * sin(phase_var * M_PI);
}

// Generate single 32,768 Hz clock sample
double clock_tick(ClockState *clk) {
    // Advance phase accumulator
    double phase_increment = 2.0 * M_PI * clk->frequency * DT;
    clk->phase += phase_increment;

    // Keep phase in range [0, 2π]
    while (clk->phase >= 2.0 * M_PI) {
        clk->phase -= 2.0 * M_PI;
    }

    // Generate sample with amplitude modulation
    double sample = clk->amplitude * sin(clk->phase);

    // Add subtle phase noise from lattice variance
    double noise = clk->phase_variance * 1e-4 * sin(clk->tick_count * 0.01);
    sample += noise;

    // Increment counters
    clk->tick_count++;

    // Check if evolution boundary crossed
    if (clk->tick_count >= TICKS_PER_EVOLUTION * clk->evolution_count) {
        // Evolution boundary - synchronization point
        clk->time_offset = 0.0;  // Reset offset at each evolution
    }

    return sample;
}

// Generate buffer of samples for streaming
void clock_generate_buffer(ClockState *clk, double *buffer, size_t num_samples) {
    for (size_t i = 0; i < num_samples; i++) {
        buffer[i] = clock_tick(clk);
    }
}

// Get current clock metrics
void clock_get_metrics(ClockState *clk, char *json_output, size_t max_len) {
    snprintf(json_output, max_len,
        "{"
        "\"xtal_freq\":%.1f,"
        "\"current_freq\":%.6f,"
        "\"phase\":%.6f,"
        "\"phase_variance\":%.9f,"
        "\"amplitude\":%.6f,"
        "\"tick_count\":%llu,"
        "\"evolution_count\":%llu,"
        "\"ticks_per_evolution\":%.0f,"
        "\"time_per_tick_us\":%.6f,"
        "\"sync_accuracy\":%.9f"
        "}",
        XTAL_FREQ,
        clk->frequency,
        clk->phase,
        clk->phase_variance,
        clk->amplitude,
        (unsigned long long)clk->tick_count,
        (unsigned long long)clk->evolution_count,
        TICKS_PER_EVOLUTION,
        DT * 1e6,
        fabs(clk->time_offset)
    );
}

// Calculate exact tick count for given evolution
uint64_t clock_evolution_to_ticks(uint64_t evolution) {
    return (uint64_t)(evolution * TICKS_PER_EVOLUTION);
}

// Calculate evolution from tick count
uint64_t clock_ticks_to_evolution(uint64_t ticks) {
    return ticks / (uint64_t)TICKS_PER_EVOLUTION;
}

// Synchronize clock to exact evolution count
void clock_sync_to_evolution(ClockState *clk, uint64_t target_evolution, double phase_var) {
    clk->evolution_count = target_evolution;
    clk->tick_count = clock_evolution_to_ticks(target_evolution);
    clk->phase_variance = phase_var;
    clk->time_offset = 0.0;

    printf("[Clock Sync] Evolution %llu → %llu ticks (%.6f seconds)\n",
           (unsigned long long)target_evolution,
           (unsigned long long)clk->tick_count,
           clk->tick_count * DT);
}

// Main function for standalone testing
#ifdef CLOCK_STANDALONE
int main(int argc, char **argv) {
    ClockState clock;
    clock_init(&clock);

    printf("HDGL 32,768 Hz Clock Generator\n");
    printf("==============================\n");
    printf("Crystal Frequency: %.1f Hz\n", XTAL_FREQ);
    printf("Tick Period: %.6f μs\n", DT * 1e6);
    printf("Ticks per Evolution: %.0f (5 seconds)\n", TICKS_PER_EVOLUTION);
    printf("\n");

    // Simulate 3 evolutions
    for (int evo = 1; evo <= 3; evo++) {
        double phase_var = 5.5 + 0.1 * evo;  // Simulated phase variance
        clock_update_from_lattice(&clock, phase_var, evo);

        printf("Evolution %d: Phase Var = %.6f\n", evo, phase_var);

        // Generate samples for this evolution (5 seconds = 163,840 samples)
        // Print first 10 and last 10 samples
        for (uint64_t tick = 0; tick < TICKS_PER_EVOLUTION; tick++) {
            double sample = clock_tick(&clock);

            if (tick < 10 || tick >= TICKS_PER_EVOLUTION - 10) {
                double time_sec = clock.tick_count * DT;
                printf("  Tick %8llu | Time %10.6fs | Sample %+.6f\n",
                       (unsigned long long)clock.tick_count, time_sec, sample);
            } else if (tick == 10) {
                printf("  ... (%llu samples omitted) ...\n",
                       (unsigned long long)(TICKS_PER_EVOLUTION - 20));
            }
        }

        // Print metrics
        char metrics[512];
        clock_get_metrics(&clock, metrics, sizeof(metrics));
        printf("  Metrics: %s\n\n", metrics);
    }

    return 0;
}
#endif

// Python FFI interface function (for ctypes)
// Returns JSON string with current sample and metrics
void clock_get_sample_json(ClockState *clk, char *output, size_t max_len) {
    double sample = clock_tick(clk);
    snprintf(output, max_len,
        "{"
        "\"sample\":%.9f,"
        "\"tick\":%llu,"
        "\"evolution\":%llu,"
        "\"phase\":%.6f,"
        "\"frequency\":%.6f,"
        "\"phase_variance\":%.9f"
        "}",
        sample,
        (unsigned long long)clk->tick_count,
        (unsigned long long)clk->evolution_count,
        clk->phase,
        clk->frequency,
        clk->phase_variance
    );
}
