/*
 * HDGL IPFS Client Module - Checkpoint Storage & Retrieval
 * Uses IPFS HTTP API for checkpoint management
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <curl/curl.h>
#include <pthread.h>
#include <jansson.h>  // JSON library

#define IPFS_API_URL "http://localhost:5001/api/v0"
#define MAX_CID_LENGTH 64
#define MAX_CHECKPOINT_SIZE 4096

// Memory structure for curl response
struct MemoryStruct {
    char *memory;
    size_t size;
};

// Callback for curl write
static size_t write_memory_callback(void *contents, size_t size, size_t nmemb, void *userp) {
    size_t realsize = size * nmemb;
    struct MemoryStruct *mem = (struct MemoryStruct *)userp;

    char *ptr = realloc(mem->memory, mem->size + realsize + 1);
    if(!ptr) {
        printf("IPFS: Out of memory\n");
        return 0;
    }

    mem->memory = ptr;
    memcpy(&(mem->memory[mem->size]), contents, realsize);
    mem->size += realsize;
    mem->memory[mem->size] = 0;

    return realsize;
}

// Upload checkpoint to IPFS
char* ipfs_checkpoint_upload(const char* checkpoint_data, size_t data_len) {
    CURL *curl;
    CURLcode res;
    struct MemoryStruct chunk;
    static char cid[MAX_CID_LENGTH];

    chunk.memory = malloc(1);
    chunk.size = 0;

    curl = curl_easy_init();
    if (!curl) {
        printf("IPFS: Failed to initialize curl\n");
        free(chunk.memory);
        return NULL;
    }

    // Prepare multipart form data
    curl_mime *form = curl_mime_init(curl);
    curl_mimepart *field = curl_mime_addpart(form);

    curl_mime_name(field, "file");
    curl_mime_data(field, checkpoint_data, data_len);
    curl_mime_filename(field, "checkpoint.json");

    // Set curl options
    char url[256];
    snprintf(url, sizeof(url), "%s/add?pin=true", IPFS_API_URL);

    curl_easy_setopt(curl, CURLOPT_URL, url);
    curl_easy_setopt(curl, CURLOPT_MIMEPOST, form);
    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, write_memory_callback);
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&chunk);
    curl_easy_setopt(curl, CURLOPT_TIMEOUT, 10L);

    // Perform request
    res = curl_easy_perform(curl);

    if (res != CURLE_OK) {
        printf("IPFS: Upload failed: %s\n", curl_easy_strerror(res));
        curl_mime_free(form);
        curl_easy_cleanup(curl);
        free(chunk.memory);
        return NULL;
    }

    // Parse response to extract CID
    json_error_t error;
    json_t *root = json_loads(chunk.memory, 0, &error);

    if (root) {
        json_t *hash = json_object_get(root, "Hash");
        if (hash && json_is_string(hash)) {
            strncpy(cid, json_string_value(hash), MAX_CID_LENGTH - 1);
            cid[MAX_CID_LENGTH - 1] = '\0';

            printf("IPFS: Uploaded checkpoint: %s\n", cid);

            json_decref(root);
            curl_mime_free(form);
            curl_easy_cleanup(curl);
            free(chunk.memory);

            return cid;
        }
        json_decref(root);
    }

    curl_mime_free(form);
    curl_easy_cleanup(curl);
    free(chunk.memory);

    printf("IPFS: Failed to parse upload response\n");
    return NULL;
}

// Download checkpoint from IPFS
char* ipfs_checkpoint_download(const char* cid, size_t* out_len) {
    CURL *curl;
    CURLcode res;
    struct MemoryStruct chunk;

    chunk.memory = malloc(1);
    chunk.size = 0;

    curl = curl_easy_init();
    if (!curl) {
        printf("IPFS: Failed to initialize curl\n");
        free(chunk.memory);
        return NULL;
    }

    // Build URL
    char url[256];
    snprintf(url, sizeof(url), "%s/cat?arg=%s", IPFS_API_URL, cid);

    // Set curl options
    curl_easy_setopt(curl, CURLOPT_URL, url);
    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, write_memory_callback);
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&chunk);
    curl_easy_setopt(curl, CURLOPT_TIMEOUT, 30L);

    // Perform request
    res = curl_easy_perform(curl);

    if (res != CURLE_OK) {
        printf("IPFS: Download failed: %s\n", curl_easy_strerror(res));
        curl_easy_cleanup(curl);
        free(chunk.memory);
        return NULL;
    }

    curl_easy_cleanup(curl);

    printf("IPFS: Downloaded checkpoint from %s (%zu bytes)\n", cid, chunk.size);

    if (out_len) {
        *out_len = chunk.size;
    }

    return chunk.memory;
}

// Publish to IPFS pubsub topic
int ipfs_pubsub_publish(const char* topic, const char* message) {
    CURL *curl;
    CURLcode res;

    curl = curl_easy_init();
    if (!curl) {
        return -1;
    }

    // Build URL with topic and message
    char *encoded_topic = curl_easy_escape(curl, topic, 0);
    char *encoded_msg = curl_easy_escape(curl, message, 0);

    char url[512];
    snprintf(url, sizeof(url), "%s/pubsub/pub?arg=%s&arg=%s",
             IPFS_API_URL, encoded_topic, encoded_msg);

    curl_easy_setopt(curl, CURLOPT_URL, url);
    curl_easy_setopt(curl, CURLOPT_TIMEOUT, 5L);
    curl_easy_setopt(curl, CURLOPT_NOBODY, 1L);  // HEAD request

    res = curl_easy_perform(curl);

    curl_free(encoded_topic);
    curl_free(encoded_msg);
    curl_easy_cleanup(curl);

    if (res != CURLE_OK) {
        printf("IPFS: Pubsub publish failed: %s\n", curl_easy_strerror(res));
        return -1;
    }

    return 0;
}

// Get IPFS peer count
int ipfs_get_peer_count() {
    CURL *curl;
    CURLcode res;
    struct MemoryStruct chunk;
    int peer_count = 0;

    chunk.memory = malloc(1);
    chunk.size = 0;

    curl = curl_easy_init();
    if (!curl) {
        free(chunk.memory);
        return -1;
    }

    char url[256];
    snprintf(url, sizeof(url), "%s/swarm/peers", IPFS_API_URL);

    curl_easy_setopt(curl, CURLOPT_URL, url);
    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, write_memory_callback);
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&chunk);
    curl_easy_setopt(curl, CURLOPT_TIMEOUT, 5L);

    res = curl_easy_perform(curl);

    if (res == CURLE_OK) {
        json_error_t error;
        json_t *root = json_loads(chunk.memory, 0, &error);

        if (root) {
            json_t *peers = json_object_get(root, "Peers");
            if (peers && json_is_array(peers)) {
                peer_count = json_array_size(peers);
            }
            json_decref(root);
        }
    }

    curl_easy_cleanup(curl);
    free(chunk.memory);

    return peer_count;
}

// Initialize IPFS client
int ipfs_init() {
    curl_global_init(CURL_GLOBAL_DEFAULT);

    // Test connection
    int peer_count = ipfs_get_peer_count();
    if (peer_count >= 0) {
        printf("IPFS: Connected (%d peers)\n", peer_count);
        return 0;
    } else {
        printf("IPFS: Connection failed\n");
        return -1;
    }
}

// Cleanup IPFS client
void ipfs_cleanup() {
    curl_global_cleanup();
}
