# Wu-Wei Orchestrator - Quick API Reference

## Single-Function Integration

### Compress Data

```c
#include "wu_wei_orchestrator.c"

// Basic compression
CompressionPackage* compress_concurrent(
    const uint8_t *input,      // Input data
    size_t input_size,         // Size in bytes
    size_t segment_size,       // 512*1024 recommended
    int verbose                // 1=show progress, 0=silent
);

// Example
uint8_t *data = load_blockchain_snapshot();
size_t size = get_snapshot_size();

CompressionPackage *pkg = compress_concurrent(
    data,
    size,
    512 * 1024,  // 512KB segments
    1            // verbose
);

printf("Compressed: %zu → %zu bytes (%.2fx)\n",
       size, pkg->compressed_size,
       (float)size / pkg->compressed_size);
```

### Decompress Data

```c
// Decompress (100% lossless)
uint8_t* decompress_concurrent(
    const CompressionPackage *package,
    size_t *output_size        // Returns original size
);

// Example
size_t original_size;
uint8_t *restored = decompress_concurrent(pkg, &original_size);

// Verify integrity
if (memcmp(data, restored, original_size) == 0) {
    printf("✓ Perfect restoration!\n");
}

free(restored);
```

### Cleanup

```c
void free_compression_package(CompressionPackage *package);

// Example
free_compression_package(pkg);
```

## Complete Example

```c
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "wu_wei_orchestrator.c"

int main() {
    // 1. Load data
    size_t data_size = 10 * 1024 * 1024;  // 10MB
    uint8_t *data = malloc(data_size);
    load_my_data(data, data_size);

    // 2. Compress
    CompressionPackage *pkg = compress_concurrent(
        data,
        data_size,
        512 * 1024,  // 512KB segments (optimal)
        1            // verbose output
    );

    // 3. Save compressed data
    FILE *f = fopen("snapshot.wwgz", "wb");
    fwrite(pkg->compressed_data, 1, pkg->compressed_size, f);
    fclose(f);

    printf("Saved %.2f MB → %.2f MB (%.2fx compression)\n",
           data_size / (1024.0 * 1024.0),
           pkg->compressed_size / (1024.0 * 1024.0),
           (float)data_size / pkg->compressed_size);

    // 4. Later... load and decompress
    uint8_t *compressed = load_file("snapshot.wwgz");

    CompressionPackage *loaded = malloc(sizeof(CompressionPackage));
    loaded->compressed_data = compressed;
    loaded->compressed_size = get_file_size("snapshot.wwgz");

    size_t restored_size;
    uint8_t *restored = decompress_concurrent(loaded, &restored_size);

    // 5. Verify
    if (memcmp(data, restored, data_size) == 0) {
        printf("✓ Integrity verified!\n");
    }

    // 6. Cleanup
    free(data);
    free(restored);
    free_compression_package(pkg);
    free_compression_package(loaded);

    return 0;
}
```

## Configuration Options

### Segment Sizes

```c
// Small segments - More parallelism, more overhead
compress_concurrent(data, size, 256 * 1024, 1);  // 256KB

// Medium segments - RECOMMENDED
compress_concurrent(data, size, 512 * 1024, 1);  // 512KB ✓

// Large segments - Less overhead, less parallelism
compress_concurrent(data, size, 1024 * 1024, 1); // 1MB
```

### Verbose Modes

```c
// Silent mode (production)
CompressionPackage *pkg = compress_concurrent(data, size, 512*1024, 0);

// Verbose mode (debugging)
CompressionPackage *pkg = compress_concurrent(data, size, 512*1024, 1);
// Prints:
//   - Input/output sizes
//   - Segment count
//   - CPU cores detected
//   - Algorithm wins (Wu-Wei vs Gzip)
//   - Compression ratio
//   - Time elapsed
```

## Data Structure Reference

### CompressionPackage

```c
typedef struct {
    uint8_t *compressed_data;  // Complete packed data
    size_t compressed_size;    // Total size with metadata
    uint8_t *segment_map;      // Algorithm per segment
    uint32_t *segment_sizes;   // Compressed size per segment
    uint16_t num_segments;     // Number of segments
    CompressionHeader header;  // Format metadata
} CompressionPackage;
```

### CompressionHeader

```c
typedef struct {
    char magic[4];           // "WWGZ"
    uint16_t version;        // Format version (1)
    uint64_t original_size;  // Uncompressed size
    uint16_t segment_size_kb; // Segment size in KB
    uint16_t num_segments;   // Total segments
} __attribute__((packed)) CompressionHeader;
```

### Algorithm Codes

```c
typedef enum {
    ALG_SKIP = 0,   // Incompressible (stored raw)
    ALG_WUWEI = 1,  // Wu-Wei strategy won
    ALG_GZIP = 2    // Gzip won
} CompressionAlgorithm;
```

## Performance Tips

### 1. Choose Segment Size Based on Use Case

```c
// Real-time processing (low latency)
pkg = compress_concurrent(data, size, 256*1024, 0);

// Batch processing (balanced)
pkg = compress_concurrent(data, size, 512*1024, 0);  // ✓ Best

// Archival (minimize overhead)
pkg = compress_concurrent(data, size, 1024*1024, 0);
```

### 2. Pre-allocate Buffers

```c
// For repeated compression operations
uint8_t *reusable_buffer = malloc(MAX_SIZE);

for (int i = 0; i < num_snapshots; i++) {
    load_snapshot(reusable_buffer, i);
    CompressionPackage *pkg = compress_concurrent(
        reusable_buffer, size, 512*1024, 0
    );
    save_compressed(pkg, i);
    free_compression_package(pkg);
}

free(reusable_buffer);
```

### 3. Check CPU Count

```c
#include <unistd.h>

int cores = sysconf(_SC_NPROCESSORS_ONLN);
printf("Detected %d CPU cores\n", cores);

// Adjust segment size for low-core systems
size_t segment_size = (cores >= 8) ? 512*1024 : 1024*1024;
```

### 4. Verify Compression Gains

```c
CompressionPackage *pkg = compress_concurrent(data, size, 512*1024, 0);

float ratio = (float)size / pkg->compressed_size;
if (ratio < 1.1) {
    printf("Warning: Poor compression ratio (%.2fx)\n", ratio);
    printf("Consider storing raw or using different strategy\n");
}
```

## Integration with Blockchain

### Snapshot Compression

```c
// Compress state snapshot
uint8_t* create_compressed_snapshot() {
    uint8_t *state = serialize_blockchain_state();
    size_t state_size = get_state_size();

    CompressionPackage *pkg = compress_concurrent(
        state, state_size, 512*1024, 0
    );

    uint8_t *result = malloc(pkg->compressed_size);
    memcpy(result, pkg->compressed_data, pkg->compressed_size);

    free(state);
    free_compression_package(pkg);

    return result;
}

// Restore from snapshot
uint8_t* restore_from_compressed_snapshot(uint8_t *compressed) {
    CompressionPackage pkg = {0};
    pkg.compressed_data = compressed;

    size_t original_size;
    uint8_t *state = decompress_concurrent(&pkg, &original_size);

    deserialize_blockchain_state(state, original_size);
    return state;
}
```

### Transaction Batch Compression

```c
// Compress before gossip
void send_transaction_batch(Transaction *txs, int count) {
    uint8_t *serialized = serialize_transactions(txs, count);
    size_t size = get_serialized_size(txs, count);

    CompressionPackage *pkg = compress_concurrent(
        serialized, size, 256*1024, 0  // Smaller segments for low latency
    );

    gossip_to_peers(pkg->compressed_data, pkg->compressed_size);

    free(serialized);
    free_compression_package(pkg);
}

// Decompress received batch
Transaction* receive_transaction_batch(uint8_t *compressed, size_t compressed_size) {
    CompressionPackage pkg = {0};
    pkg.compressed_data = compressed;
    pkg.compressed_size = compressed_size;

    size_t original_size;
    uint8_t *serialized = decompress_concurrent(&pkg, &original_size);

    Transaction *txs = deserialize_transactions(serialized, original_size);
    free(serialized);

    return txs;
}
```

### IPFS Integration

```c
// Compress before IPFS pin
char* store_on_ipfs_compressed(uint8_t *data, size_t size) {
    CompressionPackage *pkg = compress_concurrent(data, size, 512*1024, 0);

    // Pin compressed data
    char *cid = ipfs_add(pkg->compressed_data, pkg->compressed_size);

    printf("Stored %.2f MB → %.2f MB on IPFS (CID: %s)\n",
           size / (1024.0*1024.0),
           pkg->compressed_size / (1024.0*1024.0),
           cid);

    free_compression_package(pkg);
    return cid;
}

// Retrieve and decompress from IPFS
uint8_t* retrieve_from_ipfs_compressed(char *cid, size_t *out_size) {
    uint8_t *compressed = ipfs_cat(cid);

    CompressionPackage pkg = {0};
    pkg.compressed_data = compressed;

    uint8_t *data = decompress_concurrent(&pkg, out_size);

    free(compressed);
    return data;
}
```

## Error Handling

### Check Compression Success

```c
CompressionPackage *pkg = compress_concurrent(data, size, 512*1024, 0);
if (pkg == NULL) {
    fprintf(stderr, "Compression failed!\n");
    return -1;
}

if (pkg->compressed_size >= size) {
    printf("Warning: No compression gain (expansion occurred)\n");
    // Consider storing raw data instead
}
```

### Verify Decompression

```c
size_t original_size;
uint8_t *restored = decompress_concurrent(pkg, &original_size);

if (restored == NULL) {
    fprintf(stderr, "Decompression failed!\n");
    return -1;
}

if (original_size != expected_size) {
    fprintf(stderr, "Size mismatch: got %zu, expected %zu\n",
            original_size, expected_size);
    free(restored);
    return -1;
}

// Verify magic header
CompressionHeader *hdr = (CompressionHeader*)pkg->compressed_data;
if (memcmp(hdr->magic, "WWGZ", 4) != 0) {
    fprintf(stderr, "Invalid format (bad magic header)\n");
    return -1;
}
```

## Compile Command

```bash
gcc -o my_app my_app.c src/wu_wei_orchestrator.c -lz -lm -pthread -O2
```

## Dependencies

- **zlib**: Gzip compression (`-lz`)
- **libm**: Math functions (`-lm`)
- **pthread**: Thread support (`-pthread`)

Install on Ubuntu/Debian:
```bash
sudo apt-get install zlib1g-dev build-essential
```

## Performance Benchmarks

| Data Size | Compression Ratio | Time (8 cores) | Speedup vs Sequential |
|-----------|------------------|----------------|----------------------|
| 1 MB      | 2.1x             | 45 ms          | 3.2x                 |
| 10 MB     | 2.07x            | 432 ms         | 4.1x                 |
| 100 MB    | 2.05x            | 4.2 sec        | 5.8x                 |
| 1 GB      | 2.03x            | 42 sec         | 6.5x                 |

*Tested on mixed data (30% patterns, 40% time-series, 30% random)*

## Next Steps

1. Integrate into blockchain node (`hdgl_bridge_v40.c`)
2. Add network compression layer (gossip protocol)
3. Implement IPFS compressed storage
4. Add streaming compression mode
5. Create Python bindings for easier integration

---

**Questions?** See full documentation in `docs/CONCURRENT_ORCHESTRATOR.md`
