#!/bin/bash
#
# Weekly Full Cleanup Script
#
# Performs comprehensive maintenance across all containers
# Runs every Sunday at 4 AM (after consolidation and archiving)
#
# BULLETPROOF MODE: Never exits with error, always completes

set +e  # Don't exit on error

LOG_FILE="/var/log/hdgl/weekly-cleanup.log"

# Logging with fallbacks
log() {
    local msg="[$(date '+%Y-%m-%d %H:%M:%S')] $1"

    # Try primary log
    if echo "$msg" >> "$LOG_FILE" 2>/dev/null; then
        :
    elif echo "$msg" | sudo tee -a "$LOG_FILE" >/dev/null 2>&1; then
        :
    else
        echo "$msg" >> "/tmp/hdgl-weekly-cleanup.log" 2>/dev/null || true
    fi

    # Always echo to stdout
    echo "$msg"
}

# Cleanup POA nodes
cleanup_poa() {
    local container="$1"
    log "Cleaning up $container..."

    # Remove old ancient data
    lxc-attach -n "$container" -- bash -c "
        find /root/.ethereum/geth/chaindata/ancient -type f -mtime +7 -delete 2>/dev/null || true
    " 2>/dev/null || true

    # Compact database
    lxc-attach -n "$container" -- bash -c "
        geth --datadir /root/.ethereum removedb compact 2>/dev/null || true
    " 2>/dev/null || true

    log "✅ Cleaned $container"
}

# Cleanup IPFS
cleanup_ipfs() {
    log "Cleaning up IPFS..."

    # Force garbage collection
    lxc-attach -n hdgl-bridge-lxc -- bash -c "
        ipfs repo gc 2>/dev/null || true
        ipfs repo verify 2>/dev/null || true
    " 2>/dev/null || true

    log "✅ Cleaned IPFS"
}

# Cleanup old logs
cleanup_logs() {
    log "Cleaning up old logs..."

    # Remove logs older than 7 days
    find /var/log/hdgl -name "*.log" -mtime +7 -delete 2>/dev/null || true

    # Compress logs older than 1 day
    find /var/log/hdgl -name "*.log" -mtime +1 -exec gzip {} \; 2>/dev/null || true

    # Remove compressed logs older than 30 days
    find /var/log/hdgl -name "*.log.gz" -mtime +30 -delete 2>/dev/null || true

    log "✅ Cleaned logs"
}

# Cleanup temporary files
cleanup_temp() {
    log "Cleaning up temporary files..."

    # Clean /tmp files
    find /tmp -name "hdgl-*" -mtime +7 -delete 2>/dev/null || true
    find /tmp -name "checkpoint-*" -mtime +7 -delete 2>/dev/null || true

    # Clean container temp dirs
    for container in hdgl-dns-lxc hdgl-bridge-lxc hdgl-web-lxc poa-rpc-lxc poa-miner1-lxc poa-miner2-lxc; do
        lxc-attach -n "$container" -- bash -c "
            find /tmp -type f -mtime +7 -delete 2>/dev/null || true
        " 2>/dev/null || true
    done

    log "✅ Cleaned temporary files"
}

# Cleanup old snapshots (beyond retention policy)
cleanup_old_snapshots() {
    log "Cleaning up old unprotected snapshots..."

    # Remove unprotected checkpoints older than 48 hours
    find /app/data/checkpoints/raw -name "*.dat" -mtime +2 -delete 2>/dev/null || true
    find /app/data/checkpoints/hourly -name "*.tar.zst" -mtime +2 -delete 2>/dev/null || true

    log "✅ Cleaned old snapshots"
}

# Optimize container filesystems
optimize_containers() {
    log "Optimizing container filesystems..."

    for container in hdgl-dns-lxc hdgl-bridge-lxc hdgl-web-lxc poa-rpc-lxc poa-miner1-lxc poa-miner2-lxc; do
        # Run fstrim to optimize storage
        lxc-attach -n "$container" -- bash -c "
            fstrim -av 2>/dev/null || true
        " 2>/dev/null || true
    done

    log "✅ Optimized containers"
}

# Report disk usage
report_usage() {
    log "=== Disk Usage Report ==="

    # Overall usage
    local total_size=$(du -sh /var/lib/lxc 2>/dev/null | cut -f1 || echo "unknown")
    log "Total LXC storage: $total_size"

    # Per container
    for container in hdgl-dns-lxc hdgl-bridge-lxc hdgl-web-lxc poa-rpc-lxc poa-miner1-lxc poa-miner2-lxc; do
        local size=$(du -sh "/var/lib/lxc/$container" 2>/dev/null | cut -f1 || echo "unknown")
        log "  $container: $size"
    done

    # Snapshots
    local snap_size=$(du -sh /app/data/snapshots 2>/dev/null | cut -f1 || echo "unknown")
    log "Permanent snapshots: $snap_size"

    # Archives
    local arch_size=$(du -sh /app/data/archives 2>/dev/null | cut -f1 || echo "unknown")
    log "Historical archives: $arch_size"

    log "========================="
}

# Main execution
main() {
    log "=== Weekly Full Cleanup Starting ==="

    # Create log directory if needed
    mkdir -p "$(dirname "$LOG_FILE")" 2>/dev/null || true

    # Run cleanup tasks
    cleanup_poa "poa-rpc-lxc"
    cleanup_poa "poa-miner1-lxc"
    cleanup_poa "poa-miner2-lxc"
    cleanup_ipfs
    cleanup_logs
    cleanup_temp
    cleanup_old_snapshots
    optimize_containers

    # Report final usage
    report_usage

    log "=== Weekly Full Cleanup Complete ==="
}

# Run main function
main

# Always exit success (bulletproof)
exit 0
