/**
 * Wu-Wei Post-Processing Test
 *
 * Tests if Wu-Wei can improve compression when applied AFTER Gzip:
 * 1. Gzip compress chunks individually
 * 2. Concatenate compressed chunks (headers, metadata repeated)
 * 3. Apply Wu-Wei to the concatenated compressed stream
 *
 * Hypothesis: Gzip headers and metadata have patterns that Wu-Wei can compress
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <math.h>
#include <zlib.h>

#define MB_10 (10 * 1024 * 1024)
#define CHUNK_SIZE (64 * 1024)  // 64KB chunks (more overhead)

// Timing helper
double get_time_ms() {
    struct timeval tv;
    gettimeofday(&tv, NULL);
    return (tv.tv_sec * 1000.0) + (tv.tv_usec / 1000.0);
}

// Wu-Wei analysis
float calculate_entropy(const uint8_t *data, size_t size) {
    if (size == 0) return 0.0f;
    uint32_t freq[256] = {0};
    for (size_t i = 0; i < size; i++) {
        freq[data[i]]++;
    }
    float entropy = 0.0f;
    for (int i = 0; i < 256; i++) {
        if (freq[i] > 0) {
            float p = (float)freq[i] / size;
            entropy -= p * log2f(p);
        }
    }
    return entropy;
}

float calculate_correlation(const uint8_t *data, size_t size) {
    if (size < 2) return 0.0f;
    float mean = 0.0f;
    for (size_t i = 0; i < size; i++) {
        mean += data[i];
    }
    mean /= size;
    float autocorr = 0.0f, variance = 0.0f;
    for (size_t i = 0; i < size - 1; i++) {
        float dev1 = data[i] - mean;
        float dev2 = data[i+1] - mean;
        autocorr += dev1 * dev2;
        variance += dev1 * dev1;
    }
    return (variance > 0) ? (autocorr / variance) : 0.0f;
}

// Wu-Wei compression primitives
size_t delta_encode(const uint8_t *input, size_t size, uint8_t *output) {
    if (size == 0) return 0;
    output[0] = input[0];
    for (size_t i = 1; i < size; i++) {
        output[i] = input[i] - input[i-1];
    }
    return size;
}

size_t delta_decode(const uint8_t *input, size_t size, uint8_t *output) {
    if (size == 0) return 0;
    output[0] = input[0];
    for (size_t i = 1; i < size; i++) {
        output[i] = input[i] + output[i-1];
    }
    return size;
}

size_t rle_encode(const uint8_t *input, size_t input_size, uint8_t *output) {
    if (input_size == 0) return 0;
    size_t out_pos = 0;
    size_t i = 0;
    while (i < input_size) {
        uint8_t value = input[i];
        size_t run_length = 1;
        while (i + run_length < input_size &&
               input[i + run_length] == value &&
               run_length < 255) {
            run_length++;
        }
        if (run_length >= 3) {
            output[out_pos++] = 0xFF;
            output[out_pos++] = (uint8_t)run_length;
            output[out_pos++] = value;
            i += run_length;
        } else {
            for (size_t j = 0; j < run_length; j++) {
                output[out_pos++] = value;
            }
            i += run_length;
        }
    }
    return out_pos;
}

size_t rle_decode(const uint8_t *input, size_t input_size, uint8_t *output) {
    size_t out_pos = 0;
    size_t i = 0;
    while (i < input_size) {
        if (input[i] == 0xFF && i + 2 < input_size) {
            uint8_t run_length = input[i + 1];
            uint8_t value = input[i + 2];
            for (int j = 0; j < run_length; j++) {
                output[out_pos++] = value;
            }
            i += 3;
        } else {
            output[out_pos++] = input[i++];
        }
    }
    return out_pos;
}

// Gzip compression
size_t compress_gzip(const uint8_t *src, size_t src_size, uint8_t *dst, size_t dst_capacity) {
    z_stream stream = {0};
    stream.next_in = (Bytef*)src;
    stream.avail_in = src_size;
    stream.next_out = dst;
    stream.avail_out = dst_capacity;

    deflateInit2(&stream, 9, Z_DEFLATED, 15 | 16, 8, Z_DEFAULT_STRATEGY);
    deflate(&stream, Z_FINISH);
    size_t compressed_size = stream.total_out;
    deflateEnd(&stream);

    return compressed_size;
}

size_t decompress_gzip(const uint8_t *src, size_t src_size, uint8_t *dst, size_t dst_capacity) {
    z_stream stream = {0};
    stream.next_in = (Bytef*)src;
    stream.avail_in = src_size;
    stream.next_out = dst;
    stream.avail_out = dst_capacity;

    inflateInit2(&stream, 15 | 16);
    inflate(&stream, Z_FINISH);
    size_t decompressed_size = stream.total_out;
    inflateEnd(&stream);

    return decompressed_size;
}

// Test data generator
void generate_mixed_data(uint8_t *data, size_t size) {
    size_t structured_size = size * 30 / 100;
    size_t correlated_size = size * 40 / 100;

    for (size_t i = 0; i < structured_size; i++) {
        data[i] = (i / 1024) % 256;
    }

    double *values = (double*)(data + structured_size);
    size_t count = correlated_size / sizeof(double);
    double base_value = 20.5;
    double drift = 0.0;
    for (size_t i = 0; i < count; i++) {
        drift += ((rand() % 100) - 50) * 0.001;
        double noise = ((rand() % 100) - 50) * 0.01;
        values[i] = base_value + drift + noise;
    }

    for (size_t i = structured_size + correlated_size; i < size; i++) {
        data[i] = rand() % 256;
    }
}

void print_header(const char *title) {
    printf("\n╔════════════════════════════════════════════════════════════════════╗\n");
    printf("║ %-66s ║\n", title);
    printf("╚════════════════════════════════════════════════════════════════════╝\n\n");
}

int main() {
    printf("\n");
    print_header("WU-WEI POST-PROCESSING TEST (10MB → Chunks → Gzip → Wu-Wei)");
    printf("Testing if Wu-Wei can compress concatenated Gzip chunks...\n\n");

    uint8_t *original_data = malloc(MB_10);
    uint8_t *temp_buffer = malloc(MB_10 * 2);

    srand(42);
    generate_mixed_data(original_data, MB_10);

    printf("Original data: %.2f MB\n", MB_10 / (1024.0 * 1024.0));
    printf("Chunk size: %.2f KB\n", CHUNK_SIZE / 1024.0);
    printf("Number of chunks: %d\n\n", MB_10 / CHUNK_SIZE);

    // ═══════════════════════════════════════════════════════════
    // Method 1: Compress entire file as one block
    // ═══════════════════════════════════════════════════════════

    print_header("Method 1: Single Gzip Compression (Baseline)");

    double start = get_time_ms();
    size_t single_gzip_size = compress_gzip(original_data, MB_10, temp_buffer, MB_10 * 2);
    double single_gzip_time = get_time_ms() - start;

    printf("Compressed: %.2f MB → %.2f MB\n",
           MB_10 / (1024.0 * 1024.0),
           single_gzip_size / (1024.0 * 1024.0));
    printf("Ratio: %.2fx\n", (float)MB_10 / single_gzip_size);
    printf("Time: %.2f ms\n", single_gzip_time);

    // ═══════════════════════════════════════════════════════════
    // Method 2: Compress chunks individually, concatenate
    // ═══════════════════════════════════════════════════════════

    print_header("Method 2: Chunked Gzip (256KB chunks)");

    uint8_t *chunked_compressed = malloc(MB_10 * 2);
    size_t total_compressed = 0;
    size_t num_chunks = MB_10 / CHUNK_SIZE;

    start = get_time_ms();
    for (size_t i = 0; i < num_chunks; i++) {
        size_t chunk_compressed = compress_gzip(
            original_data + (i * CHUNK_SIZE),
            CHUNK_SIZE,
            chunked_compressed + total_compressed,
            MB_10 * 2 - total_compressed
        );
        total_compressed += chunk_compressed;
    }
    double chunked_time = get_time_ms() - start;

    printf("Compressed: %.2f MB → %.2f MB\n",
           MB_10 / (1024.0 * 1024.0),
           total_compressed / (1024.0 * 1024.0));
    printf("Ratio: %.2fx\n", (float)MB_10 / total_compressed);
    printf("Time: %.2f ms\n", chunked_time);
    printf("Overhead vs single: %.2f%% (%.2f KB)\n",
           ((float)total_compressed / single_gzip_size - 1.0) * 100.0,
           (total_compressed - single_gzip_size) / 1024.0);

    // Analyze the concatenated compressed stream
    printf("\nChunked compressed stream analysis:\n");
    float chunk_entropy = calculate_entropy(chunked_compressed, total_compressed);
    float chunk_correlation = calculate_correlation(chunked_compressed, total_compressed);
    printf("  Entropy: %.2f bits/byte\n", chunk_entropy);
    printf("  Correlation: %.2f\n", chunk_correlation);

    // ═══════════════════════════════════════════════════════════
    // Method 3: Chunked Gzip → Wu-Wei Delta post-processing
    // ═══════════════════════════════════════════════════════════

    print_header("Method 3: Chunked Gzip → Wu-Wei Delta → Gzip");

    uint8_t *post_delta = malloc(MB_10 * 2);
    uint8_t *post_compressed = malloc(MB_10 * 2);

    start = get_time_ms();

    // Apply delta encoding to compressed stream
    size_t delta_size = delta_encode(chunked_compressed, total_compressed, post_delta);

    // Compress again with Gzip
    size_t post_gzip_size = compress_gzip(post_delta, delta_size, post_compressed, MB_10 * 2);

    double post_delta_time = get_time_ms() - start;

    printf("Chunked Gzip: %.2f MB\n", total_compressed / (1024.0 * 1024.0));
    printf("After Delta → Gzip: %.2f MB\n", post_gzip_size / (1024.0 * 1024.0));
    printf("Total ratio: %.2fx (from original %.2f MB)\n",
           (float)MB_10 / post_gzip_size,
           MB_10 / (1024.0 * 1024.0));
    printf("Post-processing time: %.2f ms\n", post_delta_time);
    printf("vs Single Gzip: %.2f%% %s\n",
           ((float)post_gzip_size / single_gzip_size - 1.0) * 100.0,
           post_gzip_size < single_gzip_size ? "✓ Better" : "Worse");
    printf("vs Chunked: %.2f%% improvement\n",
           (1.0 - (float)post_gzip_size / total_compressed) * 100.0);

    // ═══════════════════════════════════════════════════════════
    // Method 4: Chunked Gzip → Wu-Wei RLE post-processing
    // ═══════════════════════════════════════════════════════════

    print_header("Method 4: Chunked Gzip → Wu-Wei RLE → Gzip");

    uint8_t *post_rle = malloc(MB_10 * 2);

    start = get_time_ms();

    // Apply RLE to compressed stream
    size_t rle_size = rle_encode(chunked_compressed, total_compressed, post_rle);

    // Compress again with Gzip
    size_t post_rle_gzip_size = compress_gzip(post_rle, rle_size, post_compressed, MB_10 * 2);

    double post_rle_time = get_time_ms() - start;

    printf("Chunked Gzip: %.2f MB\n", total_compressed / (1024.0 * 1024.0));
    printf("After RLE → Gzip: %.2f MB\n", post_rle_gzip_size / (1024.0 * 1024.0));
    printf("Total ratio: %.2fx (from original %.2f MB)\n",
           (float)MB_10 / post_rle_gzip_size,
           MB_10 / (1024.0 * 1024.0));
    printf("Post-processing time: %.2f ms\n", post_rle_time);
    printf("vs Single Gzip: %.2f%% %s\n",
           ((float)post_rle_gzip_size / single_gzip_size - 1.0) * 100.0,
           post_rle_gzip_size < single_gzip_size ? "✓ Better" : "Worse");
    printf("vs Chunked: %.2f%% improvement\n",
           (1.0 - (float)post_rle_gzip_size / total_compressed) * 100.0);

    // ═══════════════════════════════════════════════════════════
    // Method 5: Chunked Gzip → Pure Gzip (no Wu-Wei)
    // ═══════════════════════════════════════════════════════════

    print_header("Method 5: Chunked Gzip → Pure Gzip (No Wu-Wei)");

    start = get_time_ms();
    size_t double_gzip_size = compress_gzip(chunked_compressed, total_compressed,
                                           post_compressed, MB_10 * 2);
    double double_gzip_time = get_time_ms() - start;

    printf("Chunked Gzip: %.2f MB\n", total_compressed / (1024.0 * 1024.0));
    printf("After Gzip again: %.2f MB\n", double_gzip_size / (1024.0 * 1024.0));
    printf("Total ratio: %.2fx (from original %.2f MB)\n",
           (float)MB_10 / double_gzip_size,
           MB_10 / (1024.0 * 1024.0));
    printf("Post-processing time: %.2f ms\n", double_gzip_time);
    printf("vs Single Gzip: %.2f%% %s\n",
           ((float)double_gzip_size / single_gzip_size - 1.0) * 100.0,
           double_gzip_size < single_gzip_size ? "✓ Better" : "Worse");
    printf("vs Chunked: %.2f%% improvement\n",
           (1.0 - (float)double_gzip_size / total_compressed) * 100.0);

    // ═══════════════════════════════════════════════════════════
    // Summary & Comparison
    // ═══════════════════════════════════════════════════════════

    print_header("SUMMARY: Post-Processing Comparison");

    printf("%-35s %10s %8s %10s\n", "Method", "Size", "Ratio", "vs Single");
    printf("─────────────────────────────────────────────────────────────────\n");
    printf("%-35s %8.2f MB   %6.2fx   %9.2f%%\n",
           "1. Single Gzip (baseline)",
           single_gzip_size / (1024.0 * 1024.0),
           (float)MB_10 / single_gzip_size,
           0.0);
    printf("%-35s %8.2f MB   %6.2fx   %9.2f%%\n",
           "2. Chunked Gzip",
           total_compressed / (1024.0 * 1024.0),
           (float)MB_10 / total_compressed,
           ((float)total_compressed / single_gzip_size - 1.0) * 100.0);
    printf("%-35s %8.2f MB   %6.2fx   %9.2f%% %s\n",
           "3. Chunked → Delta → Gzip",
           post_gzip_size / (1024.0 * 1024.0),
           (float)MB_10 / post_gzip_size,
           ((float)post_gzip_size / single_gzip_size - 1.0) * 100.0,
           post_gzip_size < single_gzip_size ? "✓" : "");
    printf("%-35s %8.2f MB   %6.2fx   %9.2f%% %s\n",
           "4. Chunked → RLE → Gzip",
           post_rle_gzip_size / (1024.0 * 1024.0),
           (float)MB_10 / post_rle_gzip_size,
           ((float)post_rle_gzip_size / single_gzip_size - 1.0) * 100.0,
           post_rle_gzip_size < single_gzip_size ? "✓" : "");
    printf("%-35s %8.2f MB   %6.2fx   %9.2f%% %s\n",
           "5. Chunked → Pure Gzip",
           double_gzip_size / (1024.0 * 1024.0),
           (float)MB_10 / double_gzip_size,
           ((float)double_gzip_size / single_gzip_size - 1.0) * 100.0,
           double_gzip_size < single_gzip_size ? "✓" : "");

    printf("\n");
    printf("Key Insights:\n");
    printf("  • Chunking adds overhead (repeated headers/metadata)\n");
    printf("  • Post-processing can recover some chunking overhead\n");
    if (post_gzip_size < total_compressed) {
        printf("  • Wu-Wei Delta helps: %.1f%% improvement on chunks\n",
               (1.0 - (float)post_gzip_size / total_compressed) * 100.0);
    }
    if (post_rle_gzip_size < total_compressed) {
        printf("  • Wu-Wei RLE helps: %.1f%% improvement on chunks\n",
               (1.0 - (float)post_rle_gzip_size / total_compressed) * 100.0);
    }
    if (double_gzip_size < total_compressed) {
        printf("  • Double-Gzip works: %.1f%% improvement on chunks\n",
               (1.0 - (float)double_gzip_size / total_compressed) * 100.0);
    }
    printf("  • Best for parallel processing: Chunk + post-process\n");

    printf("\n");

    free(original_data);
    free(temp_buffer);
    free(chunked_compressed);
    free(post_delta);
    free(post_rle);
    free(post_compressed);

    return 0;
}
