/*
 * wu_wei_compress.h
 * Wu-Wei Adaptive Compression Engine
 * "Let the data guide its own optimization"
 */

#ifndef WU_WEI_COMPRESS_H
#define WU_WEI_COMPRESS_H

#include <stdint.h>
#include <stddef.h>

#ifdef __cplusplus
extern "C" {
#endif

/* ═══════════════════════════════════════════════════════
 * PUBLIC API
 * ═══════════════════════════════════════════════════════ */

/**
 * Compress data using Wu-Wei adaptive strategy
 *
 * @param input Input data buffer
 * @param input_size Size of input data
 * @param output_size Pointer to store output size
 * @param current_phase Current compression phase (0-4)
 * @return Compressed data (caller must free), or NULL on error
 *
 * The function automatically:
 * - Analyzes data characteristics (entropy, correlation, repetition)
 * - Selects optimal compression strategy
 * - Adapts phase based on variance
 * - Returns compressed data with header
 */
uint8_t* wuwei_compress(const uint8_t *input, size_t input_size,
                        size_t *output_size, int current_phase);

/**
 * Decompress Wu-Wei compressed data
 *
 * @param input Compressed data buffer (with Wu-Wei header)
 * @param input_size Size of compressed data
 * @param output_size Pointer to store decompressed size
 * @return Decompressed data (caller must free), or NULL on error
 *
 * The function automatically:
 * - Verifies magic number and checksum
 * - Applies reverse decompression passes
 * - Validates output size matches original
 */
uint8_t* wuwei_decompress(const uint8_t *input, size_t input_size,
                          size_t *output_size);

/**
 * Print compression statistics
 *
 * @param input Original input data
 * @param input_size Size of input
 * @param output Compressed output (with header)
 * @param output_size Size of compressed output
 */
void wuwei_print_stats(const uint8_t *input, size_t input_size,
                       const uint8_t *output, size_t output_size);

#ifdef __cplusplus
}
#endif

#endif /* WU_WEI_COMPRESS_H */
