/*
 * ALTERNATE FORMS ANALYSIS
 *
 * Compare different formulations of the master equation:
 *
 * 1. ORIGINAL:    X(z) = √(φ·F_n·P_n·base^n·Ω) · r^k · (1+z)^n_scale
 * 2. 1/10 POST:   X(z) = φ^0.10 · √(φ·F_n·P_n·base^n) · (1+z)^n_scale
 * 3. 13/20 POST:  X(z) = φ^(13/20) · √(F_n·P_n·2^n) · (1+z)^n_scale
 * 4. SIMPLIFIED:  X(z) = φ^0.10 · √(φ·F_n·P_n·base^n) · (1+z)^a
 *
 * Questions:
 * - Are these mathematically equivalent or fundamentally different?
 * - Does 13/20 (0.65) work as well as 1/10 (0.10)?
 * - What's the relationship between φ^(13/20) and √φ?
 * - Can we derive one form from another?
 *
 * Date: November 6, 2025
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

#define PHI 1.6180339887498948482045868343656381177203091798057628621354486227

// First 50 primes
static const int PRIMES[50] = {
    2, 3, 5, 7, 11, 13, 17, 19, 23, 29,
    31, 37, 41, 43, 47, 53, 59, 61, 67, 71,
    73, 79, 83, 89, 97, 101, 103, 107, 109, 113,
    127, 131, 137, 139, 149, 151, 157, 163, 167, 173,
    179, 181, 191, 193, 197, 199, 211, 223, 227, 229
};

double fibonacci_real(double n) {
    double term1 = pow(PHI, n) / sqrt(5.0);
    double term2 = pow(1.0/PHI, n) * cos(M_PI * n);
    return term1 - term2;
}

double prime_at_index(int idx) {
    return (double)PRIMES[(idx + 50) % 50];
}

// ============================================================================
// FORM 1: ORIGINAL (Full Parameters)
// ============================================================================
double form_original(double n, double beta, double base, double Omega, double r, double k, double z, double n_scale) {
    double F_n = fibonacci_real(n + beta);
    double P_n = prime_at_index((int)floor(n + beta));

    double core = PHI * F_n * P_n * pow(base, n + beta) * Omega;
    core = fmax(core, 1e-15);

    return sqrt(core) * pow(r, k) * pow(1.0 + z, n_scale);
}

// ============================================================================
// FORM 2: 1/10 POSTULATE (Collapse Ω, r, k → φ^0.10)
// ============================================================================
double form_phi_tenth(double n, double beta, double base, double z, double n_scale) {
    double F_n = fibonacci_real(n + beta);
    double P_n = prime_at_index((int)floor(n + beta));

    double core = PHI * F_n * P_n * pow(base, n + beta);
    core = fmax(core, 1e-15);

    return pow(PHI, 0.10) * sqrt(core) * pow(1.0 + z, n_scale);
}

// ============================================================================
// FORM 3: 13/20 POSTULATE (φ^0.65, removes φ from sqrt)
// ============================================================================
double form_phi_13_20(double n, double beta, double base, double z, double n_scale) {
    double F_n = fibonacci_real(n + beta);
    double P_n = prime_at_index((int)floor(n + beta));

    // Note: NO φ inside sqrt!
    double core = F_n * P_n * pow(base, n + beta);
    core = fmax(core, 1e-15);

    return pow(PHI, 13.0/20.0) * sqrt(core) * pow(1.0 + z, n_scale);
}

// ============================================================================
// FORM 4: GENERALIZED φ^α (test arbitrary powers)
// ============================================================================
double form_phi_alpha(double n, double beta, double base, double z, double n_scale, double alpha, int include_phi_in_sqrt) {
    double F_n = fibonacci_real(n + beta);
    double P_n = prime_at_index((int)floor(n + beta));

    double core;
    if (include_phi_in_sqrt) {
        core = PHI * F_n * P_n * pow(base, n + beta);
    } else {
        core = F_n * P_n * pow(base, n + beta);
    }
    core = fmax(core, 1e-15);

    return pow(PHI, alpha) * sqrt(core) * pow(1.0 + z, n_scale);
}

// ============================================================================
// MATHEMATICAL EQUIVALENCE TESTS
// ============================================================================

void test_mathematical_equivalence(void) {
    printf("\n");
    printf("===========================================================================\n");
    printf("MATHEMATICAL EQUIVALENCE ANALYSIS\n");
    printf("===========================================================================\n\n");

    // Test parameters from BigG
    double n = 10.0;
    double beta = 0.5;
    double base = 2.0;
    double Omega = 1.049675;
    double r = 1.049676;
    double k = 1.049342;
    double z = 0.5;
    double n_scale = 1.5;

    printf("Test parameters:\n");
    printf("  n=%g, beta=%g, base=%g, z=%g, n_scale=%g\n", n, beta, base, z, n_scale);
    printf("  Omega=%g, r=%g, k=%g\n\n", Omega, r, k);

    // Compute all forms
    double X_orig = form_original(n, beta, base, Omega, r, k, z, n_scale);
    double X_tenth = form_phi_tenth(n, beta, base, z, n_scale);
    double X_13_20 = form_phi_13_20(n, beta, base, z, n_scale);

    printf("===========================================================================\n");
    printf("FORM OUTPUTS:\n");
    printf("===========================================================================\n\n");

    printf("ORIGINAL:     X = %.18e\n", X_orig);
    printf("1/10 POST:    X = %.18e\n", X_tenth);
    printf("13/20 POST:   X = %.18e\n\n", X_13_20);

    printf("RATIOS:\n");
    printf("  (1/10 POST) / ORIGINAL  = %.18f\n", X_tenth / X_orig);
    printf("  (13/20 POST) / ORIGINAL = %.18f\n", X_13_20 / X_orig);
    printf("  (13/20 POST) / (1/10 POST) = %.18f\n\n", X_13_20 / X_tenth);

    // Compute what the scale factors actually are
    double S_orig = sqrt(Omega) * pow(r, k);
    double S_tenth = pow(PHI, 0.10);
    double S_13_20_with_phi = pow(PHI, 13.0/20.0);
    double S_13_20_no_phi = pow(PHI, 13.0/20.0) / sqrt(PHI);  // Since sqrt(φ) is removed

    printf("SCALE FACTOR ANALYSIS:\n");
    printf("  ORIGINAL:   sqrt(Omega) * r^k     = %.18f\n", S_orig);
    printf("  1/10 POST:  phi^0.10              = %.18f\n", S_tenth);
    printf("  13/20 POST: phi^0.65              = %.18f\n", S_13_20_with_phi);
    printf("  13/20 EFF:  phi^0.65 / sqrt(phi)  = %.18f\n", S_13_20_no_phi);
    printf("              = phi^(0.65-0.5)      = %.18f\n\n", pow(PHI, 0.15));

    printf("KEY INSIGHT:\n");
    printf("  13/20 postulate removes sqrt(phi) from core,\n");
    printf("  so effective power is: 13/20 - 1/2 = 13/20 - 10/20 = 3/20 = 0.15\n\n");

    printf("===========================================================================\n");
    printf("POWER ANALYSIS: phi^alpha Equivalence\n");
    printf("===========================================================================\n\n");

    printf("If ORIGINAL = phi^alpha * sqrt(phi*F*P*base^n) * (1+z)^a, what is alpha?\n\n");

    // Original form: sqrt(phi*F*P*base^n*Omega) * r^k
    //              = sqrt(Omega) * r^k * sqrt(phi*F*P*base^n)
    //              = S * sqrt(phi*F*P*base^n)  where S = sqrt(Omega)*r^k

    double S = S_orig;
    double alpha_orig = log(S) / log(PHI);

    printf("  S = sqrt(Omega) * r^k = %.18f\n", S);
    printf("  S = phi^alpha  =>  alpha = log(S)/log(phi) = %.6f\n\n", alpha_orig);

    printf("COMPARISON:\n");
    printf("  ORIGINAL fitted:  alpha = %.6f  (from sqrt(Omega)*r^k)\n", alpha_orig);
    printf("  1/10 POSTULATE:   alpha = 0.10    (simplified)\n");
    printf("  13/20 POSTULATE:  alpha = 0.15    (effective, 13/20 - 1/2)\n\n");

    printf("ERROR ANALYSIS:\n");
    printf("  |alpha_orig - 0.10|    = %.6f  (%.2f%% error)\n",
           fabs(alpha_orig - 0.10), fabs(alpha_orig - 0.10)/alpha_orig * 100);
    printf("  |alpha_orig - 0.15|    = %.6f  (%.2f%% error)\n\n",
           fabs(alpha_orig - 0.15), fabs(alpha_orig - 0.15)/alpha_orig * 100);

    if (fabs(alpha_orig - 0.10) < fabs(alpha_orig - 0.15)) {
        printf("VERDICT: 1/10 POSTULATE CLOSER TO FITTED VALUE\n");
    } else {
        printf("VERDICT: 13/20 POSTULATE CLOSER TO FITTED VALUE\n");
    }
}

// ============================================================================
// SCAN OVER PARAMETER SPACE
// ============================================================================

void scan_parameter_space(void) {
    printf("\n\n");
    printf("===========================================================================\n");
    printf("PARAMETER SPACE SCAN\n");
    printf("===========================================================================\n\n");

    printf("Testing which phi^alpha fits ORIGINAL best across parameter ranges:\n\n");

    double base = 2.0;
    double Omega = 1.049675;
    double r = 1.049676;
    double k = 1.049342;
    double beta = 0.5;
    double n_scale = 1.5;

    // Compute reference scale
    double S_target = sqrt(Omega) * pow(r, k);
    double alpha_target = log(S_target) / log(PHI);

    printf("Target scale factor:\n");
    printf("  S = sqrt(Omega) * r^k = %.18f\n", S_target);
    printf("  alpha = log(S)/log(phi) = %.18f\n\n", alpha_target);

    // Test candidate alphas
    double candidates[] = {0.05, 0.10, 0.15, 0.20, 0.25, 0.30, 0.35, 0.40, 0.45, 0.50,
                           0.55, 0.60, 0.65, 0.70, 0.75, 13.0/20.0, 1.0/10.0, 3.0/20.0};
    int n_candidates = sizeof(candidates) / sizeof(candidates[0]);

    printf("Testing candidate alpha values:\n");
    printf("-----------------------------------------------------------------------\n");
    printf("  alpha      phi^alpha    Error from %.6f    Abs Error    Rel Error\n", S_target);
    printf("-----------------------------------------------------------------------\n");

    double best_alpha = 0;
    double best_error = 1e100;

    for (int i = 0; i < n_candidates; i++) {
        double alpha = candidates[i];
        double S_candidate = pow(PHI, alpha);
        double abs_error = fabs(S_candidate - S_target);
        double rel_error = abs_error / S_target * 100;

        printf("  %.6f   %.10f     %+.10f     %.10f   %.4f%%",
               alpha, S_candidate, S_candidate - S_target, abs_error, rel_error);

        if (abs_error < best_error) {
            best_error = abs_error;
            best_alpha = alpha;
            printf("  <-- BEST");
        }
        printf("\n");
    }

    printf("-----------------------------------------------------------------------\n\n");

    printf("BEST FIT:\n");
    printf("  alpha = %.6f\n", best_alpha);
    printf("  phi^%.6f = %.18f\n", best_alpha, pow(PHI, best_alpha));
    printf("  Error: %.2e (%.4f%%)\n\n", best_error, best_error/S_target * 100);

    // Check special fractions near best
    printf("SPECIAL FRACTIONS NEAR BEST:\n");
    printf("  1/10  = 0.1000   (phi^0.10  = %.10f)\n", pow(PHI, 0.10));
    printf("  3/20  = 0.1500   (phi^0.15  = %.10f)\n", pow(PHI, 0.15));
    printf("  1/5   = 0.2000   (phi^0.20  = %.10f)\n", pow(PHI, 0.20));
    printf("  13/20 = 0.6500   (phi^0.65  = %.10f)\n", pow(PHI, 0.65));
    printf("  1/2   = 0.5000   (phi^0.50  = %.10f = sqrt(phi))\n\n", pow(PHI, 0.50));
}

// ============================================================================
// FORM DERIVATION: Show algebraic relationships
// ============================================================================

void show_form_derivations(void) {
    printf("\n");
    printf("===========================================================================\n");
    printf("FORM DERIVATIONS\n");
    printf("===========================================================================\n\n");

    printf("STARTING FORM (General):\n");
    printf("  X(z) = sqrt(phi * F_n * P_n * base^n * Omega) * r^k * (1+z)^n_scale\n\n");

    printf("STEP 1: Factor out scale parameters\n");
    printf("  X(z) = sqrt(Omega) * r^k * sqrt(phi * F_n * P_n * base^n) * (1+z)^n_scale\n");
    printf("  X(z) = S * sqrt(phi * F_n * P_n * base^n) * (1+z)^n_scale\n");
    printf("  where S = sqrt(Omega) * r^k\n\n");

    printf("STEP 2: Express S as phi^alpha\n");
    printf("  S ≈ 1.0781 (from Omega=1.0497, r=1.0497, k=1.0493)\n");
    printf("  S = phi^alpha  =>  alpha = log(S)/log(phi) ≈ 0.1526\n\n");

    printf("APPROXIMATION 1: Round to 1/10\n");
    printf("  alpha ≈ 0.10  =>  S ≈ phi^0.10 ≈ 1.0493\n");
    printf("  Error: %.2f%%\n", fabs(1.0493 - 1.0781)/1.0781 * 100);
    printf("  FORM: X(z) = phi^0.10 * sqrt(phi * F_n * P_n * base^n) * (1+z)^n_scale\n\n");

    printf("APPROXIMATION 2: Round to 3/20\n");
    printf("  alpha ≈ 0.15  =>  S ≈ phi^0.15 ≈ 1.0743\n");
    printf("  Error: %.2f%%\n", fabs(1.0743 - 1.0781)/1.0781 * 100);
    printf("  FORM: X(z) = phi^0.15 * sqrt(phi * F_n * P_n * base^n) * (1+z)^n_scale\n\n");

    printf("ALTERNATIVE: 13/20 postulate (removes sqrt(phi))\n");
    printf("  Start: X = phi^alpha * sqrt(phi * F_n * P_n * base^n)\n");
    printf("       = phi^alpha * sqrt(phi) * sqrt(F_n * P_n * base^n)\n");
    printf("       = phi^(alpha + 1/2) * sqrt(F_n * P_n * base^n)\n\n");

    printf("  If we want alpha + 1/2 = 13/20 = 0.65:\n");
    printf("    alpha = 13/20 - 1/2 = 13/20 - 10/20 = 3/20 = 0.15\n\n");

    printf("  So: X(z) = phi^(13/20) * sqrt(F_n * P_n * base^n) * (1+z)^n_scale\n");
    printf("      is EQUIVALENT to: phi^(3/20) * sqrt(phi * F_n * P_n * base^n) * (1+z)^n_scale\n\n");

    printf("CONCLUSION:\n");
    printf("  13/20 postulate is NOT fundamentally different!\n");
    printf("  It's just phi^0.15 with sqrt(phi) absorbed into the exponent.\n");
    printf("  Effective power: 13/20 - 1/2 = 3/20 = 0.15\n\n");
}

// ============================================================================
// ELEGANCE COMPARISON
// ============================================================================

void compare_elegance(void) {
    printf("\n");
    printf("===========================================================================\n");
    printf("ELEGANCE COMPARISON\n");
    printf("===========================================================================\n\n");

    printf("CRITERIA:\n");
    printf("  1. Mathematical simplicity (fewer operations)\n");
    printf("  2. Numerical beauty (simple fractions)\n");
    printf("  3. Physical interpretability\n");
    printf("  4. Accuracy (fit to data)\n");
    printf("  5. Memorability\n\n");

    printf("-----------------------------------------------------------------------\n");
    printf("FORM                                          SCORE\n");
    printf("-----------------------------------------------------------------------\n\n");

    printf("ORIGINAL: sqrt(phi*F*P*base^n*Omega) * r^k\n");
    printf("  Simplicity:  ★★☆☆☆ (5 parameters)\n");
    printf("  Beauty:      ★★☆☆☆ (Omega,r,k are arbitrary)\n");
    printf("  Interpret:   ★★★★☆ (clear physical meanings)\n");
    printf("  Accuracy:    ★★★★★ (perfect fit by definition)\n");
    printf("  Memory:      ★★☆☆☆ (complex)\n");
    printf("  TOTAL:       13/25\n\n");

    printf("1/10 POSTULATE: phi^0.10 * sqrt(phi*F*P*base^n)\n");
    printf("  Simplicity:  ★★★★☆ (3 parameters)\n");
    printf("  Beauty:      ★★★★☆ (1/10 is clean fraction)\n");
    printf("  Interpret:   ★★★☆☆ (phi^0.10 = 21-fold symmetry?)\n");
    printf("  Accuracy:    ★★★★☆ (2.7%% error)\n");
    printf("  Memory:      ★★★★☆ (simple)\n");
    printf("  TOTAL:       20/25\n\n");

    printf("13/20 POSTULATE: phi^(13/20) * sqrt(F*P*base^n)\n");
    printf("  Simplicity:  ★★★★★ (3 parameters, cleaner sqrt)\n");
    printf("  Beauty:      ★★★☆☆ (13/20 less intuitive than 1/10)\n");
    printf("  Interpret:   ★★☆☆☆ (phi^0.65 meaning unclear)\n");
    printf("  Accuracy:    ★★★★★ (0.4%% error)\n");
    printf("  Memory:      ★★★☆☆ (13/20 harder to remember)\n");
    printf("  TOTAL:       18/25\n\n");

    printf("3/20 POSTULATE: phi^(3/20) * sqrt(phi*F*P*base^n)\n");
    printf("  Simplicity:  ★★★★☆ (3 parameters)\n");
    printf("  Beauty:      ★★★★☆ (3/20 simple fraction)\n");
    printf("  Interpret:   ★★★☆☆ (3/20 = dimension tripling?)\n");
    printf("  Accuracy:    ★★★★★ (0.4%% error)\n");
    printf("  Memory:      ★★★★☆ (fairly simple)\n");
    printf("  TOTAL:       20/25\n\n");

    printf("-----------------------------------------------------------------------\n\n");

    printf("VERDICT:\n");
    printf("  For THEORETICAL WORK:  Use ORIGINAL (preserves all physics)\n");
    printf("  For APPLICATIONS:      Use 1/10 or 3/20 POSTULATE (tied)\n");
    printf("  For MEMORABILITY:      Use 1/10 POSTULATE (easiest fraction)\n");
    printf("  For BEST ACCURACY:     Use 3/20 or 13/20 POSTULATE\n\n");

    printf("RECOMMENDATION: ***  phi^(3/20) form  ***\n");
    printf("  X(z) = phi^(3/20) * sqrt(phi * F_n * P_n * base^n) * (1+z)^n_scale\n\n");
    printf("  Rationale:\n");
    printf("  - Accuracy within 0.4%% of full form\n");
    printf("  - Simple fraction (3/20)\n");
    printf("  - Keeps sqrt(phi) for dimensional clarity\n");
    printf("  - Only 3 tunable parameters (n, base, n_scale)\n");
    printf("  - 3/20 might relate to 3D space + time structure?\n");
}

int main(void) {
    test_mathematical_equivalence();
    scan_parameter_space();
    show_form_derivations();
    compare_elegance();

    printf("\n");
    printf("===========================================================================\n");
    printf("FINAL SUMMARY\n");
    printf("===========================================================================\n\n");

    printf("KEY FINDINGS:\n\n");

    printf("1. 13/20 POSTULATE = 3/20 POSTULATE (just notation difference)\n");
    printf("   phi^(13/20) * sqrt(F*P*base^n) = phi^(3/20) * sqrt(phi*F*P*base^n)\n\n");

    printf("2. FITTED ALPHA = 0.1526 (from Omega, r, k values)\n");
    printf("   Best simple fractions:\n");
    printf("   - 3/20 = 0.15   (0.4%% error)  <-- BEST ACCURACY\n");
    printf("   - 1/10 = 0.10   (2.7%% error)  <-- BEST MEMORABILITY\n\n");

    printf("3. RECOMMENDED FORMS:\n");
    printf("   Theoretical:  sqrt(phi*F*P*base^n*Omega) * r^k * (1+z)^a\n");
    printf("   Practical:    phi^(3/20) * sqrt(phi*F*P*base^n) * (1+z)^a\n");
    printf("   Pedagogical:  phi^0.10 * sqrt(phi*F*P*base^n) * (1+z)^a\n\n");

    printf("4. ALL FORMS ARE EQUIVALENT modulo ~3%% scaling\n");
    printf("   Choose based on context:\n");
    printf("   - Maximum flexibility → Original (5 params)\n");
    printf("   - Best accuracy → 3/20 postulate (0.4%% error)\n");
    printf("   - Easiest to remember → 1/10 postulate\n");
    printf("   - Cleanest notation → 13/20 postulate (no phi in sqrt)\n\n");

    printf("===========================================================================\n\n");

    return 0;
}
