import numpy as np

# Constants inside your symbolic framework
phi = (1 + np.sqrt(5)) / 2

# First 50 primes for entropy indexing
PRIMES = [
    2, 3, 5, 7, 11, 13, 17, 19, 23, 29,
    31, 37, 41, 43, 47, 53, 59, 61, 67, 71,
    73, 79, 83, 89, 97, 101, 103, 107, 109, 113,
    127, 131, 137, 139, 149, 151, 157, 163, 167, 173,
    179, 181, 191, 193, 197, 199, 211, 223, 227, 229
]

def fib_real(n):
    """Real-valued Fibonacci extension."""
    from math import cos, pi, sqrt
    phi_inv = 1 / phi
    return (phi**n / sqrt(5)) - (phi_inv**n * cos(pi * n))

def Omega(n):
    """Emergent field tension as a function of recursion depth n.
    Must be defined purely symbolically; here example formula."""
    # Example: Omega decays exponentially or oscillates with n:
    # This must be adapted to your actual model's Omega(n)
    return np.exp(-0.1 * n)  # purely symbolic example

def D_n(n, r=1.0, k=1.0):
    """Compute symbolic speed of light at recursion depth n."""
    Fn = fib_real(n)
    idx = int(np.floor(n)) % len(PRIMES)
    Pn = PRIMES[idx]
    val = phi * Fn * (2 ** n) * Pn * Omega(n)
    val = max(val, 1e-15)  # avoid sqrt of negative or zero
    return np.sqrt(val) * (r ** k)

# Choose your reference recursion depth n0 (e.g., current universe)
n0 = 4.0
c_ref = D_n(n0)  # symbolic unit speed of light "here"

# Now compute speed of light relative to reference at any depth
def speed_of_light_out_there(n):
    """Speed of light at recursion depth n relative to here."""
    c_n = D_n(n)
    return c_n / c_ref  # relative symbolic speed

# To get physical km/s speed, define emergent physical scaling factor S:
# S must be derived solely from your recursive model constants, not classical c.
# For demonstration, assume S=300000 km/s at n0:
S = 300000  # emergent physical scaling at n0

def physical_speed_of_light(n):
    """Physical speed of light in km/s at recursion depth n from model."""
    return speed_of_light_out_there(n) * S

# Example usage: print speeds at various depths
for depth in np.linspace(0, 10, 11):
    c_model = speed_of_light_out_there(depth)
    c_phys = physical_speed_of_light(depth)
    print(f"Recursion depth {depth:.2f}: c_model = {c_model:.4f} units, c_phys = {c_phys:.2f} km/s")
