#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import sys, time, json, math, struct
import numpy as np
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileShader, compileProgram
import unicodedata

# -------------------------------
# CONFIG
# -------------------------------
LATTICE_WIDTH = 4096
LATTICE_HEIGHT_TOTAL = 4096
MAX_TILE_HEIGHT = 1024
PHI = 1.6180339887
PHI_POWERS = np.array([1.0 / pow(PHI, 7*(i+1)) for i in range(72)], dtype=np.float32)
THRESHOLD = math.sqrt(PHI)
MAX_SLOTS = 16_777_216
RECURSION_DEPTH = 3  # adjustable

EXPORT_JSON = "hdgl_vectors.json"
EXPORT_BINARY = "hdgl_lattice.hdgl"

# -------------------------------
# PROCEDURAL SLOT / CHAR
# -------------------------------
def hdgl_char(idx):
    h = (idx * 2654435761) % 0x110000
    c = chr(h) if is_valid_char(chr(h)) else chr((h+1)%0x110000)
    return c

def is_valid_char(c):
    try:
        name = unicodedata.name(c)
        cat = unicodedata.category(c)
        if any(bad in name for bad in ['CONTROL','PRIVATE USE','SURROGATE','UNASSIGNED','TAG']):
            return False
        if cat in ['Mn','Mc','Me','Cc','Cf','Cs','Cn','Co','Zs']:
            return False
        return True
    except ValueError:
        return False

# -------------------------------
# RECURSIVE UNFOLDING
# -------------------------------
def unfold_slot(idx, depth=0):
    """Recursively generate a slot and its children."""
    val = (idx * 2654435761) % 4096 / 4096.0
    slot = {
        "idx": idx,
        "value": val,
        "char": hdgl_char(idx),
        "children": []
    }
    if depth < RECURSION_DEPTH:
        # spawn 2 deterministic children per slot
        for offset in [1,2]:
            child_idx = (idx*offset + depth*1337) % MAX_SLOTS
            slot["children"].append(unfold_slot(child_idx, depth+1))
    return slot

def export_recursive_vectors(num_samples=1024):
    """Export a small sample of recursive vectors as JSON."""
    vectors = []
    for idx in range(num_samples):
        vectors.append(unfold_slot(idx))
    with open(EXPORT_JSON, "w", encoding="utf-8") as f:
        json.dump(vectors, f, ensure_ascii=False, indent=2)
    print(f"✅ Exported {num_samples} recursive vectors to {EXPORT_JSON}")

# -------------------------------
# BINARY EXPORT
# -------------------------------
def export_binary_lattice(num_samples=4096):
    """Export lattice as compact .hdgl binary."""
    with open(EXPORT_BINARY, "wb") as f:
        for idx in range(num_samples):
            val = (idx * 2654435761) % 4096 / 4096.0
            packed = struct.pack("fI", val, idx)
            f.write(packed)
    print(f"✅ Exported {num_samples} lattice slots to binary {EXPORT_BINARY}")

# -------------------------------
# SHADERS
# -------------------------------
VERTEX_SRC = """
#version 450 core
layout(location = 0) in vec2 pos;
out vec2 texCoord;
void main(){ texCoord = (pos + 1.0)*0.5; gl_Position = vec4(pos,0,1); }
"""

FRAGMENT_SRC = """
#version 450 core
in vec2 texCoord;
out vec4 fragColor;

uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform int latticeWidth;
uniform int latticeHeight;
uniform int yOffset;

float hash_float(int i, int seed) {
    uint ui = uint(i*374761393 + seed*668265263u);
    return float(ui & 0xFFFFFFFFu)/4294967295.0;
}

vec3 computeVectorColor(int idx, float slot) {
    float hue = hash_float(idx,0)*360.0;
    float grad_hue = mod(hue + 180.0,360.0);
    return vec3(hue/360.0, grad_hue/360.0, slot);
}

float hdgl_slot(float val, float r_dim, float omega, int x, int y, int idx){
    float resonance = (x%4==0?0.1*sin(omegaTime*0.05 + float(y)):0.0);
    float wave = (x%3==0?0.3:(x%3==1?0.0:-0.3));
    float omega_inst = phiPowers[y%72];
    float rec = r_dim*val*0.5 + 0.25*sin(omegaTime*r_dim + float(x));
    float new_val = val + omega_inst + resonance + wave + rec + omega*0.05;
    return new_val>threshold?1.0:0.0;
}

void main() {
    int x = int(texCoord.x * float(latticeWidth));
    int y = int(texCoord.y * float(latticeHeight)) + yOffset;
    int idx = y*latticeWidth + x;
    float val = hash_float(idx,12345);
    float r_dim = 0.3 + 0.01*float(y);
    float slot = hdgl_slot(val, r_dim, sin(omegaTime), x, y, idx);
    vec3 color = computeVectorColor(idx, slot);
    fragColor = vec4(color.rgb,1.0);
}
"""

# -------------------------------
# OPENGL + DISPLAY
# -------------------------------
window = None
shader = None
vao = None
textures = []
fbos = []
current = 0
cycle = 0.0
omega_time = 0.0
tile_heights = []
tile_count = 0
frame_times = []

# -------------------------------
# MAIN
# -------------------------------
def main():
    print("✅ Recursive HDGL node starting...")
    export_recursive_vectors()
    export_binary_lattice()
    print("🖥 Starting GPU HDGL folding...")

    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
    glutInitWindowSize(1280,720)
    glutCreateWindow(b"HDGL 16M Slots Node - Recursive")
    # init_gl() and glutMainLoop() would go here as in previous version
    print("⚡ GPU folding ready (OpenGL lattice)")

if __name__=="__main__":
    main()
