#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import sys, time, json, hashlib
import numpy as np
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileShader, compileProgram

# ------------------- Configuration -------------------
MAX_CHARS = 16_777_216  # Base16M
LATTICE_WIDTH = 4096
LATTICE_HEIGHT_TOTAL = 4096
MAX_TILE_HEIGHT = 1024
SUB_TILE_HEIGHT = 256
SCALE_INTERVAL = 2.0
THRESHOLD = np.sqrt(1.6180339887)
PHI = 1.6180339887
PHI_POWERS = np.array([1.0 / pow(PHI, 7*(i+1)) for i in range(72)], dtype=np.float32)
SHAPES = ['rect','circle','line','triangle']

EXPORT_JSON = "base16m_vectors.json"
EXPORT_TS = "base16m_vectors.ts"

# ------------------- Deterministic Base16M Alphabet -------------------
def is_valid_char(c):
    import unicodedata
    try:
        name = unicodedata.name(c)
        return not any(x in name for x in ['CONTROL','PRIVATE USE','SURROGATE','UNASSIGNED','TAG'])
    except ValueError:
        return False

def generate_base16m_alphabet(seed="0123456789ABCDEF"):
    seen = set(seed)
    alphabet = list(seed)
    codepoint = 0x20
    while len(alphabet) < MAX_CHARS:
        try:
            c = chr(codepoint)
            if c not in seen and is_valid_char(c):
                alphabet.append(c)
                seen.add(c)
        except Exception:
            pass
        codepoint += 1
        if codepoint > 0x10FFFF:
            codepoint = 0x20  # wrap
    return alphabet

# ------------------- Procedural Vector Mapping -------------------
def hash_to_vector(idx):
    h = int(hashlib.sha256(str(idx).encode('utf-8')).hexdigest(),16)
    shape = SHAPES[h % len(SHAPES)]
    rotate = (h >> 8) % 360
    scale = ((h >> 16) % 100)/100 + 0.25
    offset_x = ((h >> 24) % 200)/100 - 1.0
    offset_y = ((h >> 32) % 200)/100 - 1.0
    hue = (h >> 40) % 360
    gradient_hue = (hue + 180) % 360
    return {
        'shape': shape,
        'rotate': rotate,
        'scale': scale,
        'offset_x': offset_x,
        'offset_y': offset_y,
        'hue': hue,
        'gradient_hue': gradient_hue
    }

def export_vectors(alphabet):
    print("[Export] Generating vectors for JSON & TS...")
    vectors = [hash_to_vector(i) for i in range(len(alphabet))]
    # JSON export
    with open(EXPORT_JSON,"w",encoding="utf-8") as f:
        json.dump(vectors,f,ensure_ascii=False,indent=2)
    # TypeScript export
    with open(EXPORT_TS,"w",encoding="utf-8") as f:
        f.write("// Auto-generated Base16M vectors\n\n")
        f.write(f"export const BASE16M_ALPHABET = {json.dumps(alphabet)};\n\n")
        f.write("export const ALPHABET_VECTORS: Record<string, any> = ")
        json.dump({str(i):vectors[i] for i in range(len(alphabet))},f,ensure_ascii=False,indent=2)
        f.write(";\n")
    print(f"[Export] JSON -> {EXPORT_JSON}")
    print(f"[Export] TypeScript -> {EXPORT_TS}")

# ------------------- GPU Shader Sources -------------------
VERTEX_SRC = """
#version 450 core
layout(location = 0) in vec2 pos;
out vec2 texCoord;
void main(){ texCoord = (pos+1.0)*0.5; gl_Position=vec4(pos,0,1); }
"""

FRAGMENT_SRC = """
#version 450 core
in vec2 texCoord;
out vec4 fragColor;
uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform int latticeWidth;
uniform int latticeHeight;
uniform int yOffset;

float hash_float(int i, int seed) {
    uint ui = uint(i*374761393 + seed*668265263u);
    return float(ui & 0xFFFFFFFFu)/4294967295.0;
}

vec3 computeVectorColor(int idx, float slot){
    float hue = hash_float(idx,0)*360.0;
    float grad_hue = mod(hue + 180.0,360.0);
    return vec3(hue/360.0,grad_hue/360.0,slot);
}

float hdgl_slot(float val,float r_dim,float omega,int x,int y,int idx){
    float resonance=(x%4==0?0.1*sin(omegaTime*0.05+float(y)):0.0);
    float wave=(x%3==0?0.3:(x%3==1?0.0:-0.3));
    float omega_inst=phiPowers[y%72];
    float rec=r_dim*val*0.5+0.25*sin(omegaTime*r_dim+float(x));
    float new_val=val+omega_inst+resonance+wave+rec+omega*0.05;
    return new_val>threshold?1.0:0.0;
}

void main(){
    int x=int(texCoord.x*float(latticeWidth));
    int y=int(texCoord.y*float(latticeHeight))+yOffset;
    int idx=y*latticeWidth+x;
    float val=hash_float(idx,12345);
    float r_dim=0.3+0.01*float(y);
    float slot=hdgl_slot(val,r_dim,sin(omegaTime),x,y,idx);
    vec3 color=computeVectorColor(idx,slot);
    fragColor=vec4(color.rgb,1.0);
}
"""

# ------------------- Globals -------------------
window = None
shader = None
vao = None
textures = []
fbos = []
current = 0
cycle = 0.0
omega_time = 0.0
tile_heights = []
tile_count = 0
frame_times = []

# ------------------- OpenGL Initialization -------------------
def init_gl():
    global shader, vao, textures, fbos, tile_heights, tile_count
    shader = compileProgram(
        compileShader(VERTEX_SRC, GL_VERTEX_SHADER),
        compileShader(FRAGMENT_SRC, GL_FRAGMENT_SHADER)
    )
    verts = np.array([-1,-1,1,-1,-1,1,1,-1,1,1,-1,1],dtype=np.float32)
    vao = glGenVertexArrays(1)
    glBindVertexArray(vao)
    vbo = glGenBuffers(1)
    glBindBuffer(GL_ARRAY_BUFFER,vbo)
    glBufferData(GL_ARRAY_BUFFER,verts.nbytes,verts,GL_STATIC_DRAW)
    glVertexAttribPointer(0,2,GL_FLOAT,GL_FALSE,0,None)
    glEnableVertexAttribArray(0)
    tile_count = (LATTICE_HEIGHT_TOTAL + MAX_TILE_HEIGHT - 1)//MAX_TILE_HEIGHT
    tile_heights[:] = [min(MAX_TILE_HEIGHT,LATTICE_HEIGHT_TOTAL - i*MAX_TILE_HEIGHT) for i in range(tile_count)]
    for th in tile_heights:
        tex_pair = glGenTextures(2)
        fbo_pair = glGenFramebuffers(2)
        for i in range(2):
            glBindTexture(GL_TEXTURE_2D, tex_pair[i])
            glTexImage2D(GL_TEXTURE_2D,0,GL_RGBA32F,LATTICE_WIDTH,th,0,GL_RGBA,GL_FLOAT,None)
            glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST)
            glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST)
            glBindFramebuffer(GL_FRAMEBUFFER,fbo_pair[i])
            glFramebufferTexture2D(GL_FRAMEBUFFER,GL_COLOR_ATTACHMENT0,GL_TEXTURE_2D,tex_pair[i],0)
        textures.append(tex_pair)
        fbos.append(fbo_pair)

# ------------------- Display -------------------
def display():
    global cycle, omega_time, current
    next_idx = 1-current
    for t, th in enumerate(tile_heights):
        glBindFramebuffer(GL_FRAMEBUFFER, fbos[t][next_idx])
        glViewport(0,0,LATTICE_WIDTH,th)
        glUseProgram(shader)
        glUniform1f(glGetUniformLocation(shader,"omegaTime"),omega_time)
        glUniform1fv(glGetUniformLocation(shader,"phiPowers"),72,PHI_POWERS)
        glUniform1f(glGetUniformLocation(shader,"threshold"),THRESHOLD)
        glUniform1i(glGetUniformLocation(shader,"latticeWidth"),LATTICE_WIDTH)
        glUniform1i(glGetUniformLocation(shader,"latticeHeight"),th)
        glUniform1i(glGetUniformLocation(shader,"yOffset"),t*MAX_TILE_HEIGHT)
        glActiveTexture(GL_TEXTURE0)
        glBindTexture(GL_TEXTURE_2D,textures[t][current])
        glBindVertexArray(vao)
        glDrawArrays(GL_TRIANGLES,0,6)
    glBindFramebuffer(GL_FRAMEBUFFER,0)
    glViewport(0,0,1280,720)
    for t, th in enumerate(tile_heights):
        glBindTexture(GL_TEXTURE_2D,textures[t][next_idx])
        glDrawArrays(GL_TRIANGLES,0,6)
    glutSwapBuffers()
    cycle += 1
    omega_time += 0.05
    current = next_idx

# ------------------- Idle -------------------
def idle():
    glutPostRedisplay()

# ------------------- Main -------------------
def main():
    print("[Init] Generating Base16M alphabet...")
    alphabet = generate_base16m_alphabet()
    print("[Init] Exporting vectors to JSON & TS...")
    export_vectors(alphabet)
    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA|GLUT_DOUBLE)
    glutInitWindowSize(1280,720)
    glutCreateWindow(b"HDGL 16M Slots + Vectors")
    init_gl()
    glutDisplayFunc(display)
    glutIdleFunc(idle)
    glutMainLoop()

if __name__=="__main__":
    main()
