#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
colorful9opengl_stream.py
Streaming-safe HDGL recursive node:
- Base4096 chunked export/import with HMAC
- JSON and binary lattice export
- GPU/OpenGL lattice stub
- Streams vectors to handle huge trees safely
"""

import sys, json, math, struct, unicodedata, hmac, hashlib, pickle, os
import numpy as np
from base4096 import encode, decode
from OpenGL.GL import *
from OpenGL.GLUT import *

# -------------------------------
# CONFIG
# -------------------------------
LATTICE_WIDTH = 4096
LATTICE_HEIGHT_TOTAL = 4096
PHI = 1.6180339887
PHI_POWERS = np.array([1.0 / pow(PHI, 7*(i+1)) for i in range(72)], dtype=np.float32)
THRESHOLD = math.sqrt(PHI)
MAX_SLOTS = 16_777_216
RECURSION_DEPTH = 3

EXPORT_JSON = "hdgl_vectors.json"
EXPORT_BINARY = "hdgl_lattice.hdgl"
EXPORT_BASE4096 = "vectors_stream.b4096"
CHUNK_SIZE = 512  # Number of vectors per Base4096 chunk
HMAC_KEY = b"SuperSecretHDGLKey_Stream"

# -------------------------------
# CHAR SLOT HELPERS
# -------------------------------
def is_valid_char(c):
    try:
        cp = ord(c)
        if 0xD800 <= cp <= 0xDFFF:
            return False
        name = unicodedata.name(c)
        cat = unicodedata.category(c)
        if any(bad in name for bad in ['CONTROL','PRIVATE USE','UNASSIGNED','TAG']):
            return False
        if cat in ['Mn','Mc','Me','Cc','Cf','Cs','Cn','Co','Zs']:
            return False
        return True
    except ValueError:
        return False

def hdgl_char(idx):
    h = (idx * 2654435761) % 0x110000
    return chr(h) if is_valid_char(chr(h)) else chr((h+1)%0x110000)

# -------------------------------
# RECURSIVE VECTORS
# -------------------------------
def unfold_slot(idx, depth=0):
    val = (idx * 2654435761) % 4096 / 4096.0
    slot = {
        "idx": idx,
        "value": val,
        "char": hdgl_char(idx),
        "children": []
    }
    if depth < RECURSION_DEPTH:
        for offset in [1,2]:
            child_idx = (idx*offset + depth*1337) % MAX_SLOTS
            slot["children"].append(unfold_slot(child_idx, depth+1))
    return slot

def generate_vectors_stream(num_vectors):
    for idx in range(num_vectors):
        yield unfold_slot(idx)

# -------------------------------
# SERIALIZATION & HMAC
# -------------------------------
def serialize_vector_chunk(chunk):
    return pickle.dumps(chunk, protocol=4)

def deserialize_vector_chunk(b):
    return pickle.loads(b)

# -------------------------------
# STREAMING BASE4096 EXPORT
# -------------------------------
def export_vectors_base4096_streamed(num_vectors, outfile=EXPORT_BASE4096, chunk_size=CHUNK_SIZE):
    with open(outfile, "w", encoding="utf-8") as f:
        buffer = []
        count = 0
        for vector in generate_vectors_stream(num_vectors):
            buffer.append(vector)
            if len(buffer) >= chunk_size:
                data = serialize_vector_chunk(buffer)
                sig = hmac.new(HMAC_KEY, data, hashlib.sha256).digest()
                encoded = encode(sig + data)
                f.write(encoded + "\n")
                count += len(buffer)
                buffer.clear()
        if buffer:
            data = serialize_vector_chunk(buffer)
            sig = hmac.new(HMAC_KEY, data, hashlib.sha256).digest()
            encoded = encode(sig + data)
            f.write(encoded + "\n")
            count += len(buffer)
    print(f"✅ Exported {count} vectors in streamed Base4096 chunks to {outfile}")

def import_vectors_base4096_streamed(infile=EXPORT_BASE4096):
    vectors = []
    with open(infile, "r", encoding="utf-8") as f:
        for line in f:
            line = line.strip()
            if not line:
                continue
            decoded_bytes = decode(line)
            sig_stored = decoded_bytes[:32]
            data = decoded_bytes[32:]
            sig_check = hmac.new(HMAC_KEY, data, hashlib.sha256).digest()
            if sig_check != sig_stored:
                raise ValueError("❌ HMAC mismatch — file corrupted or tampered.")
            chunk_vectors = deserialize_vector_chunk(data)
            vectors.extend(chunk_vectors)
    print(f"✅ Successfully reconstructed {len(vectors)} vectors from streamed Base4096")
    return vectors

# -------------------------------
# JSON & Binary EXPORT
# -------------------------------
def export_vectors_json_stream(vectors, outfile=EXPORT_JSON):
    def filter_surrogates(obj):
        if isinstance(obj, dict):
            return {k: filter_surrogates(v) for k,v in obj.items()}
        if isinstance(obj, list):
            return [filter_surrogates(x) for x in obj]
        if isinstance(obj, str):
            return ''.join(c for c in obj if 0xD800 > ord(c) or ord(c) > 0xDFFF)
        return obj
    safe_vectors = filter_surrogates(vectors)
    with open(outfile, "w", encoding="utf-8") as f:
        json.dump(safe_vectors, f, ensure_ascii=False, indent=2)
    print(f"✅ Exported {len(vectors)} vectors to JSON: {outfile}")

def export_binary_lattice(num_samples=128, outfile=EXPORT_BINARY):
    with open(outfile, "wb") as f:
        for idx in range(num_samples):
            val = (idx * 2654435761) % 4096 / 4096.0
            f.write(struct.pack("fI", val, idx))
    print(f"✅ Exported {num_samples} lattice slots to binary {outfile}")

# -------------------------------
# OPENGL SHADERS (stub)
# -------------------------------
VERTEX_SRC = """
#version 450 core
layout(location = 0) in vec2 pos;
out vec2 texCoord;
void main(){ texCoord = (pos + 1.0)*0.5; gl_Position = vec4(pos,0,1); }
"""

FRAGMENT_SRC = """
#version 450 core
in vec2 texCoord;
out vec4 fragColor;

uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform int latticeWidth;
uniform int latticeHeight;
uniform int yOffset;

float hash_float(int i, int seed) {
    uint ui = uint(i*374761393 + seed*668265263u);
    return float(ui & 0xFFFFFFFFu)/4294967295.0;
}

vec3 computeVectorColor(int idx, float slot) {
    float hue = hash_float(idx,0)*360.0;
    float grad_hue = mod(hue + 180.0,360.0);
    return vec3(hue/360.0, grad_hue/360.0, slot);
}

float hdgl_slot(float val, float r_dim, float omega, int x, int y, int idx){
    float resonance = (x%4==0?0.1*sin(omegaTime*0.05 + float(y)):0.0);
    float wave = (x%3==0?0.3:(x%3==1?0.0:-0.3));
    float omega_inst = phiPowers[y%72];
    float rec = r_dim*val*0.5 + 0.25*sin(omegaTime*r_dim + float(x));
    float new_val = val + omega_inst + resonance + wave + rec + omega*0.05;
    return new_val>threshold?1.0:0.0;
}

void main() {
    int x = int(texCoord.x * float(latticeWidth));
    int y = int(texCoord.y * float(latticeHeight)) + yOffset;
    int idx = y*latticeWidth + x;
    float val = hash_float(idx,12345);
    float r_dim = 0.3 + 0.01*float(y);
    float slot = hdgl_slot(val, r_dim, sin(omegaTime), x, y, idx);
    vec3 color = computeVectorColor(idx, slot);
    fragColor = vec4(color.rgb,1.0);
}
"""

# -------------------------------
# MAIN
# -------------------------------
def main():
    NUM_VECTORS = 1024  # or higher for large trees
    print("🚀 Starting streaming-safe HDGL node...")

    # Streamed Base4096 export
    export_vectors_base4096_streamed(NUM_VECTORS)

    # Reconstruct to verify round-trip
    reconstructed = import_vectors_base4096_streamed()

    # JSON & binary export (after reconstruction)
    export_vectors_json_stream(reconstructed)
    export_binary_lattice(num_samples=128)

    # OpenGL stub
    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
    glutInitWindowSize(1280,720)
    glutCreateWindow(b"HDGL Recursive Node - Streaming")
    print("⚡ GPU folding ready (OpenGL lattice stub)")

if __name__=="__main__":
    main()
