#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
colorful9opengl_streaming.py
Streaming-safe HDGL recursive node with real-time OpenGL color folding.
Handles massive lattices using chunked generation and GPU texture streaming.
"""

import sys, json, math, struct, unicodedata, hashlib, hmac, time
import numpy as np
from base4096 import encode, decode
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileShader, compileProgram

# -------------------------------
# CONFIG
# -------------------------------
LATTICE_WIDTH = 4096
LATTICE_HEIGHT_TOTAL = 4096
CHUNK_HEIGHT = 128            # How many rows per streaming chunk
PHI = 1.6180339887
PHI_POWERS = np.array([1.0 / pow(PHI, 7*(i+1)) for i in range(72)], dtype=np.float32)
THRESHOLD = math.sqrt(PHI)
MAX_SLOTS = 16_777_216
RECURSION_DEPTH = 3

EXPORT_JSON = "hdgl_vectors.json"
EXPORT_BASE4096 = "vectors.b4096"
HMAC_KEY = b"HDGL_STREAM_SECRET_KEY"

# -------------------------------
# CHAR SLOT HELPERS
# -------------------------------
def is_valid_char(c):
    try:
        cp = ord(c)
        if 0xD800 <= cp <= 0xDFFF:
            return False
        name = unicodedata.name(c)
        cat = unicodedata.category(c)
        if any(bad in name for bad in ['CONTROL','PRIVATE USE','UNASSIGNED','TAG']):
            return False
        if cat in ['Mn','Mc','Me','Cc','Cf','Cs','Cn','Co','Zs']:
            return False
        return True
    except ValueError:
        return False

def hdgl_char(idx):
    h = (idx * 2654435761) % 0x110000
    c = chr(h) if is_valid_char(chr(h)) else chr((h+1)%0x110000)
    return c

# -------------------------------
# RECURSIVE VECTORS
# -------------------------------
def unfold_slot(idx, depth=0):
    val = (idx * 2654435761) % 4096 / 4096.0
    slot = {
        "idx": idx,
        "value": val,
        "char": hdgl_char(idx),
        "children": []
    }
    if depth < RECURSION_DEPTH:
        for offset in [1,2]:
            child_idx = (idx*offset + depth*1337) % MAX_SLOTS
            slot["children"].append(unfold_slot(child_idx, depth+1))
    return slot

def flatten_vector(v):
    if isinstance(v, (list, tuple)):
        return ''.join(flatten_vector(x) for x in v)
    if isinstance(v, dict):
        return ''.join(flatten_vector(v[k]) for k in ['char','value','children'] if k in v)
    return str(v)

def generate_vectors_stream(num_samples=1024, chunk_size=16):
    """Yield recursive vectors in streaming chunks."""
    for start in range(0, num_samples, chunk_size):
        end = min(start+chunk_size, num_samples)
        yield [unfold_slot(idx) for idx in range(start, end)]

# -------------------------------
# BASE4096 + HMAC EXPORT
# -------------------------------
def string_to_base4096_bytes(s):
    """Convert string to surrogate-safe Base4096 bytes."""
    data = bytearray()
    for c in s:
        cp = ord(c)
        if 0xD800 <= cp <= 0xDFFF or cp > 0x10FFFF:
            continue
        data.extend(cp.to_bytes(3,'big', signed=False))
    return bytes(data)

def export_base4096_stream(vectors_iter, outfile=EXPORT_BASE4096):
    """Stream vectors to Base4096 with embedded HMAC."""
    h = hmac.new(HMAC_KEY, digestmod=hashlib.sha256)
    with open(outfile, 'w', encoding='utf-8') as f:
        for chunk in vectors_iter:
            for vec in chunk:
                combined = flatten_vector(vec)
                data = string_to_base4096_bytes(combined)
                h.update(data)
                encoded = encode(data)
                f.write(encoded + "\n")
        f.write("# HMAC:" + h.hexdigest() + "\n")
    print(f"✅ Base4096 vectors exported: {outfile} (signature embedded)")

# -------------------------------
# OPENGL STREAMING
# -------------------------------
VERTEX_SRC = """
#version 450 core
layout(location = 0) in vec2 pos;
out vec2 texCoord;
void main(){ texCoord = (pos + 1.0)*0.5; gl_Position = vec4(pos,0,1); }
"""

FRAGMENT_SRC = """
#version 450 core
in vec2 texCoord;
out vec4 fragColor;
uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform int latticeWidth;
uniform int latticeHeight;
uniform int yOffset;

float hash_float(int i, int seed){
    uint ui = uint(i*374761393 + seed*668265263u);
    return float(ui & 0xFFFFFFFFu)/4294967295.0;
}

vec3 computeVectorColor(int idx,float slot){
    float hue = hash_float(idx,0)*360.0;
    float grad_hue = mod(hue+180.0,360.0);
    return vec3(hue/360.0, grad_hue/360.0, slot);
}

float hdgl_slot(float val,float r_dim,float omega,int x,int y,int idx){
    float resonance = (x%4==0?0.1*sin(omegaTime*0.05+float(y)):0.0);
    float wave = (x%3==0?0.3:(x%3==1?0.0:-0.3));
    float omega_inst = phiPowers[y%72];
    float rec = r_dim*val*0.5 + 0.25*sin(omegaTime*r_dim+float(x));
    float new_val = val + omega_inst + resonance + wave + rec + omega*0.05;
    return new_val>threshold?1.0:0.0;
}

void main(){
    int x=int(texCoord.x*float(latticeWidth));
    int y=int(texCoord.y*float(latticeHeight))+yOffset;
    int idx = y*latticeWidth + x;
    float val = hash_float(idx,12345);
    float r_dim = 0.3+0.01*float(y);
    float slot = hdgl_slot(val,r_dim,sin(omegaTime),x,y,idx);
    vec3 color = computeVectorColor(idx,slot);
    fragColor=vec4(color.rgb,1.0);
}
"""

# -------------------------------
# STREAMING GPU NODE
# -------------------------------
class HDGLNode:
    def __init__(self):
        self.time = 0.0
        self.chunk_y = 0
        self.program = None
        self.init_gl()

    def init_gl(self):
        self.program = compileProgram(
            compileShader(VERTEX_SRC, GL_VERTEX_SHADER),
            compileShader(FRAGMENT_SRC, GL_FRAGMENT_SHADER)
        )
        glUseProgram(self.program)
        glViewport(0,0,1280,720)
        # Upload PHI powers uniform
        loc = glGetUniformLocation(self.program, "phiPowers")
        glUniform1fv(loc, len(PHI_POWERS), PHI_POWERS)
        # Lattice uniforms
        glUniform1i(glGetUniformLocation(self.program,"latticeWidth"), LATTICE_WIDTH)
        glUniform1i(glGetUniformLocation(self.program,"latticeHeight"), LATTICE_HEIGHT_TOTAL)

    def render_chunk(self, y_offset):
        glUseProgram(self.program)
        glUniform1f(glGetUniformLocation(self.program,"omegaTime"), self.time)
        glUniform1i(glGetUniformLocation(self.program,"yOffset"), y_offset)
        # Full-screen quad
        glBegin(GL_QUADS)
        glVertex2f(-1,-1); glVertex2f(1,-1)
        glVertex2f(1,1); glVertex2f(-1,1)
        glEnd()

    def update(self):
        self.time += 0.016
        self.render_chunk(self.chunk_y)
        self.chunk_y = (self.chunk_y + CHUNK_HEIGHT) % LATTICE_HEIGHT_TOTAL
        glutSwapBuffers()
        glutPostRedisplay()

# -------------------------------
# MAIN
# -------------------------------
def main():
    print("🚀 Starting HDGL recursive node...")

    # Streaming Base4096 export
    vectors_iter = generate_vectors_stream(num_samples=32, chunk_size=8)
    export_base4096_stream(vectors_iter)

    # JSON streaming
    vectors_iter = generate_vectors_stream(num_samples=32, chunk_size=8)
    safe_json = []
    for chunk in vectors_iter:
        safe_json.extend(chunk)
    with open(EXPORT_JSON, 'w', encoding='utf-8') as f:
        json.dump(safe_json, f, ensure_ascii=False, indent=2)
    print(f"✅ Exported {len(safe_json)} vectors to JSON: {EXPORT_JSON}")

    # OpenGL streaming
    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
    glutInitWindowSize(1280,720)
    glutCreateWindow(b"HDGL Streaming Node")
    node = HDGLNode()
    glutDisplayFunc(node.update)
    print("⚡ GPU folding ready (real-time streaming)")
    glutMainLoop()

if __name__=="__main__":
    main()
