#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import unicodedata
import json
import hashlib

# -------- CONFIG --------
OUTPUT_TS_FILE = "frozen_base32768_alphabet.ts"
SEED = (
    "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz"
    "!@#$%^&*()-_+=[{]};:',\"<>?/`|~"
)
MAX_CHARS = 32768

# -------- DETERMINISTIC ALPHABET GENERATOR --------
def generate_alphabet(seed, max_chars):
    seen = set()
    alphabet = []

    # Add seed first
    for ch in seed:
        if ch not in seen:
            alphabet.append(ch)
            seen.add(ch)

    # Fill remaining from Unicode codepoints
    for codepoint in range(0x20, 0x110000):
        if len(alphabet) >= max_chars:
            break
        c = chr(codepoint)
        if c in seen:
            continue
        try:
            cat = unicodedata.category(c)
            if cat in ['Zs','Cc','Cf','Cs','Cn','Co']:  # skip spaces, controls, private, surrogate, unassigned
                continue
            alphabet.append(c)
            seen.add(c)
        except ValueError:
            continue

    if len(alphabet) < max_chars:
        raise ValueError(f"Only generated {len(alphabet)} characters; expected {max_chars}.")
    return ''.join(alphabet[:max_chars])

# -------- COLOR/GRADIENT GENERATION --------
def style_for_char(c):
    """Deterministically generate a full-spectrum gradient for a character."""
    # Hash the character for deterministic hue
    h = int(hashlib.sha256(c.encode("utf-8")).hexdigest(), 16) % 360
    s, l = 70, 50
    color = f"hsl({h},{s}%,{l}%)"
    # Full 360° opposite hue for gradient
    opposite_hue = (h + 180) % 360
    gradient = f"linear-gradient(hsl({h},{s}%,{l}%),hsl({opposite_hue},{s}%,{l}%))"
    return {"color": color, "gradient": gradient}

# -------- TYPE-SCRIPT EXPORT --------
def export_typescript(alphabet, ts_file):
    styles = {c: style_for_char(c) for c in alphabet}
    with open(ts_file, "w", encoding="utf-8") as f:
        f.write("// Auto-generated Base-32,768 alphabet with full-spectrum gradients\n\n")
        f.write(f'export const BASE32768_ALPHABET = {json.dumps(alphabet)};\n\n')
        f.write("export const ALPHABET_STYLES: Record<string, {color:string, gradient:string}> = ")
        json.dump(styles, f, ensure_ascii=False, indent=2)
        f.write(";\n")
    print(f"✅ TypeScript export written to {ts_file}")
    print(f"Total characters: {len(alphabet)}")

# -------- MAIN --------
def main():
    alphabet = generate_alphabet(SEED, MAX_CHARS)
    export_typescript(alphabet, OUTPUT_TS_FILE)

if __name__ == "__main__":
    main()
