#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import unicodedata
import json
import hashlib

# -------- CONFIG --------
OUTPUT_TS_FILE = "frozen_base524288_alphabet.ts"
SEED = (
    "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz"
    "!@#$%^&*()-_+=[{]};:',\"<>?/`|~"
)
MAX_CHARS = 524_288  # Updated base

# -------- DETERMINISTIC ALPHABET GENERATOR --------
def generate_alphabet(seed, max_chars):
    seen = set()
    alphabet = []

    # Add seed first
    for ch in seed:
        if ch not in seen:
            alphabet.append(ch)
            seen.add(ch)

    # Fill remaining from Unicode codepoints (loop multiple times if needed)
    codepoint = 0x20
    while len(alphabet) < max_chars:
        if codepoint >= 0x110000:  # Unicode max
            codepoint = 0x20  # wrap around for repeated plane coverage
        c = chr(codepoint)
        codepoint += 1
        if c in seen:
            continue
        try:
            cat = unicodedata.category(c)
            if cat in ['Zs','Cc','Cf','Cs','Cn','Co']:
                continue
            alphabet.append(c)
            seen.add(c)
        except ValueError:
            continue

    return ''.join(alphabet[:max_chars])

# -------- COLOR/GRADIENT GENERATION --------
def style_for_char(c):
    """Deterministically generate a full-spectrum gradient for a character."""
    h = int(hashlib.sha256(c.encode("utf-8")).hexdigest(), 16) % 360
    s, l = 70, 50
    color = f"hsl({h},{s}%,{l}%)"
    opposite_hue = (h + 180) % 360
    gradient = f"linear-gradient(hsl({h},{s}%,{l}%),hsl({opposite_hue},{s}%,{l}%))"
    return {"color": color, "gradient": gradient}

# -------- TYPE-SCRIPT EXPORT --------
def export_typescript(alphabet, ts_file):
    styles = {c: style_for_char(c) for c in alphabet}
    with open(ts_file, "w", encoding="utf-8") as f:
        f.write("// Auto-generated Base-524,288 alphabet with full-spectrum gradients\n\n")
        f.write(f'export const BASE524288_ALPHABET = {json.dumps(alphabet)};\n\n')
        f.write("export const ALPHABET_STYLES: Record<string, {color:string, gradient:string}> = ")
        json.dump(styles, f, ensure_ascii=False, indent=2)
        f.write(";\n")
    print(f"✅ TypeScript export written to {ts_file}")
    print(f"Total characters: {len(alphabet)}")

# -------- MAIN --------
def main():
    alphabet = generate_alphabet(SEED, MAX_CHARS)
    export_typescript(alphabet, OUTPUT_TS_FILE)

if __name__ == "__main__":
    main()
