#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
tv10_ultrabatch_gl.py
HDGL recursive node with 131,072 channels, optimized OpenCL HMAC,
streaming-safe Base4096 export, and OpenGL real-time folding with audio/video modulation.
"""

import sys, math, struct, json, unicodedata, hmac, hashlib
import numpy as np
from base4096 import encode
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileShader, compileProgram
import pyopencl as cl
from concurrent.futures import ThreadPoolExecutor
import pygame
import scipy.io.wavfile as wavfile
import os
import tempfile
import time
from pydub import AudioSegment
from pydub.exceptions import CouldntDecodeError
from moviepy.editor import VideoFileClip
from decimal import Decimal, getcontext
from scipy.special import jv, jn_zeros
from collections import OrderedDict  # Added for LRU cache

# -------------------------------
# CONFIG
# -------------------------------
LATTICE_WIDTH = 1920
LATTICE_HEIGHT = 1080
CHANNELS = 131_072
SAMPLES_PER_CHANNEL = 32
PHI = 1.6180339887498948482
PHI_POWERS = np.array([1.0 / pow(PHI, 7*(i+1)) for i in range(72)], dtype=np.float32)
THRESHOLD = math.sqrt(PHI)
MAX_SLOTS = 16_777_216
USE_OPENCL_HMAC = True
CHUNK_HEIGHT = LATTICE_HEIGHT // 24
getcontext().prec = 120  # For phi-based compression

EXPORT_JSON = "hdgl_vectors.json"
EXPORT_BINARY = "hdgl_lattice.hdgl"
EXPORT_BASE4096 = "vectors_ultrabatch.b4096"
HMAC_KEY = b"ZCHG-Base4096-Signature-Key"

MAX_MEDIA_CACHE = 10  # Added: Limit the number of media data entries in cache to prevent OOM

# -------------------------------
# CHAR SLOT HELPERS
# -------------------------------
def hdgl_char(idx):
    h = (idx * 2654435761) % 0x110000
    c = chr(h) if 0xD800 > h or h > 0xDFFF else chr((h+1)%0x110000)
    return c

# -------------------------------
# RECURSIVE VECTORS
# -------------------------------
def unfold_slot(idx, depth=0):
    val = (idx * 2654435761) % 4096 / 4096.0
    slot = {"idx": idx, "value": val, "char": hdgl_char(idx), "children": []}
    if depth < 3:
        for offset in [1,2]:
            child_idx = (idx*offset + depth*1337) % MAX_SLOTS
            slot["children"].append(unfold_slot(child_idx, depth+1))
    return slot

def build_recursive_vectors(num_samples=SAMPLES_PER_CHANNEL):
    return [unfold_slot(idx) for idx in range(num_samples)]

def flatten_indices_to_bytes(indices):
    arr = bytearray()
    for idx in indices:
        cp = int(idx) % 0x10FFFF
        if 0xD800 <= cp <= 0xDFFF:
            cp += 1
        arr.extend(cp.to_bytes(3,'big'))
    return bytes(arr)

# -------------------------------
# Cymatic Generators
# -------------------------------
def get_cymatic_params(f_query):
    alpha = 0.5 + 0.5 * np.sin(f_query / 100)
    beta = 0.5 + 0.5 * np.cos(f_query / 200)
    eta = 0.3 + 0.3 * np.sin(f_query / 50)
    zeta = 0.3 + 0.3 * np.cos(f_query / 70)
    return {"alpha": alpha, "beta": beta, "eta": eta, "zeta": zeta}

def generate_cartesian(X, Y, params, t):
    Xn, Yn = (X + 1) / 2, (Y + 1) / 2
    return (np.sin(params["alpha"] * np.pi * Xn + t) * np.sin(params["beta"] * np.pi * Yn + t) +
            params["eta"] * np.cos(params["zeta"] * np.pi * (Xn + Yn) + t))

def note_to_freq(note_val):
    return 220.0 * 2 ** (note_val / 12)

# -------------------------------
# Phi Compression
# -------------------------------
class PhiCompressionInteractive:
    def __init__(self, phi=str(PHI), epsilon="1e-12", max_value=256):
        self.PHI = Decimal(phi)
        self.EPSILON = Decimal(epsilon)
        self.MAX_VALUE = max_value
        self.log_phi = self.PHI.ln()

    def encode_sequence(self, sequence):
        return sum(Decimal(x) for x in sequence)

    def decode_sequence(self, exponent, seq_length):
        sequence = []
        current_sum = exponent
        for _ in range(seq_length):
            if current_sum <= 0:
                break
            b = int(current_sum.to_integral_value(rounding="ROUND_FLOOR"))
            b = min(max(b, 0), self.MAX_VALUE - 1)
            sequence.append(b)
            current_sum -= Decimal(b)
        return sequence

# -------------------------------
# Nodal Hash
# -------------------------------
def nodal_hash(alpha, beta, eta, zeta, Nx=32, Ny=32):
    x = np.linspace(0, 1, Nx)
    y = np.linspace(0, 1, Ny)
    Xg, Yg = np.meshgrid(x, y)
    Z = np.sin(alpha * np.pi * Xg) * np.sin(beta * np.pi * Yg) + eta * np.cos(zeta * np.pi * (Xg + Yg))
    flat = Z.flatten()
    flat_bytes = (np.round(flat * 1e5)).astype(np.int64).tobytes()
    h = hashlib.sha256(flat_bytes).hexdigest()
    h_num = int(h[:16], 16)
    return Decimal(h_num)

# -------------------------------
# Optimized OpenCL HMAC
# -------------------------------
OPENCL_HMAC_KERNEL = """
#define SHA256_BLOCK_SIZE 64
#define SHA256_DIGEST_SIZE 32

__constant uint k[64] = {
    0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5, 0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5,
    0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3, 0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174,
    0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc, 0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da,
    0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7, 0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967,
    0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13, 0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85,
    0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3, 0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070,
    0x19a4c116, 0x1e376c08, 0x2748774c, 0x34b0bcb5, 0x391c0cb3, 0x4ed8aa4a, 0x5b9cca4f, 0x682e6ff3,
    0x748f82ee, 0x78a5636f, 0x84c87814, 0x8cc70208, 0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2
};

void sha256_transform(__private uint state[8], __global uchar *block) {
    uint a, b, c, d, e, f, g, h, t1, t2, m[64];
    for (int i = 0, j = 0; i < 16; ++i, j += 4)
        m[i] = (block[j] << 24) | (block[j + 1] << 16) | (block[j + 2] << 8) | (block[j + 3]);
    for (int i = 16; i < 64; ++i)
        m[i] = (m[i - 16] + m[i - 7] + (rotate(m[i - 15], 25U) ^ rotate(m[i - 15], 14U) ^ (m[i - 15] >> 3)) +
                (rotate(m[i - 2], 15U) ^ rotate(m[i - 2], 13U) ^ (m[i - 2] >> 10))) & 0xFFFFFFFF;
    a = state[0]; b = state[1]; c = state[2]; d = state[3];
    e = state[4]; f = state[5]; g = state[6]; h = state[7];
    for (int i = 0; i < 64; ++i) {
        t1 = h + (rotate(e, 26U) ^ rotate(e, 21U) ^ rotate(e, 7U)) + ((e & f) ^ (~e & g)) + k[i] + m[i];
        t2 = (rotate(a, 30U) ^ rotate(a, 19U) ^ rotate(a, 10U)) + ((a & b) ^ (a & c) ^ (b & c));
        h = g; g = f; f = e; e = d + t1; d = c; c = b; b = a; a = t1 + t2;
    }
    state[0] += a; state[1] += b; state[2] += c; state[3] += d;
    state[4] += e; state[5] += f; state[6] += g; state[7] += h;
}

__kernel void hmac_sha256(__global uchar *key, __global uchar *data, __global uchar *out, uint data_len, uint key_len) {
    int gid = get_global_id(0);
    uint state[8] = {0x6a09e667, 0xbb67ae85, 0x3c6ef372, 0xa54ff53a, 0x510e527f, 0x9b05688c, 0x1f83d9ab, 0x5be0cd19};
    uchar block[SHA256_BLOCK_SIZE];
    uchar o_key_pad[SHA256_BLOCK_SIZE];
    uchar i_key_pad[SHA256_BLOCK_SIZE];
    
    for (int i = 0; i < SHA256_BLOCK_SIZE; ++i) {
        uchar k = (i < key_len) ? key[i] : 0;
        o_key_pad[i] = k ^ 0x5c;
        i_key_pad[i] = k ^ 0x36;
    }
    
    for (int i = 0; i < SHA256_BLOCK_SIZE; ++i)
        block[i] = i_key_pad[i];
    for (int i = 0; i < data_len && (gid * data_len + i) < data_len * get_global_size(0); ++i)
        block[i] = data[gid * data_len + i];
    sha256_transform(state, block);
    
    for (int i = 0; i < SHA256_BLOCK_SIZE; ++i)
        block[i] = o_key_pad[i];
    for (int i = 0; i < SHA256_DIGEST_SIZE; ++i)
        block[i] = (state[i >> 2] >> (24 - (i % 4) * 8)) & 0xFF;
    sha256_transform(state, block);
    
    for (int i = 0; i < SHA256_DIGEST_SIZE; ++i)
        out[gid * SHA256_DIGEST_SIZE + i] = (state[i >> 2] >> (24 - (i % 4) * 8)) & 0xFF;
}
"""

def init_opencl():
    platform = cl.get_platforms()[0]
    device = platform.get_devices()[0]
    ctx = cl.Context([device])
    queue = cl.CommandQueue(ctx)
    program = cl.Program(ctx, OPENCL_HMAC_KERNEL).build()
    kernel = cl.Kernel(program, "hmac_sha256")
    return ctx, queue, kernel

# -------------------------------
# Modulated Lattice and Media
# -------------------------------
def compute_modulated_lattice(channel, x, y, audio_level, video_brightness, t):
    indices = np.arange(SAMPLES_PER_CHANNEL, dtype=np.uint32) + (channel * MAX_SLOTS) // CHANNELS
    val = (indices * 2654435761 % 4096) / 4096.0
    f_query = note_to_freq(val[0] * 72)
    params = get_cymatic_params(f_query)
    Xg, Yg = np.meshgrid(np.linspace(-1, 1, LATTICE_WIDTH), np.linspace(-1, 1, LATTICE_HEIGHT))
    cymatic_pattern = generate_cartesian(Xg, Yg, params, t)
    slot_val = val[0] + audio_level * 0.2 + video_brightness * 0.2 + cymatic_pattern[y % LATTICE_HEIGHT, x % LATTICE_WIDTH] * 0.1
    return slot_val if slot_val > THRESHOLD else 0.0

# -------------------------------
# Ultra-Batch Export with Modulation
# -------------------------------
def export_channels_ultrabatch(channel_list, out_file=EXPORT_BASE4096):
    global USE_OPENCL_HMAC
    print(f"🚀 Exporting batch of {len(channel_list)} channels...")
    
    compressor = PhiCompressionInteractive()
    all_indices = np.array([np.arange(SAMPLES_PER_CHANNEL, dtype=np.uint32) + (ch * MAX_SLOTS) // CHANNELS for ch in channel_list])
    modulated_bytes = []
    
    for ch in channel_list:
        media_path = get_media_for_channel(ch)
        audio_level = 0.0
        video_brightness = 0.0
        if media_path and media_path in media_data_cache:
            rate, data = media_data_cache[media_path]
            sample_pos = int((pygame.mixer.music.get_pos() / 1000.0) * rate) if pygame.mixer.music.get_busy() else 0
            audio_level = abs(data[sample_pos % len(data)]) / 32768.0 if sample_pos < len(data) else 0.0
            if current_clip and media_path.lower().endswith(('.avi', '.mp4', '.mpeg')):
                pos_sec = (pygame.mixer.music.get_pos() / 1000.0) % current_clip_duration
                try:
                    frame = current_clip.get_frame(pos_sec)
                    video_brightness = np.mean(frame) / 255.0
                except:
                    video_brightness = 0.0
        indices = all_indices[channel_list.index(ch)]
        modulated_vals = [compute_modulated_lattice(ch, x, y, audio_level, video_brightness, omega_time) for x in range(LATTICE_WIDTH) for y in range(LATTICE_HEIGHT)][:SAMPLES_PER_CHANNEL]
        compressed = compressor.encode_sequence(modulated_vals)
        modulated_bytes.append(flatten_indices_to_bytes([int(v * 4096) % 0x10FFFF for v in modulated_vals]))
    
    all_bytes = b''.join(modulated_bytes)
    offsets = np.cumsum([0] + [len(b) for b in modulated_bytes[:-1]])
    
    ctx = queue = kernel = None
    hmac_digests = []
    
    if USE_OPENCL_HMAC:
        try:
            ctx, queue, kernel = init_opencl()
            key_buf = cl.Buffer(ctx, cl.mem_flags.READ_ONLY | cl.mem_flags.COPY_HOST_PTR, hostbuf=HMAC_KEY)
            data_buf = cl.Buffer(ctx, cl.mem_flags.READ_ONLY | cl.mem_flags.COPY_HOST_PTR, hostbuf=np.frombuffer(all_bytes, np.uint8))
            out_buf = cl.Buffer(ctx, cl.mem_flags.WRITE_ONLY, len(channel_list) * 32)
            kernel(queue, (len(channel_list),), None, key_buf, data_buf, out_buf, np.uint32(len(modulated_bytes[0])), np.uint32(len(HMAC_KEY)))
            result = np.empty(len(channel_list) * 32, dtype=np.uint8)
            cl.enqueue_copy(queue, result, out_buf)
            queue.finish()
            for i in range(len(channel_list)):
                params = get_cymatic_params(note_to_freq(channel_list[i] % 72))
                nodal_h = nodal_hash(**params)
                digest = result[i * 32:(i + 1) * 32]
                digest = hashlib.sha256(digest + str(nodal_h).encode()).digest()
                hmac_digests.append(digest)
            print("⚡ OpenCL HMAC ultra-batch completed")
        except Exception as e:
            print(f"⚠️ OpenCL HMAC failed: {e}, falling back to CPU")
            USE_OPENCL_HMAC = False
    
    if not USE_OPENCL_HMAC:
        with ThreadPoolExecutor() as exe:
            def compute_hmac(b, ch):
                params = get_cymatic_params(note_to_freq(ch % 72))
                nodal_h = nodal_hash(**params)
                return hashlib.sha256(b + str(nodal_h).encode()).digest()
            hmac_digests = list(exe.map(lambda b, ch: compute_hmac(b, ch), modulated_bytes, channel_list))
    
    with open(out_file, "a", encoding="utf-8") as f:
        for i, (bts, digest) in enumerate(zip(modulated_bytes, hmac_digests)):
            ch = channel_list[i]
            f.write(f"#Channel:{ch}\n")
            f.write(encode(bts) + "\n")
            f.write(f"#HMAC:{encode(digest)}\n")
    print(f"✅ Batch export complete for channels {channel_list}")

# -------------------------------
# JSON/Binary Exports with Modulation
# -------------------------------
def export_recursive_vectors_json(vectors, outfile=EXPORT_JSON):
    def filter_surrogates(obj):
        if isinstance(obj, dict): return {k: filter_surrogates(v) for k,v in obj.items()}
        if isinstance(obj, list): return [filter_surrogates(x) for x in obj]
        if isinstance(obj, str): return ''.join(c for c in obj if 0xD800 > ord(c) or ord(c) > 0xDFFF)
        return obj
    compressor = PhiCompressionInteractive()
    modulated_vectors = []
    for v in vectors:
        ch = v["idx"] % CHANNELS
        media_path = get_media_for_channel(ch)
        audio_level = video_brightness = 0.0
        if media_path and media_data_cache.get(media_path):
            rate, data = media_data_cache[media_path]
            sample_pos = int((pygame.mixer.music.get_pos() / 1000.0) * rate) if pygame.mixer.music.get_busy() else 0
            audio_level = abs(data[sample_pos % len(data)]) / 32768.0 if sample_pos < len(data) else 0.0
            if current_clip and media_path.lower().endswith(('.avi', '.mp4', '.mpeg')):
                pos_sec = (pygame.mixer.music.get_pos() / 1000.0) % current_clip_duration
                try:
                    frame = current_clip.get_frame(pos_sec)
                    video_brightness = np.mean(frame) / 255.0
                except:
                    video_brightness = 0.0
        f_query = note_to_freq(v["idx"] % 72)
        params = get_cymatic_params(f_query)
        mod_val = v["value"] + audio_level * 0.2 + video_brightness * 0.2
        mod_vector = v.copy()
        mod_vector["value"] = float(compressor.encode_sequence([mod_val]))
        mod_vector["cymatic_params"] = params
        mod_vector["media"] = media_path or "none"
        modulated_vectors.append(mod_vector)
    safe_vectors = filter_surrogates(modulated_vectors)
    with open(outfile, "w", encoding="utf-8") as f:
        json.dump(safe_vectors, f, ensure_ascii=False, indent=2)
    print(f"✅ Exported {len(vectors)} modulated vectors to JSON: {outfile}")

def export_binary_lattice(num_samples=LATTICE_WIDTH, outfile=EXPORT_BINARY):
    compressor = PhiCompressionInteractive()
    with open(outfile, "wb") as f:
        for idx in range(num_samples):
            val = (idx * 2654435761) % 4096 / 4096.0
            ch = idx % CHANNELS
            media_path = get_media_for_channel(ch)
            audio_level = video_brightness = 0.0
            if media_path and media_data_cache.get(media_path):
                rate, data = media_data_cache[media_path]
                sample_pos = int((pygame.mixer.music.get_pos() / 1000.0) * rate) if pygame.mixer.music.get_busy() else 0
                audio_level = abs(data[sample_pos % len(data)]) / 32768.0 if sample_pos < len(data) else 0.0
                if current_clip and media_path.lower().endswith(('.avi', '.mp4', '.mpeg')):
                    pos_sec = (pygame.mixer.music.get_pos() / 1000.0) % current_clip_duration
                    try:
                        frame = current_clip.get_frame(pos_sec)
                        video_brightness = np.mean(frame) / 255.0
                    except:
                        video_brightness = 0.0
            f_query = note_to_freq(idx % 72)
            params = get_cymatic_params(f_query)
            cymatic_val = generate_cartesian(np.array([0]), np.array([0]), params, omega_time)[0]
            mod_val = val + audio_level * 0.2 + video_brightness * 0.2 + cymatic_val * 0.1
            compressed_val = float(compressor.encode_sequence([mod_val]))
            packed = struct.pack("fI", compressed_val, idx)
            f.write(packed)
    print(f"✅ Exported {num_samples} modulated lattice slots to binary: {outfile}")

# -------------------------------
# OpenGL Control
# -------------------------------
omega_time = 0.0
shader = None
yOffset = 0
current_channel = 0
previous_channel = -1
frame_count = 0
auto_scroll = True
audio_level = 0.0
current_rate = None
current_original_data = None
media_files = []
media_data_cache = OrderedDict()  # Changed to OrderedDict for LRU eviction
exported_channels = set()
video_texture = 0
current_clip = None
current_clip_duration = 0.0
current_frame = None

def get_media_for_channel(ch):
    if not media_files:
        return None
    file = media_files[ch % len(media_files)]
    return file['path']

def load_audio_data(media_path):
    try:
        if media_path.lower().endswith('.wav'):
            rate, orig = wavfile.read(media_path)
            if len(orig.shape) == 2:
                data = np.mean(orig, axis=1).astype(np.float32)
            else:
                data = orig.astype(np.float32)
            print(f"ℹ️ Loaded audio from {media_path}: rate={rate}, channels={len(orig.shape)}")
            return rate, data
        else:
            format_hint = 'm4a' if media_path.lower().endswith('.mp3') else None  # Added: Hint m4a for mp3 files
            try:
                audio = AudioSegment.from_file(media_path, format=format_hint)
                print(f"ℹ️ Loaded audio from {media_path}: rate={audio.frame_rate}, channels={audio.channels}")
            except CouldntDecodeError:
                print(f"ℹ️ Load failed for {media_path}; retrying as M4A...")
                audio = AudioSegment.from_file(media_path, format='m4a')
                print(f"ℹ️ Loaded audio from {media_path} as M4A: rate={audio.frame_rate}, channels={audio.channels}")
            rate = audio.frame_rate
            data = np.array(audio.get_array_of_samples(), dtype=np.float32)
            if audio.channels == 2:
                data = data.reshape(-1, 2).mean(axis=1)
            return rate, data
    except Exception as e:
        print(f"⚠️ Failed to load audio from {media_path}: {e}")
        return None, None

def ensure_exported(channel_group):
    global exported_channels
    to_export = [ch for ch in channel_group if ch not in exported_channels]
    if to_export:
        export_channels_ultrabatch(to_export)
        for ch in to_export:
            exported_channels.add(ch)

def update_music():
    global previous_channel, current_rate, current_original_data, current_clip, current_clip_duration
    if current_channel == previous_channel:
        return
    pygame.mixer.music.stop()
    if current_clip:
        current_clip.close()  # Added: Explicitly close previous clip to free resources
    current_clip = None
    current_clip_duration = 0.0
    media_path = get_media_for_channel(current_channel)
    if media_path:
        print(f"🎵 Playing media for channel {current_channel}: {media_path}")
        if media_path not in media_data_cache:
            rate, data = load_audio_data(media_path)
            if rate is None:
                print(f"⚠️ No audio data loaded for {media_path}")
                previous_channel = current_channel
                return
            media_data_cache[media_path] = (rate, data)
            media_data_cache.move_to_end(media_path)  # Move to end (most recently used)
            if len(media_data_cache) > MAX_MEDIA_CACHE:
                evicted_path, _ = media_data_cache.popitem(last=False)  # Evict least recently used
                print(f"ℹ️ Evicted media data for {evicted_path} from cache to save memory")
        else:
            media_data_cache.move_to_end(media_path)  # Update LRU
        rate, original_data = media_data_cache[media_path]
        # Modulation similar to older version
        indices = np.arange(SAMPLES_PER_CHANNEL, dtype=np.uint32) + (current_channel * MAX_SLOTS) // CHANNELS
        mod_values = (indices * 2654435761 % 4096) / 4096.0
        num_repeats = math.ceil(len(original_data) / len(mod_values))
        mod_signal = np.tile(mod_values, num_repeats)[:len(original_data)]
        # Optional cymatic modulation (conservative)
        f_query = note_to_freq(current_channel % 72)
        params = get_cymatic_params(f_query)
        Xg, Yg = np.meshgrid(np.linspace(-1, 1, LATTICE_WIDTH), np.linspace(-1, 1, LATTICE_HEIGHT))
        cymatic_mod = generate_cartesian(Xg, Yg, params, omega_time).flatten()
        cymatic_indices = np.linspace(0, len(cymatic_mod) - 1, len(original_data), dtype=int)
        cymatic_mod = cymatic_mod[cymatic_indices]
        # Combine modulations, ensuring positive scaling
        mod_signal = mod_signal * 0.8 + cymatic_mod * 0.02 + 0.2  # Add offset to avoid zeroing
        data_mod = (original_data * mod_signal).clip(-32768, 32767).astype(np.int16)
        # Debug: Check audio data
        if np.all(data_mod == 0):
            print(f"⚠️ Modulated audio is all zeros for channel {current_channel}")
            data_mod = original_data.astype(np.int16)  # Fallback to original
        modulated_wav = os.path.join(tempfile.gettempdir(), f"modulated_{current_channel}_{int(time.time())}.wav")
        try:
            pygame.mixer.quit()
            pygame.mixer.init(frequency=rate, size=-16, channels=1, buffer=4096)
            pygame.mixer.music.set_volume(1.0)
            wavfile.write(modulated_wav, rate, data_mod)
            if os.path.getsize(modulated_wav) < 1000:
                print(f"⚠️ Modulated WAV file {modulated_wav} is too small or empty")
                temp_wav = os.path.join(tempfile.gettempdir(), f"original_{current_channel}_{int(time.time())}.wav")
                wavfile.write(temp_wav, rate, original_data.astype(np.int16))
                pygame.mixer.music.load(temp_wav)
            else:
                pygame.mixer.music.load(modulated_wav)
            pygame.mixer.music.play(-1)
            current_rate = rate
            current_original_data = data_mod
            print(f"ℹ️ Audio loaded: {modulated_wav}, samples={len(data_mod)}, max_amplitude={np.max(np.abs(data_mod))}")
            if media_path.lower().endswith(('.avi', '.mp4', '.mpeg')):
                try:
                    current_clip = VideoFileClip(media_path)
                    current_clip_duration = current_clip.duration
                    print(f"ℹ️ Loaded video from {media_path}: duration={current_clip_duration}s, size={current_clip.size}")
                except Exception as e:
                    print(f"⚠️ Failed to load video from {media_path}: {e}")
                    current_clip = None
                    current_clip_duration = 0.0
        except Exception as e:
            print(f"⚠️ Error during media processing: {e}")
            temp_wav = os.path.join(tempfile.gettempdir(), f"original_{current_channel}_{int(time.time())}.wav")
            wavfile.write(temp_wav, rate, original_data.astype(np.int16))
            pygame.mixer.music.load(temp_wav)
            pygame.mixer.music.play(-1)
            current_rate = rate
            current_original_data = original_data.astype(np.int16)
            print(f"ℹ️ Fallback audio loaded: {temp_wav}, samples={len(original_data)}, max_amplitude={np.max(np.abs(original_data))}")
    else:
        print(f"⚠️ No media path for channel {current_channel}")
        current_rate = None
        current_original_data = None
    previous_channel = current_channel

def display():
    global omega_time, yOffset, audio_level, current_frame, video_texture
    glClear(GL_COLOR_BUFFER_BIT)
    glUseProgram(shader)
    glUniform1f(glGetUniformLocation(shader, "omegaTime"), omega_time)
    glUniform1i(glGetUniformLocation(shader, "yOffset"), yOffset)
    glUniform1i(glGetUniformLocation(shader, "channelHighlight"), current_channel % 24)
    glUniform1f(glGetUniformLocation(shader, "audioLevel"), audio_level)
    glUniform1i(glGetUniformLocation(shader, "hasVideo"), 1 if current_frame is not None else 0)
    if current_frame is not None:
        try:
            glActiveTexture(GL_TEXTURE0)
            glBindTexture(GL_TEXTURE_2D, video_texture)
            glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, current_frame.shape[1], current_frame.shape[0], 0, GL_RGB, GL_UNSIGNED_BYTE, current_frame)
        except Exception as e:
            print(f"⚠️ Failed to update texture: {e}")
    omega_time += 0.01
    glBegin(GL_TRIANGLES)
    glVertex2f(-1, -1)
    glVertex2f(3, -1)
    glVertex2f(-1, 3)
    glEnd()
    glutSwapBuffers()
    glBindTexture(GL_TEXTURE_2D, 0)

def idle():
    global yOffset, current_channel, frame_count, audio_level, current_frame
    frame_count += 1
    if auto_scroll and frame_count % 120 == 0:
        current_channel = (current_channel + 1) % CHANNELS
    channel_group = list(range((current_channel // 24) * 24, min(((current_channel // 24) + 1) * 24, CHANNELS)))
    ensure_exported(channel_group)
    update_music()
    yOffset = (current_channel % 24) * CHUNK_HEIGHT
    audio_level = 0.0
    current_frame = None
    if pygame.mixer.music.get_busy() and current_original_data is not None:
        pos = pygame.mixer.music.get_pos() / 1000.0
        sample_pos = int(pos * current_rate)
        if sample_pos < len(current_original_data):
            audio_level = abs(current_original_data[sample_pos]) / 32768.0
        if current_clip is not None:
            pos_sec = pos % current_clip_duration
            try:
                frame = current_clip.get_frame(pos_sec)
                current_frame = frame
                print(f"ℹ️ Rendered frame at {pos_sec:.2f}s, shape={current_frame.shape}")
            except Exception as e:
                print(f"⚠️ Failed to get video frame at {pos_sec}s: {e}")
                current_frame = None
    glutPostRedisplay()

def keyboard(key, x, y):
    global current_channel, auto_scroll
    if key == b'w':
        current_channel = (current_channel - 1) % CHANNELS
        auto_scroll = False
    elif key == b's':
        current_channel = (current_channel + 1) % CHANNELS
        auto_scroll = False
    elif key == b'a':
        auto_scroll = not auto_scroll
    channel_group = list(range((current_channel // 24) * 24, min(((current_channel // 24) + 1) * 24, CHANNELS)))
    ensure_exported(channel_group)
    update_music()
    glutPostRedisplay()

def init_gl():
    global shader, video_texture
    try:
        print(f"ℹ️ GL_TEXTURE_2D: {GL_TEXTURE_2D}, GL_RGB: {GL_RGB}, GL_UNSIGNED_BYTE: {GL_UNSIGNED_BYTE}")
    except NameError as e:
        print(f"⚠️ OpenGL constant missing: {e}")
        raise
    vs = compileShader(VERTEX_SRC, GL_VERTEX_SHADER)
    fs = compileShader(FRAGMENT_SRC, GL_FRAGMENT_SHADER)
    shader = compileProgram(vs, fs)
    glUseProgram(shader)
    loc = glGetUniformLocation(shader, "phiPowers")
    glUniform1fv(loc, len(PHI_POWERS), PHI_POWERS)
    glUniform1f(glGetUniformLocation(shader, "threshold"), THRESHOLD)
    glUniform1i(glGetUniformLocation(shader, "latticeWidth"), LATTICE_WIDTH)
    glUniform1i(glGetUniformLocation(shader, "latticeHeight"), LATTICE_HEIGHT)
    glUniform1f(glGetUniformLocation(shader, "audioLevel"), 0.0)
    glUniform1i(glGetUniformLocation(shader, "videoTex"), 0)
    glUniform1i(glGetUniformLocation(shader, "hasVideo"), 0)
    video_texture = glGenTextures(1)
    try:
        glBindTexture(GL_TEXTURE_2D, video_texture)
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR)
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR)
        glBindTexture(GL_TEXTURE_2D, 0)
        print("ℹ️ OpenGL texture initialized successfully")
    except Exception as e:
        print(f"⚠️ Failed to initialize OpenGL texture: {e}")
        raise

# -------------------------------
# OpenGL Shaders
# -------------------------------
VERTEX_SRC = """
#version 450 core
layout(location=0) in vec2 pos;
out vec2 texCoord;
void main(){ texCoord=(pos+1.0)*0.5; gl_Position=vec4(pos,0,1); }
"""

FRAGMENT_SRC = """
#version 450 core
in vec2 texCoord;
out vec4 fragColor;
uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform int latticeWidth;
uniform int latticeHeight;
uniform int yOffset;
uniform int channelHighlight;
uniform float audioLevel;
uniform sampler2D videoTex;
uniform int hasVideo;

vec3 channelColors[24] = vec3[24](
vec3(1,0,0),vec3(0,1,0),vec3(0,0,1),vec3(1,1,0),vec3(1,0,1),vec3(0,1,1),
vec3(0.5,0,0),vec3(0,0.5,0),vec3(0,0,0.5),vec3(0.5,0.5,0),vec3(0.5,0,0.5),vec3(0,0.5,0.5),
vec3(0.25,0,0),vec3(0,0.25,0),vec3(0,0,0.25),vec3(0.25,0.25,0),vec3(0.25,0,0.25),vec3(0,0.25,0.25),
vec3(0.75,0,0),vec3(0,0.75,0),vec3(0,0,0.75),vec3(0.75,0.75,0),vec3(0.75,0,0.75),vec3(0,0.75,0.75)
);

float hash_float(int i,int seed){ uint ui=uint(i*374761393 + seed*668265263u); return float(ui & 0xFFFFFFFFu)/4294967295.0; }
vec3 computeVectorColor(int idx,float slot){ return channelColors[channelHighlight%24]*slot; }
float cymatic_pattern(float x, float y, float t) {
    float xn = (x + 1.0) * 0.5;
    float yn = (y + 1.0) * 0.5;
    float alpha = 0.5 + 0.5 * sin(float(channelHighlight % 72) / 100.0);
    float beta = 0.5 + 0.5 * cos(float(channelHighlight % 72) / 200.0);
    float eta = 0.3 + 0.3 * sin(float(channelHighlight % 72) / 50.0);
    float zeta = 0.3 + 0.3 * cos(float(channelHighlight % 72) / 70.0);
    return sin(alpha * 3.14159 * xn + t) * sin(beta * 3.14159 * yn + t) +
           eta * cos(zeta * 3.14159 * (xn + yn) + t);
}
float hdgl_slot(float val,float r_dim,float omega,int x,int y,int idx){
    float resonance = (x%4==0 ? 0.1*sin(omegaTime*0.05+float(y)) : 0.0) + audioLevel * 0.2;
    float wave = (x%3==0 ? 0.3 : (x%3==1 ? 0.0 : -0.3));
    float omega_inst = phiPowers[y%72];
    float rec = r_dim*val*0.5+0.25*sin(omegaTime*r_dim+float(x));
    float cymatic = cymatic_pattern(float(x)/latticeWidth, float(y)/latticeHeight, omegaTime);
    float new_val = val + omega_inst + resonance + wave + rec + omega*0.05 + cymatic * 0.1;
    if(hasVideo == 1){
        vec2 flippedTexCoord = vec2(texCoord.x, 1.0 - texCoord.y);
        vec4 videoColor = texture(videoTex, flippedTexCoord);
        float videoBrightness = (videoColor.r + videoColor.g + videoColor.b) / 3.0;
        new_val += videoBrightness * 0.2;
    }
    return new_val > threshold ? 1.0 : 0.0;
}

void main(){
    int x = int(texCoord.x * float(latticeWidth));
    int y = int(texCoord.y * float(latticeHeight)) + yOffset;
    int idx = y * latticeWidth + x;
    float val = hash_float(idx, 12345);
    float r_dim = 0.3 + 0.01 * float(y);
    float slot = hdgl_slot(val, r_dim, sin(omegaTime), x, y, idx);
    vec3 color = computeVectorColor(idx, slot);
    vec3 finalColor = color.rgb;
    if(hasVideo == 1){
        vec2 flippedTexCoord = vec2(texCoord.x, 1.0 - texCoord.y);
        vec4 videoColor = texture(videoTex, flippedTexCoord);
        float videoBrightness = (videoColor.r + videoColor.g + videoColor.b) / 3.0;
        finalColor *= (videoBrightness + 0.5);
        finalColor = mix(finalColor, videoColor.rgb, val);
    }
    fragColor = vec4(finalColor, 1.0);
}
"""

# -------------------------------
# Main Function
# -------------------------------
def main():
    global media_files, exported_channels
    print(f"🚀 Starting HDGL node ultra-batch with {CHANNELS} channels...")
    try:
        glutInit(sys.argv)
        glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
        glutInitWindowSize(1280, 720)
        glutCreateWindow(b"HDGL Streaming Node - Ultra-Batch OpenCL + OpenGL")
        print("ℹ️ GLUT window initialized successfully")
    except Exception as e:
        print(f"⚠️ Failed to initialize GLUT: {e}")
        sys.exit(1)

    vectors = build_recursive_vectors()
    export_recursive_vectors_json(vectors)
    export_binary_lattice()

    if os.path.exists(EXPORT_BASE4096):
        with open(EXPORT_BASE4096, "r", encoding="utf-8") as f:
            for line in f:
                if line.startswith("#Channel:"):
                    try:
                        ch = int(line[9:].strip())
                        exported_channels.add(ch)
                    except ValueError:
                        pass
    print(f"Loaded {len(exported_channels)} pre-exported channels.")

    for folder, extensions in [('music', ('.wav', '.mp3')), ('movies', ('.avi', '.mp4', '.mpeg'))]:
        if os.path.exists(folder):
            files = sorted([f for f in os.listdir(folder) if f.lower().endswith(extensions)])
            media_files.extend([{'path': os.path.join(folder, f), 'type': folder} for f in files])
            print(f"Loaded {len(files)} {folder} files from '{folder}': {files}")
        else:
            print(f"⚠️ Directory '{folder}' not found.")
    if not media_files:
        print("⚠️ No media files found in 'music' or 'movies'. No audio/video will be played.")

    try:
        init_gl()
        glutDisplayFunc(display)
        glutIdleFunc(idle)
        glutKeyboardFunc(keyboard)
    except Exception as e:
        print(f"⚠️ Failed to set up OpenGL callbacks: {e}")
        sys.exit(1)

    pygame.init()
    pygame.mixer.init()

    channel_group = list(range(0, min(24, CHANNELS)))
    ensure_exported(channel_group)
    update_music()

    print("🖥 OpenGL folding running... (w/s to scroll, a to toggle auto-scroll)")
    glutMainLoop()

if __name__ == "__main__":
    main()