#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
tv10_ultrabatch_gl.py
HDGL recursive node with 131,072 channels, optimized OpenCL HMAC,
streaming-safe Base4096 export, and OpenGL real-time folding with audio/video modulation.
"""

import sys, math, struct, json, unicodedata, hmac, hashlib
import numpy as np
from base4096 import encode
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileShader, compileProgram
from OpenGL.error import GLError  # Added for error catching
import pyopencl as cl
from concurrent.futures import ThreadPoolExecutor
import pygame
import scipy.io.wavfile as wavfile
import os
import tempfile
import time
from pydub import AudioSegment
from pydub.exceptions import CouldntDecodeError
from moviepy.editor import VideoFileClip
from decimal import Decimal, getcontext
from scipy.special import jv, jn_zeros
from collections import OrderedDict

# CONFIG
LATTICE_WIDTH = 1920
LATTICE_HEIGHT = 1080
CHANNELS = 131_072
SAMPLES_PER_CHANNEL = 32
PHI = 1.6180339887498948482
PHI_POWERS = np.array([1.0 / pow(PHI, 7*(i+1)) for i in range(72)], dtype=np.float32)
THRESHOLD = math.sqrt(PHI)
MAX_SLOTS = 16_777_216
USE_OPENCL_HMAC = True
CHUNK_HEIGHT = LATTICE_HEIGHT // 24
getcontext().prec = 120

EXPORT_JSON = "hdgl_vectors.json"
EXPORT_BINARY = "hdgl_lattice.hdgl"
EXPORT_BASE4096 = "vectors_ultrabatch.b4096"
HMAC_KEY = b"ZCHG-Base4096-Signature-Key"

MAX_MEDIA_CACHE = 10

# CHAR SLOT HELPERS
def hdgl_char(idx):
    h = (idx * 2654435761) % 0x110000
    c = chr(h) if 0xD800 > h or h > 0xDFFF else chr((h+1)%0x110000)
    return c

# RECURSIVE VECTORS
def unfold_slot(idx, depth=0):
    val = (idx * 2654435761) % 4096 / 4096.0
    slot = {"idx": idx, "value": val, "char": hdgl_char(idx), "children": []}
    if depth < 3:
        for offset in [1,2]:
            child_idx = (idx*offset + depth*1337) % MAX_SLOTS
            slot["children"].append(unfold_slot(child_idx, depth+1))
    return slot

def build_recursive_vectors(num_samples=SAMPLES_PER_CHANNEL):
    return [unfold_slot(idx) for idx in range(num_samples)]

def flatten_indices_to_bytes(indices):
    arr = bytearray()
    for idx in indices:
        cp = int(idx) % 0x10FFFF
        if 0xD800 <= cp <= 0xDFFF:
            cp += 1
        arr.extend(cp.to_bytes(3,'big'))
    return bytes(arr)

# Cymatic Generators
def get_cymatic_params(f_query):
    alpha = 0.5 + 0.5 * np.sin(f_query / 100)
    beta = 0.5 + 0.5 * np.cos(f_query / 200)
    eta = 0.3 + 0.3 * np.sin(f_query / 50)
    zeta = 0.3 + 0.3 * np.cos(f_query / 70)
    return {"alpha": alpha, "beta": beta, "eta": eta, "zeta": zeta}

def generate_cartesian(X, Y, params, t):
    Xn, Yn = (X + 1) / 2, (Y + 1) / 2
    return (np.sin(params["alpha"] * np.pi * Xn + t) * np.sin(params["beta"] * np.pi * Yn + t) +
            params["eta"] * np.cos(params["zeta"] * np.pi * (Xn + Yn) + t))

def note_to_freq(note_val):
    return 220.0 * 2 ** (note_val / 12)

# Phi Compression
class PhiCompressionInteractive:
    def __init__(self, phi=str(PHI), epsilon="1e-12", max_value=256):
        self.PHI = Decimal(phi)
        self.EPSILON = Decimal(epsilon)
        self.MAX_VALUE = max_value
        self.log_phi = self.PHI.ln()

    def encode_sequence(self, sequence):
        return sum(Decimal(x) for x in sequence)

    def decode_sequence(self, exponent, seq_length):
        sequence = []
        current_sum = exponent
        for _ in range(seq_length):
            if current_sum <= 0:
                break
            b = int(current_sum.to_integral_value(rounding="ROUND_FLOOR"))
            b = min(max(b, 0), self.MAX_VALUE - 1)
            sequence.append(b)
            current_sum -= Decimal(b)
        return sequence

# Nodal Hash
def nodal_hash(alpha, beta, eta, zeta, Nx=32, Ny=32):
    x = np.linspace(0, 1, Nx)
    y = np.linspace(0, 1, Ny)
    Xg, Yg = np.meshgrid(x, y)
    Z = np.sin(alpha * np.pi * Xg) * np.sin(beta * np.pi * Yg) + eta * np.cos(zeta * np.pi * (Xg + Yg))
    flat = Z.flatten()
    flat_bytes = (np.round(flat * 1e5)).astype(np.int64).tobytes()
    h = hashlib.sha256(flat_bytes).hexdigest()
    h_num = int(h[:16], 16)
    return Decimal(h_num)

# Optimized OpenCL HMAC
OPENCL_HMAC_KERNEL = """
# (same as before, omitted for brevity)
"""

def init_opencl():
    # (same as before)
    pass

# Modulated Lattice and Media
def compute_modulated_lattice(channel, x, y, audio_level, video_brightness, t):
    # (same as before)
    pass

# Ultra-Batch Export with Modulation
def export_channels_ultrabatch(channel_list, out_file=EXPORT_BASE4096):
    # (same as before)
    pass

# JSON/Binary Exports with Modulation
def export_recursive_vectors_json(vectors, outfile=EXPORT_JSON):
    # (same as before)
    pass

def export_binary_lattice(num_samples=LATTICE_WIDTH, outfile=EXPORT_BINARY):
    # (same as before)
    pass

# OpenGL Control
omega_time = 0.0
shader = None
yOffset = 0
current_channel = 0
previous_channel = -1
frame_count = 0
auto_scroll = True
audio_level = 0.0
current_rate = None
current_original_data = None
media_files = []
media_data_cache = OrderedDict()
exported_channels = set()
video_texture = 0
current_clip = None
current_clip_duration = 0.0
current_frame = None

def get_media_for_channel(ch):
    # (same as before)
    pass

def load_audio_data(media_path):
    # (same as before, with M4A retry)
    pass

def ensure_exported(channel_group):
    # (same as before)
    pass

def update_music():
    global previous_channel, current_rate, current_original_data, current_clip, current_clip_duration
    if current_channel == previous_channel:
        return
    pygame.mixer.music.stop()
    if current_clip:
        current_clip.close()
    current_clip = None
    current_clip_duration = 0.0
    media_path = get_media_for_channel(current_channel)
    if media_path:
        print(f"🎵 Playing media for channel {current_channel}: {media_path}")
        if media_path not in media_data_cache:
            rate, data = load_audio_data(media_path)
            if rate is None:
                print(f"⚠️ No audio data loaded for {media_path}")
                previous_channel = current_channel
                return
            media_data_cache[media_path] = (rate, data)
            media_data_cache.move_to_end(media_path)
            if len(media_data_cache) > MAX_MEDIA_CACHE:
                evicted_path, _ = media_data_cache.popitem(last=False)
                print(f"ℹ️ Evicted media data for {evicted_path} from cache to save memory")
        else:
            media_data_cache.move_to_end(media_path)
        rate, original_data = media_data_cache[media_path]
        # Modulation (reverted to old with optional cymatic)
        indices = np.arange(SAMPLES_PER_CHANNEL, dtype=np.uint32) + (current_channel * MAX_SLOTS) // CHANNELS
        mod_values = (indices * 2654435761 % 4096) / 4096.0
        num_repeats = math.ceil(len(original_data) / len(mod_values))
        mod_signal = np.tile(mod_values, num_repeats)[:len(original_data)]
        # Optional cymatic (comment out if issues persist)
        f_query = note_to_freq(current_channel % 72)
        params = get_cymatic_params(f_query)
        Xg, Yg = np.meshgrid(np.linspace(-1, 1, LATTICE_WIDTH), np.linspace(-1, 1, LATTICE_HEIGHT))
        cymatic_mod = generate_cartesian(Xg, Yg, params, omega_time).flatten()
        cymatic_indices = np.linspace(0, len(cymatic_mod) - 1, len(original_data), dtype=int)
        cymatic_mod = cymatic_mod[cymatic_indices]
        mod_signal = mod_signal * 0.8 + cymatic_mod * 0.02 + 0.2  # Positive offset
        data_mod = (original_data * mod_signal).clip(-32768, 32767).astype(np.int16)
        if np.all(data_mod == 0):
            print(f"⚠️ Modulated audio is all zeros for channel {current_channel}")
            data_mod = original_data.astype(np.int16)
        modulated_wav = os.path.join(tempfile.gettempdir(), f"modulated_{current_channel}_{int(time.time())}.wav")
        try:
            pygame.mixer.quit()
            pygame.mixer.init(frequency=rate, size=-16, channels=1, buffer=4096)
            pygame.mixer.music.set_volume(1.0)
            wavfile.write(modulated_wav, rate, data_mod)
            if os.path.getsize(modulated_wav) < 1000:
                print(f"⚠️ Modulated WAV file {modulated_wav} is too small or empty")
                temp_wav = os.path.join(tempfile.gettempdir(), f"original_{current_channel}_{int(time.time())}.wav")
                wavfile.write(temp_wav, rate, original_data.astype(np.int16))
                pygame.mixer.music.load(temp_wav)
            else:
                pygame.mixer.music.load(modulated_wav)
            pygame.mixer.music.play(-1)
            current_rate = rate
            current_original_data = data_mod
            print(f"ℹ️ Audio loaded: {modulated_wav}, samples={len(data_mod)}, max_amplitude={np.max(np.abs(data_mod))}")
            if media_path.lower().endswith(('.avi', '.mp4', '.mpeg')):
                try:
                    current_clip = VideoFileClip(media_path)
                    current_clip_duration = current_clip.duration
                    print(f"ℹ️ Loaded video from {media_path}: duration={current_clip_duration}s, size={current_clip.size}")
                except Exception as e:
                    print(f"⚠️ Failed to load video from {media_path}: {e}")
                    current_clip = None
                    current_clip_duration = 0.0
        except Exception as e:
            print(f"⚠️ Error during media processing: {e}")
            temp_wav = os.path.join(tempfile.gettempdir(), f"original_{current_channel}_{int(time.time())}.wav")
            wavfile.write(temp_wav, rate, original_data.astype(np.int16))
            pygame.mixer.music.load(temp_wav)
            pygame.mixer.music.play(-1)
            current_rate = rate
            current_original_data = original_data.astype(np.int16)
            print(f"ℹ️ Fallback audio loaded: {temp_wav}, samples={len(original_data)}, max_amplitude={np.max(np.abs(original_data))}")
    else:
        print(f"⚠️ No media path for channel {current_channel}")
        current_rate = None
        current_original_data = None
    previous_channel = current_channel

# Display, idle, keyboard (same as before)
# ...

def init_gl():
    global shader, video_texture
    try:
        print(f"ℹ️ GL_TEXTURE_2D: {GL_TEXTURE_2D}, GL_RGB: {GL_RGB}, GL_UNSIGNED_BYTE: {GL_UNSIGNED_BYTE}")
    except NameError as e:
        print(f"⚠️ OpenGL constant missing: {e}")
        raise
    try:
        vs = compileShader(VERTEX_SRC, GL_VERTEX_SHADER)
        fs = compileShader(FRAGMENT_SRC, GL_FRAGMENT_SHADER)
        shader = compileProgram(vs, fs)
    except GLError as e:
        print(f"⚠️ Shader compilation/program error: {e}")
        sys.exit(1)
    glUseProgram(shader)
    # (rest same as before)
    print("ℹ️ init_gl completed")

# OpenGL Shaders (lowered version)
VERTEX_SRC = """
#version 330 core  // Lowered from 450
layout(location=0) in vec2 pos;
out vec2 texCoord;
void main(){ texCoord=(pos+1.0)*0.5; gl_Position=vec4(pos,0,1); }
"""

FRAGMENT_SRC = """
#version 330 core  // Lowered from 450
// (rest of shader code same as before, omitted for brevity)
"""

# Main Function
def main():
    # (same as before, with added print before glutMainLoop)
    print("ℹ️ Entering glutMainLoop...")
    glutMainLoop()
    print("ℹ️ Exited glutMainLoop")  # Won't print unless loop exits

if __name__ == "__main__":
    main()