#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
hdgl_server.py
Flask server for HDGL lattice streaming service.
Serves channel data (JSON/HDGL/Base4096) and coordinates media playback.
Windows-compatible with ZCHG base4096.
"""

from flask import Flask, render_template, jsonify, send_file, request, Response
from flask_cors import CORS
import json
import os
import struct
import sys

# Import ZCHG base4096 from local folder
sys.path.insert(0, os.path.join(os.getcwd(), 'base4096'))
from base4096 import encode, decode

app = Flask(__name__)
CORS(app)

# Paths
EXPORT_JSON = "hdgl_vectors.json"
EXPORT_BINARY = "hdgl_lattice.hdgl"
EXPORT_BASE4096 = "vectors_ultrabatch.b4096"
MUSIC_DIR = "music"
MOVIES_DIR = "movies"

# Channel configuration
CHANNELS = 131_072
SAMPLES_PER_CHANNEL = 32

# Cache
channel_cache = {}
media_files = []

def load_media_files():
    """Load available media files."""
    global media_files
    media_files = []
    for folder, extensions in [(MUSIC_DIR, ('.wav', '.mp3')), 
                                (MOVIES_DIR, ('.avi', '.mp4', '.mpeg', '.mkv'))]:
        if os.path.exists(folder):
            files = sorted([f for f in os.listdir(folder) 
                          if f.lower().endswith(extensions)])
            media_files.extend([{
                'path': os.path.join(folder, f).replace('\\', '/'),
                'type': folder,
                'name': f
            } for f in files])
    print(f"Loaded {len(media_files)} media files")

def get_media_for_channel(ch):
    """Map channel to media file."""
    if not media_files:
        return None
    return media_files[ch % len(media_files)]

def parse_base4096_channel(channel_num):
    """Extract channel data from Base4096 export."""
    if not os.path.exists(EXPORT_BASE4096):
        print(f"Warning: {EXPORT_BASE4096} not found")
        return None
    
    if channel_num in channel_cache:
        return channel_cache[channel_num]
    
    try:
        with open(EXPORT_BASE4096, 'r', encoding='utf-8') as f:
            lines = f.readlines()
        
        i = 0
        while i < len(lines):
            if lines[i].startswith(f"#Channel:{channel_num}\n"):
                data_line = lines[i + 1].strip() if i + 1 < len(lines) else None
                hmac_line = lines[i + 2].strip() if i + 2 < len(lines) else None
                
                if data_line and hmac_line:
                    result = {
                        'channel': channel_num,
                        'data': data_line,
                        'hmac': hmac_line.replace('#HMAC:', ''),
                        'media': get_media_for_channel(channel_num)
                    }
                    channel_cache[channel_num] = result
                    return result
            i += 1
    except Exception as e:
        print(f"Error parsing Base4096 for channel {channel_num}: {e}")
    
    return None

def get_json_metadata(limit=32):
    """Load JSON metadata."""
    if not os.path.exists(EXPORT_JSON):
        return []
    
    try:
        with open(EXPORT_JSON, 'r', encoding='utf-8') as f:
            data = json.load(f)
            return data[:limit] if isinstance(data, list) else []
    except Exception as e:
        print(f"Error loading JSON: {e}")
        return []

def get_binary_lattice_chunk(start_idx, count=1024):
    """Read chunk of binary lattice data."""
    if not os.path.exists(EXPORT_BINARY):
        return None
    
    try:
        chunk = []
        with open(EXPORT_BINARY, 'rb') as f:
            f.seek(start_idx * 8)  # 4 bytes float + 4 bytes uint
            for _ in range(count):
                data = f.read(8)
                if len(data) < 8:
                    break
                val, idx = struct.unpack('fI', data)
                chunk.append({'value': val, 'index': idx})
        return chunk
    except Exception as e:
        print(f"Error reading binary lattice: {e}")
        return None

# Routes
@app.route('/')
def index():
    """Serve main page."""
    return render_template('index.html', 
                         total_channels=CHANNELS,
                         media_count=len(media_files))

@app.route('/api/status')
def status():
    """Service status."""
    return jsonify({
        'channels': CHANNELS,
        'media_files': len(media_files),
        'exports': {
            'json': os.path.exists(EXPORT_JSON),
            'binary': os.path.exists(EXPORT_BINARY),
            'base4096': os.path.exists(EXPORT_BASE4096)
        },
        'status': 'online'
    })

@app.route('/api/channel/<int:channel_num>')
def get_channel(channel_num):
    """Get channel data."""
    if channel_num < 0 or channel_num >= CHANNELS:
        return jsonify({'error': 'Invalid channel'}), 400
    
    data = parse_base4096_channel(channel_num)
    if not data:
        # Return minimal data if channel not exported yet
        return jsonify({
            'channel': channel_num,
            'data': '',
            'hmac': '',
            'media': get_media_for_channel(channel_num),
            'status': 'not_exported'
        })
    
    return jsonify(data)

@app.route('/api/metadata')
def metadata():
    """Get JSON metadata."""
    limit = request.args.get('limit', 32, type=int)
    data = get_json_metadata(limit)
    return jsonify(data)

@app.route('/api/lattice')
def lattice():
    """Get binary lattice chunk."""
    start = request.args.get('start', 0, type=int)
    count = request.args.get('count', 1024, type=int)
    data = get_binary_lattice_chunk(start, count)
    
    if data is None:
        return jsonify({'error': 'Lattice data not found'}), 404
    
    return jsonify(data)

@app.route('/api/media/<int:channel_num>')
def get_media_info(channel_num):
    """Get media file info for channel."""
    media = get_media_for_channel(channel_num)
    if not media:
        return jsonify({'error': 'No media for channel'}), 404
    
    return jsonify(media)

@app.route('/media/<path:filepath>')
def serve_media(filepath):
    """Serve media files."""
    # Security: prevent directory traversal
    safe_path = os.path.normpath(filepath)
    if '..' in safe_path:
        return jsonify({'error': 'Invalid path'}), 400
    
    full_path = os.path.join(os.getcwd(), safe_path)
    if not os.path.exists(full_path):
        return jsonify({'error': 'File not found'}), 404
    
    return send_file(full_path)

@app.route('/export/<filename>')
def serve_export(filename):
    """Serve export files."""
    allowed = [EXPORT_JSON, EXPORT_BINARY, EXPORT_BASE4096]
    if filename not in allowed:
        return jsonify({'error': 'Invalid file'}), 400
    
    if not os.path.exists(filename):
        return jsonify({'error': 'File not found'}), 404
    
    return send_file(filename)

if __name__ == '__main__':
    print("=" * 60)
    print("HDGL Streaming Server")
    print("=" * 60)
    
    # Check for base4096
    try:
        from base4096 import encode, decode
        print("✓ ZCHG base4096 library loaded")
    except ImportError:
        print("✗ ZCHG base4096 not found in ./base4096/")
        print("  Clone from: https://github.com/ZCHGorg/base4096")
        sys.exit(1)
    
    # Load media
    load_media_files()
    
    # Check exports
    print(f"\nExport files:")
    print(f"  JSON: {'✓' if os.path.exists(EXPORT_JSON) else '✗'} {EXPORT_JSON}")
    print(f"  Binary: {'✓' if os.path.exists(EXPORT_BINARY) else '✗'} {EXPORT_BINARY}")
    print(f"  Base4096: {'✓' if os.path.exists(EXPORT_BASE4096) else '✗'} {EXPORT_BASE4096}")
    
    if not os.path.exists(EXPORT_BASE4096):
        print("\n⚠ Warning: Run tv10_ultrabatch_gl.py first to generate exports")
    
    print(f"\n✓ Serving {CHANNELS} channels")
    print(f"✓ Server starting at http://localhost:5000")
    print(f"✓ Open in browser: http://localhost:5000\n")
    
    app.run(host='127.0.0.1', port=5000, debug=True, threaded=True)