#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
tv10_ultrabatch_gl.py
HDGL recursive node with 131,072 channels,
ultra-batch OpenCL HMAC, streaming-safe Base4096 export,
and OpenGL real-time folding.
"""

import sys, math, struct, json, unicodedata, hmac, hashlib
import numpy as np
from base4096 import encode
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileShader, compileProgram
import pyopencl as cl
from concurrent.futures import ThreadPoolExecutor
import pygame

# -------------------------------
# CONFIG
# -------------------------------
LATTICE_WIDTH = 1920
LATTICE_HEIGHT = 1080
CHANNELS = 131_072
SAMPLES_PER_CHANNEL = 32
PHI = 1.6180339887
PHI_POWERS = np.array([1.0 / pow(PHI, 7*(i+1)) for i in range(72)], dtype=np.float32)
THRESHOLD = math.sqrt(PHI)
MAX_SLOTS = 16_777_216
USE_OPENCL_HMAC = True
CHUNK_HEIGHT = LATTICE_HEIGHT // 24

EXPORT_JSON = "hdgl_vectors.json"
EXPORT_BINARY = "hdgl_lattice.hdgl"
EXPORT_BASE4096 = "vectors_ultrabatch.b4096"
HMAC_KEY = b"ZCHG-Base4096-Signature-Key"

# -------------------------------
# CHAR SLOT HELPERS
# -------------------------------
def hdgl_char(idx):
    h = (idx * 2654435761) % 0x110000
    c = chr(h) if 0xD800 > h or h > 0xDFFF else chr((h+1)%0x110000)
    return c

# -------------------------------
# RECURSIVE VECTORS
# -------------------------------
def unfold_slot(idx, depth=0):
    val = (idx * 2654435761) % 4096 / 4096.0
    slot = {"idx": idx, "value": val, "char": hdgl_char(idx), "children": []}
    if depth < 3:
        for offset in [1,2]:
            child_idx = (idx*offset + depth*1337) % MAX_SLOTS
            slot["children"].append(unfold_slot(child_idx, depth+1))
    return slot

def build_recursive_vectors(num_samples=SAMPLES_PER_CHANNEL):
    return [unfold_slot(idx) for idx in range(num_samples)]

def flatten_indices_to_bytes(indices):
    arr = bytearray()
    for idx in indices:
        cp = int(idx) % 0x10FFFF
        if 0xD800 <= cp <= 0xDFFF:
            cp += 1
        arr.extend(cp.to_bytes(3,'big'))
    return bytes(arr)

# -------------------------------
# OPENCL HMAC
# -------------------------------
OPENCL_KERNEL = """
__kernel void dummy_hmac(__global uchar *in_data, __global uchar *out_hash){
    int gid = get_global_id(0);
    out_hash[gid] = in_data[gid] ^ 0xAA;
}
"""

def init_opencl():
    platform = cl.get_platforms()[0]
    device = platform.get_devices()[0]
    ctx = cl.Context([device])
    queue = cl.CommandQueue(ctx)
    program = cl.Program(ctx, OPENCL_KERNEL).build()
    kernel = cl.Kernel(program, "dummy_hmac")
    return ctx, queue, kernel

# -------------------------------
# ULTRA-BATCH EXPORT
# -------------------------------
def export_all_channels_ultrabatch(num_channels=CHANNELS, num_samples=SAMPLES_PER_CHANNEL, out_file=EXPORT_BASE4096):
    global USE_OPENCL_HMAC
    print(f"🚀 Ultra-batch exporting {num_channels} channels...")

    all_indices = np.arange(num_samples, dtype=np.uint32)[None,:] + (np.arange(num_channels)[:,None]*MAX_SLOTS)//num_channels
    all_bytes_list = [flatten_indices_to_bytes(row) for row in all_indices]
    offsets = np.cumsum([0]+[len(b) for b in all_bytes_list[:-1]])
    total_bytes = b''.join(all_bytes_list)

    ctx = queue = kernel = None
    hmac_digests = []

    if USE_OPENCL_HMAC:
        try:
            ctx, queue, kernel = init_opencl()
            print("⚡ OpenCL HMAC ultra-batch enabled")
            buf_in = cl.Buffer(ctx, cl.mem_flags.READ_ONLY | cl.mem_flags.COPY_HOST_PTR, hostbuf=np.frombuffer(total_bytes,np.uint8))
            buf_out = cl.Buffer(ctx, cl.mem_flags.WRITE_ONLY, len(total_bytes))
            kernel(queue,(len(total_bytes),),None,buf_in,buf_out)
            result = np.empty_like(np.frombuffer(total_bytes,np.uint8))
            cl.enqueue_copy(queue,result,buf_out)
            queue.finish()
            for i,b in enumerate(all_bytes_list):
                hmac_digests.append(bytes(result[offsets[i]:offsets[i]+32]))
        except Exception as e:
            print(f"⚠️ OpenCL failed: {e}, falling back to CPU")
            USE_OPENCL_HMAC = False

    if not USE_OPENCL_HMAC:
        with ThreadPoolExecutor() as exe:
            hmac_digests = list(exe.map(lambda b: hmac.new(HMAC_KEY,b,hashlib.sha256).digest(), all_bytes_list))

    with open(out_file,"w",encoding="utf-8") as f:
        for bts,digest in zip(all_bytes_list,hmac_digests):
            f.write(encode(bts)+"\n")
            f.write(f"#HMAC:{encode(digest)}\n")
    print(f"✅ Ultra-batch export complete: {out_file}")

# -------------------------------
# JSON/BINARY EXPORTS
# -------------------------------
def export_recursive_vectors_json(vectors, outfile=EXPORT_JSON):
    def filter_surrogates(obj):
        if isinstance(obj, dict): return {k: filter_surrogates(v) for k,v in obj.items()}
        if isinstance(obj, list): return [filter_surrogates(x) for x in obj]
        if isinstance(obj, str): return ''.join(c for c in obj if 0xD800 > ord(c) or ord(c) > 0xDFFF)
        return obj
    safe_vectors = filter_surrogates(vectors)
    with open(outfile,"w",encoding="utf-8") as f:
        json.dump(safe_vectors,f,ensure_ascii=False,indent=2)
    print(f"✅ Exported {len(vectors)} vectors to JSON: {outfile}")

def export_binary_lattice(num_samples=LATTICE_WIDTH, outfile=EXPORT_BINARY):
    with open(outfile,"wb") as f:
        for idx in range(num_samples):
            val = (idx * 2654435761) % 4096 / 4096.0
            packed = struct.pack("fI", val, idx)
            f.write(packed)
    print(f"✅ Exported {num_samples} lattice slots to binary: {outfile}")

# -------------------------------
# OPENGL REAL-TIME FOLDING
# -------------------------------
VERTEX_SRC = """
#version 450 core
layout(location=0) in vec2 pos;
out vec2 texCoord;
void main(){ texCoord=(pos+1.0)*0.5; gl_Position=vec4(pos,0,1); }
"""

FRAGMENT_SRC = """
#version 450 core
in vec2 texCoord;
out vec4 fragColor;
uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform int latticeWidth;
uniform int latticeHeight;
uniform int yOffset;
uniform int channelHighlight;

vec3 channelColors[24] = vec3[24](
vec3(1,0,0),vec3(0,1,0),vec3(0,0,1),vec3(1,1,0),vec3(1,0,1),vec3(0,1,1),
vec3(0.5,0,0),vec3(0,0.5,0),vec3(0,0,0.5),vec3(0.5,0.5,0),vec3(0.5,0,0.5),vec3(0,0.5,0.5),
vec3(0.25,0,0),vec3(0,0.25,0),vec3(0,0,0.25),vec3(0.25,0.25,0),vec3(0.25,0,0.25),vec3(0,0.25,0.25),
vec3(0.75,0,0),vec3(0,0.75,0),vec3(0,0,0.75),vec3(0.75,0.75,0),vec3(0.75,0,0.75),vec3(0,0.75,0.75)
);

float hash_float(int i,int seed){ uint ui=uint(i*374761393 + seed*668265263u); return float(ui & 0xFFFFFFFFu)/4294967295.0; }
vec3 computeVectorColor(int idx,float slot){ return channelColors[channelHighlight%24]*slot; }
float hdgl_slot(float val,float r_dim,float omega,int x,int y,int idx){
float resonance=(x%4==0?0.1*sin(omegaTime*0.05+float(y)):0.0);
float wave=(x%3==0?0.3:(x%3==1?0.0:-0.3));
float omega_inst=phiPowers[y%72];
float rec=r_dim*val*0.5+0.25*sin(omegaTime*r_dim+float(x));
float new_val=val+omega_inst+resonance+wave+rec+omega*0.05;
return new_val>threshold?1.0:0.0;
}

void main(){
int x=int(texCoord.x*float(latticeWidth));
int y=int(texCoord.y*float(latticeHeight))+yOffset;
int idx=y*latticeWidth+x;
float val=hash_float(idx,12345);
float r_dim=0.3+0.01*float(y);
float slot=hdgl_slot(val,r_dim,sin(omegaTime),x,y,idx);
vec3 color=computeVectorColor(idx,slot);
fragColor=vec4(color.rgb,1.0);
}
"""

# -------------------------------
# OPENGL CONTROL
# -------------------------------
omega_time = 0.0
shader = None
yOffset = 0
current_channel = 0
previous_channel = -1
frame_count = 0
auto_scroll = True

def update_music():
    global previous_channel
    channel_group = current_channel % 24
    prev_group = previous_channel % 24
    if channel_group == 3:  # Channel 4 (0-based index 3)
        if prev_group != 3:
            pygame.mixer.music.play(-1)  # Loop indefinitely
    else:
        if prev_group == 3:
            pygame.mixer.music.stop()
    previous_channel = current_channel

def display():
    global omega_time, yOffset
    glClear(GL_COLOR_BUFFER_BIT)
    glUseProgram(shader)
    glUniform1f(glGetUniformLocation(shader,"omegaTime"),omega_time)
    glUniform1i(glGetUniformLocation(shader,"yOffset"),yOffset)
    glUniform1i(glGetUniformLocation(shader,"channelHighlight"),current_channel%24)
    omega_time += 0.01
    glBegin(GL_TRIANGLES)
    glVertex2f(-1,-1)
    glVertex2f(3,-1)
    glVertex2f(-1,3)
    glEnd()
    glutSwapBuffers()

def idle():
    global yOffset,current_channel,frame_count
    frame_count += 1
    if auto_scroll and frame_count%60==0:
        current_channel = (current_channel+1)%CHANNELS
        update_music()
    yOffset = (current_channel%24) * CHUNK_HEIGHT
    glutPostRedisplay()

def keyboard(key,x,y):
    global current_channel,auto_scroll
    if key==b'w': current_channel=(current_channel-1)%CHANNELS; auto_scroll=False
    elif key==b's': current_channel=(current_channel+1)%CHANNELS; auto_scroll=False
    elif key==b'a': auto_scroll=not auto_scroll
    update_music()
    glutPostRedisplay()

def init_gl():
    global shader
    vs = compileShader(VERTEX_SRC, GL_VERTEX_SHADER)
    fs = compileShader(FRAGMENT_SRC, GL_FRAGMENT_SHADER)
    shader = compileProgram(vs, fs)
    glUseProgram(shader)
    loc = glGetUniformLocation(shader,"phiPowers")
    glUniform1fv(loc,len(PHI_POWERS),PHI_POWERS)
    glUniform1f(glGetUniformLocation(shader,"threshold"),THRESHOLD)
    glUniform1i(glGetUniformLocation(shader,"latticeWidth"),LATTICE_WIDTH)
    glUniform1i(glGetUniformLocation(shader,"latticeHeight"),LATTICE_HEIGHT)

# -------------------------------
# MAIN
# -------------------------------
def main():
    print(f"🚀 Starting HDGL node ultra-batch with {CHANNELS} channels...")
    vectors = build_recursive_vectors()
    export_all_channels_ultrabatch()
    export_recursive_vectors_json(vectors)
    export_binary_lattice()

    # Initialize OpenGL window
    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
    glutInitWindowSize(1280, 720)
    glutCreateWindow(b"HDGL Streaming Node - Ultra-Batch OpenCL + OpenGL")
    init_gl()
    glutDisplayFunc(display)
    glutIdleFunc(idle)
    glutKeyboardFunc(keyboard)

    # Initialize Pygame for audio
    pygame.init()
    pygame.mixer.init()
    pygame.mixer.music.load("music.wav")
    update_music()  # Initial check

    print("🖥 OpenGL folding running... (w/s to scroll, a to toggle auto-scroll)")
    glutMainLoop()

if __name__ == "__main__":
    main()