#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
expressive_dictionary_lattice.py

Elegant refactor of HDGL recursive lattice with Phi compression, modulation,
and export. Client holds full lattice; server sends only modulation deltas.
"""

import os, math, json, struct, tempfile, time, hashlib
from decimal import Decimal, getcontext
from concurrent.futures import ThreadPoolExecutor

import numpy as np
import pygame
from scipy.io import wavfile
from pydub import AudioSegment
from pydub.exceptions import CouldntDecodeError
from moviepy.editor import VideoFileClip
from base4096 import encode

# -------------------------------
# CONFIGURATION
# -------------------------------
LATTICE_WIDTH = 1920
LATTICE_HEIGHT = 1080
CHANNELS = 131_072
SAMPLES_PER_CHANNEL = 32
PHI = Decimal("1.6180339887498948482")
MAX_SLOTS = 16_777_216
EXPORT_BASE4096 = "vectors_ultrabatch.b4096"
CHUNK_HEIGHT = LATTICE_HEIGHT // 24
getcontext().prec = 120

# -------------------------------
# MEDIA CACHE & STATE
# -------------------------------
media_files = []
media_cache = {}
current_clip = None
current_clip_duration = 0.0
current_channel = 0
exported_channels = set()
omega_time = 0.0

# -------------------------------
# PHI COMPRESSION
# -------------------------------
class PhiCompressor:
    def __init__(self, phi=PHI, max_val=256):
        self.phi = Decimal(phi)
        self.max_val = max_val

    def encode(self, sequence):
        return sum(Decimal(x) for x in sequence)

    def decode(self, exponent, length):
        seq = []
        total = exponent
        for _ in range(length):
            if total <= 0: break
            val = min(max(int(total), 0), self.max_val - 1)
            seq.append(val)
            total -= Decimal(val)
        return seq

compressor = PhiCompressor()

# -------------------------------
# CORE LATTICE & MODULATION
# -------------------------------
def hdgl_char(idx):
    cp = (idx * 2654435761) % 0x110000
    if 0xD800 <= cp <= 0xDFFF:
        cp += 1
    return chr(cp)

def generate_cymatic_params(channel_idx):
    f = channel_idx % 72
    return {
        "alpha": 0.5 + 0.5 * math.sin(f / 100),
        "beta": 0.5 + 0.5 * math.cos(f / 200),
        "eta": 0.3 + 0.3 * math.sin(f / 50),
        "zeta": 0.3 + 0.3 * math.cos(f / 70)
    }

def cymatic_value(x, y, t, params):
    xn, yn = (x + 1)/2, (y + 1)/2
    return (
        math.sin(params["alpha"]*math.pi*xn + t) * math.sin(params["beta"]*math.pi*yn + t) +
        params["eta"] * math.cos(params["zeta"]*math.pi*(xn + yn) + t)
    )

def compute_modulated_slot(channel_idx, x, y, audio_level=0.0, video_level=0.0, t=0.0):
    base_val = ((channel_idx * 2654435761) % 4096) / 4096.0
    cymatic = cymatic_value(x, y, t, generate_cymatic_params(channel_idx))
    val = base_val + 0.2 * audio_level + 0.2 * video_level + 0.1 * cymatic
    return max(val, 0.0)

# -------------------------------
# MEDIA HANDLING
# -------------------------------
def load_audio(path):
    try:
        if path.lower().endswith(".wav"):
            rate, data = wavfile.read(path)
            data = data.mean(axis=1) if len(data.shape) == 2 else data
            return rate, data.astype(float)
        audio = AudioSegment.from_file(path)
        data = np.array(audio.get_array_of_samples(), dtype=float)
        if audio.channels == 2:
            data = data.reshape(-1,2).mean(axis=1)
        return audio.frame_rate, data
    except CouldntDecodeError:
        audio = AudioSegment.from_file(path, format="m4a")
        data = np.array(audio.get_array_of_samples(), dtype=float)
        if audio.channels == 2:
            data = data.reshape(-1,2).mean(axis=1)
        return audio.frame_rate, data
    except Exception as e:
        print(f"⚠️ Failed to load audio {path}: {e}")
        return None, None

def get_media(channel_idx):
    if not media_files: return None
    file_path = media_files[channel_idx % len(media_files)]["path"]
    if file_path not in media_cache:
        rate, data = load_audio(file_path)
        media_cache[file_path] = (rate, data)
    return media_cache.get(file_path, (None, None))

# -------------------------------
# ULTRA-BATCH EXPORT
# -------------------------------
def flatten_to_bytes(values):
    arr = bytearray()
    for v in values:
        cp = int(v*4096) % 0x10FFFF
        if 0xD800 <= cp <= 0xDFFF: cp += 1
        arr.extend(cp.to_bytes(3, "big"))
    return bytes(arr)

def export_ultrabatch(channels, out_file=EXPORT_BASE4096):
    print(f"🚀 Exporting {len(channels)} channels...")
    mod_bytes = []
    for ch in channels:
        rate, audio_data = get_media(ch)
        audio_lvl = 0.0 if audio_data is None else abs(audio_data[0])/32768.0
        slot_vals = [compute_modulated_slot(ch, x, y, audio_lvl, 0.0, omega_time)
                     for x in range(SAMPLES_PER_CHANNEL) for y in range(1)][:SAMPLES_PER_CHANNEL]
        mod_bytes.append(flatten_to_bytes(slot_vals))

    # CPU HMAC fallback
    hmacs = []
    for ch, bts in zip(channels, mod_bytes):
        digest = hashlib.sha256(bts + str(ch).encode()).digest()
        hmacs.append(digest)

    # Write Base4096 batch
    with open(out_file, "a", encoding="utf-8") as f:
        for ch, bts, h in zip(channels, mod_bytes, hmacs):
            f.write(f"#Channel:{ch}\n")
            f.write(encode(bts) + "\n")
            f.write(f"#HMAC:{encode(h)}\n")
    print(f"✅ Batch export complete.")

# -------------------------------
# JSON VECTOR EXPORT
# -------------------------------
def export_vectors_json(vectors, outfile="hdgl_vectors.json"):
    safe = []
    for v in vectors:
        vec = v.copy()
        vec["value"] = float(compressor.encode([v["value"]]))
        vec["cymatic_params"] = generate_cymatic_params(v["idx"])
        safe.append(vec)
    with open(outfile, "w", encoding="utf-8") as f:
        json.dump(safe, f, ensure_ascii=False, indent=2)
    print(f"✅ Exported {len(vectors)} vectors to {outfile}")

# -------------------------------
# SIMPLE NODE UNFOLD
# -------------------------------
def unfold_slot(idx, depth=0, max_depth=3):
    slot = {"idx": idx, "value": ((idx*2654435761)%4096)/4096.0,
            "char": hdgl_char(idx), "children":[]}
    if depth < max_depth:
        for offset in [1,2]:
            child_idx = (idx*offset + depth*1337) % MAX_SLOTS
            slot["children"].append(unfold_slot(child_idx, depth+1, max_depth))
    return slot

def build_vectors(num_samples=SAMPLES_PER_CHANNEL):
    return [unfold_slot(idx) for idx in range(num_samples)]

# -------------------------------
# SIMPLE TEST RUN
# -------------------------------
if __name__ == "__main__":
    # Example media file setup
    media_files.append({"path": "music.wav"})
    vectors = build_vectors(8)
    export_vectors_json(vectors)
    export_ultrabatch(list(range(4)))
