#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
HDGL-P2P.py
Full HDGL recursive node with PHI-weighted lattice, streaming-safe Base4096 vector snapshots,
HMAC verification, P2P propagation, and real-time OpenGL folding.
"""

import sys, time, math, socket, struct, hmac, hashlib, threading
import numpy as np
from base4096 import encode, decode
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileShader, compileProgram

# -------------------------------
# CONFIG
# -------------------------------
LATTICE_WIDTH = 1024
LATTICE_HEIGHT = 1_048_576  # full lattice
CHUNK_HEIGHT = 256           # tile height
PHI = 1.6180339887
PHI_POWERS = np.array([1.0 / pow(PHI, 7*(i+1)) for i in range(72)], dtype=np.float32)
THRESHOLD = math.sqrt(PHI)
AUTO_SCROLL_SPEED = 1
HMAC_KEY = b"ZCHG-Base4096-Signature-Key"
RECURSION_DEPTH = 2
ALPHA_MERGE = 0.05

PEERS = [("127.0.0.1", 9999)]  # example peers

# -------------------------------
# LATTICE
# -------------------------------
lattice = np.zeros((LATTICE_HEIGHT, LATTICE_WIDTH), dtype=np.float32)
current_yoffset = 0

def extract_edge_slice(rows=4):
    """Extract top rows of the lattice as current 'edge'."""
    return lattice[:rows, :].copy()

def phi_weight_edge(edge):
    """Apply PHI-weighting along Y dimension."""
    weights = np.array([1.0 / pow(PHI, i+1) for i in range(edge.shape[0])], dtype=np.float32)
    return edge * weights[:, None]

def merge_edge(edge, alpha=ALPHA_MERGE):
    """Merge received lattice slice into local lattice."""
    h = min(edge.shape[0], lattice.shape[0])
    lattice[:h, :] = (1-alpha)*lattice[:h, :] + alpha*edge[:h, :]

# -------------------------------
# RECURSIVE VECTORS
# -------------------------------
def unfold_slot(idx, depth=0):
    val = (idx * 2654435761) % 4096 / 4096.0
    slot = {"idx": idx, "value": val, "children": []}
    if depth < RECURSION_DEPTH:
        for offset in [1,2]:
            child_idx = (idx*offset + depth*1337) % 16_777_216
            slot["children"].append(unfold_slot(child_idx, depth+1))
    return slot

def build_recursive_vectors(num_samples=32):
    return [unfold_slot(idx) for idx in range(num_samples)]

def flatten_vector(v):
    if isinstance(v, (list, tuple)):
        return ''.join(flatten_vector(x) for x in v)
    if isinstance(v, dict):
        return ''.join(flatten_vector(v[k]) for k in v if k in ['value','children'])
    return str(v)

def build_edge_vectors(edge_rows):
    vectors = []
    for y, row in enumerate(edge_rows):
        vectors.append({"y":y,"row":row.tolist()})
    return vectors

def vectors_to_base4096(vectors):
    flat = []
    for vec in vectors:
        flat.extend(vec["row"])
    arr = np.array(flat, dtype=np.float32)
    b = arr.tobytes()
    return encode(b), b

def send_edge_vectors_to_peers(edge_rows):
    vectors = build_edge_vectors(edge_rows)
    encoded, raw_bytes = vectors_to_base4096(vectors)
    h = hmac.new(HMAC_KEY, raw_bytes, hashlib.sha256)
    msg = f"{encoded}\n#HMAC:{h.hexdigest()}\n".encode("utf-8")
    for ip, port in PEERS:
        try:
            with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
                s.connect((ip, port))
                s.sendall(msg)
        except Exception:
            continue

def handle_peer_vectors(data):
    try:
        lines = data.decode("utf-8").splitlines()
        sig_line = lines[-1]
        if not sig_line.startswith("#HMAC:"): return
        sig_hex = sig_line.split(":")[1]
        encoded_data = ''.join(lines[:-1])
        raw_bytes = decode(encoded_data)
        h = hmac.new(HMAC_KEY, raw_bytes, hashlib.sha256)
        if h.hexdigest() != sig_hex: return
        arr = np.frombuffer(raw_bytes, dtype=np.float32)
        rows = arr.reshape((-1, LATTICE_WIDTH))
        merge_edge(rows, alpha=ALPHA_MERGE)
    except Exception:
        return

# -------------------------------
# P2P SERVER
# -------------------------------
def p2p_server(port=9999):
    with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
        s.bind(("", port))
        s.listen()
        while True:
            conn, _ = s.accept()
            data = conn.recv(10_000_000)  # arbitrary large buffer
            handle_peer_vectors(data)
            conn.close()

def peer_send_loop():
    while True:
        if PEERS:
            edge = extract_edge_slice()
            weighted = phi_weight_edge(edge)
            merge_edge(weighted, alpha=ALPHA_MERGE)  # optional local self-blend
            send_edge_vectors_to_peers(weighted)
        time.sleep(0.1)

# -------------------------------
# OPENGL SHADERS
# -------------------------------
VERTEX_SRC = """
#version 450 core
layout(location = 0) in vec2 pos;
out vec2 texCoord;
void main(){ texCoord = (pos + 1.0)*0.5; gl_Position = vec4(pos,0,1); }
"""
FRAGMENT_SRC = """
#version 450 core
in vec2 texCoord;
out vec4 fragColor;
uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform int latticeWidth;
uniform int latticeHeight;
uniform int yOffset;
float hash_float(int i, int seed){
    uint ui = uint(i*374761393 + seed*668265263u);
    return float(ui & 0xFFFFFFFFu)/4294967295.0;
}
float hdgl_slot(float val, float r_dim, float omega, int x, int y){
    float resonance=(x%4==0?0.1*sin(omega):0.0);
    float wave=(x%3==0?0.3:(x%3==1?0.0:-0.3));
    float omega_inst=phiPowers[y%72];
    float rec=r_dim*val*0.5 + 0.25*sin(omega*r_dim + float(x));
    float new_val=val + omega_inst + resonance + wave + rec + omega*0.05;
    return new_val>threshold?1.0:0.0;
}
void main(){
    int x=int(texCoord.x*float(latticeWidth));
    int y=int(texCoord.y*float(latticeHeight))+yOffset;
    float val=hash_float(y*latticeWidth+x,12345);
    float r_dim=0.3+0.01*float(y);
    float slot=hdgl_slot(val,r_dim,sin(omegaTime),x,y);
    fragColor=vec4(slot,slot,slot,1.0);
}
"""

shader = None
omega_time = 0.0

def init_gl():
    global shader
    vs = compileShader(VERTEX_SRC, GL_VERTEX_SHADER)
    fs = compileShader(FRAGMENT_SRC, GL_FRAGMENT_SHADER)
    shader = compileProgram(vs, fs)
    glUseProgram(shader)
    glUniform1fv(glGetUniformLocation(shader,"phiPowers"), len(PHI_POWERS), PHI_POWERS)
    glUniform1f(glGetUniformLocation(shader,"threshold"), THRESHOLD)
    glUniform1i(glGetUniformLocation(shader,"latticeWidth"), LATTICE_WIDTH)
    glUniform1i(glGetUniformLocation(shader,"latticeHeight"), LATTICE_HEIGHT)
    glUniform1i(glGetUniformLocation(shader,"yOffset"), 0)

def display():
    global omega_time, current_yoffset
    glClear(GL_COLOR_BUFFER_BIT)
    glUseProgram(shader)
    glUniform1f(glGetUniformLocation(shader,"omegaTime"),omega_time)
    glUniform1i(glGetUniformLocation(shader,"yOffset"),current_yoffset)
    omega_time += 0.01
    glBegin(GL_TRIANGLES)
    glVertex2f(-1,-1)
    glVertex2f( 3,-1)
    glVertex2f(-1, 3)
    glEnd()
    glutSwapBuffers()
    current_yoffset += AUTO_SCROLL_SPEED
    if current_yoffset > LATTICE_HEIGHT - CHUNK_HEIGHT:
        current_yoffset = 0

def idle():
    glutPostRedisplay()

# -------------------------------
# MAIN
# -------------------------------
def main():
    print("🚀 Starting HDGL-P2P node...")

    vectors = build_recursive_vectors(32)

    # start P2P threads
    threading.Thread(target=p2p_server, daemon=True).start()
    threading.Thread(target=peer_send_loop, daemon=True).start()

    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
    glutInitWindowSize(1280,720)
    glutCreateWindow(b"HDGL-P2P Real-Time Folding")
    init_gl()
    glutDisplayFunc(display)
    glutIdleFunc(idle)
    print("🖥 OpenGL real-time folding running...")
    glutMainLoop()

if __name__=="__main__":
    main()
