#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
hmm7.py
HDGL lattice + audio + PHI-weighted P2P + Base4096 streaming.
Thread-safe: compute runs in OpenGL context, audio reads precomputed deltas.
"""

import sys, time, queue, threading
import numpy as np
import sounddevice as sd
from OpenGL.GL import *
from OpenGL.GL.shaders import compileShader, compileProgram
from OpenGL.GLUT import *

# -------------------------------
# CONFIG
# -------------------------------
LATTICE_WIDTH = 128
LATTICE_HEIGHT = 1024
THRESHOLD = np.sqrt(1.6180339887)
DELTA_THRESHOLD = 0.01
PHI = 1.6180339887
PHI_POWERS = np.array([1.0/pow(PHI,7*(i+1)) for i in range(72)], dtype=np.float32)
AUTO_SCROLL_SPEED = 1

# Audio
SAMPLE_RATE = 44100
FRAME_BUFFER_SIZE = 1024

# Shared delta queue
delta_queue = queue.Queue(maxsize=4096)

# Globals
cycle = 0.0
omega_time = 0.0
compute_shader = None
lattice_tex = None
prev_tex = None
delta_ssbo = None

# -------------------------------
# P2P stub
# -------------------------------
def propagate_to_peers(coords):
    """
    PHI-weighted delta propagation.
    Stub: integrate your real P2P networking here.
    """
    # For example: weight recent deltas by PHI and send to peers
    weighted = [(x, y, PHI**(-i%72)) for i, (x,y) in enumerate(coords)]
    # TODO: send `weighted` to connected peers
    pass

# -------------------------------
# Compute Shader
# -------------------------------
COMPUTE_SRC = """
#version 430
layout(local_size_x=16, local_size_y=16) in;

layout(rgba32f, binding=0) uniform image2D latticeTex;
layout(rgba32f, binding=1) uniform image2D prevTex;

layout(std430, binding=2) buffer DeltaBuffer {
    uint count;
    ivec2 coords[];
};

uniform float cycle;
uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform float deltaThreshold;

float hdgl_slot(float val, float r_dim, float omega, int x, int y){
    float resonance = (x % 4 == 0 ? 0.1 * sin(cycle*0.05 + float(y)) : 0.0);
    float wave = (x % 3 == 0 ? 0.3 : (x % 3 == 1 ? 0.0 : -0.3));
    float omega_inst = phiPowers[y % 72];
    float rec = r_dim*val*0.5 + 0.25*sin(cycle*r_dim + float(x));
    float new_val = val + omega_inst + resonance + wave + rec + omega*0.05;
    return new_val > threshold ? 1.0 : 0.0;
}

void main() {
    ivec2 coord = ivec2(gl_GlobalInvocationID.xy);
    ivec2 size = imageSize(latticeTex);
    if(coord.x >= size.x || coord.y >= size.y) return;

    vec4 prev = imageLoad(prevTex, coord);
    vec4 val = imageLoad(latticeTex, coord);

    float r_dim = 0.3 + 0.01*float(coord.y);
    float new_val = hdgl_slot(val.r, r_dim, sin(omegaTime), coord.x, coord.y);
    imageStore(latticeTex, coord, vec4(new_val,0.0,0.0,1.0));

    if(abs(new_val - prev.r) > deltaThreshold){
        uint idx = atomicAdd(count, 1);
        coords[idx] = coord;
    }

    imageStore(prevTex, coord, vec4(new_val,0.0,0.0,1.0));
}
"""

# -------------------------------
# OpenGL Init
# -------------------------------
def init_gl():
    global compute_shader, lattice_tex, prev_tex, delta_ssbo
    compute_shader = compileProgram(compileShader(COMPUTE_SRC, GL_COMPUTE_SHADER))

    # Lattice textures
    for name in ("lattice_tex", "prev_tex"):
        tex = glGenTextures(1)
        glBindTexture(GL_TEXTURE_2D, tex)
        glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA32F, LATTICE_WIDTH, LATTICE_HEIGHT,
                     0, GL_RGBA, GL_FLOAT, None)
        glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST)
        glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST)
        globals()[name] = tex

    # Delta SSBO
    max_deltas = LATTICE_WIDTH*LATTICE_HEIGHT
    delta_ssbo = glGenBuffers(1)
    glBindBuffer(GL_SHADER_STORAGE_BUFFER, delta_ssbo)
    glBufferData(GL_SHADER_STORAGE_BUFFER, 4 + max_deltas*8, None, GL_DYNAMIC_DRAW)
    glBindBufferBase(GL_SHADER_STORAGE_BUFFER, 2, delta_ssbo)

# -------------------------------
# Compute Step
# -------------------------------
def compute_step_main_thread():
    global cycle, omega_time

    glUseProgram(compute_shader)
    glBindImageTexture(0, lattice_tex, 0, GL_FALSE, 0, GL_READ_WRITE, GL_RGBA32F)
    glBindImageTexture(1, prev_tex, 0, GL_FALSE, 0, GL_READ_WRITE, GL_RGBA32F)

    glUniform1f(glGetUniformLocation(compute_shader,"cycle"),cycle)
    glUniform1f(glGetUniformLocation(compute_shader,"omegaTime"),omega_time)
    glUniform1fv(glGetUniformLocation(compute_shader,"phiPowers"),72,PHI_POWERS)
    glUniform1f(glGetUniformLocation(compute_shader,"threshold"),THRESHOLD)
    glUniform1f(glGetUniformLocation(compute_shader,"deltaThreshold"),DELTA_THRESHOLD)

    work_x = (LATTICE_WIDTH + 15)//16
    work_y = (LATTICE_HEIGHT + 15)//16
    glDispatchCompute(work_x, work_y,1)
    glMemoryBarrier(GL_SHADER_STORAGE_BARRIER_BIT | GL_SHADER_IMAGE_ACCESS_BARRIER_BIT)

    # Read deltas
    glBindBuffer(GL_SHADER_STORAGE_BUFFER, delta_ssbo)
    ptr = glMapBuffer(GL_SHADER_STORAGE_BUFFER, GL_READ_ONLY)
    count = np.frombuffer(np.core.multiarray.int_asbuffer(ptr, 4), dtype=np.uint32)[0]
    coords = np.frombuffer(np.core.multiarray.int_asbuffer(ptr+4, count*8), dtype=np.uint32).reshape(-1,2)
    glUnmapBuffer(GL_SHADER_STORAGE_BUFFER)

    # Push to shared queue
    for coord in coords:
        try:
            delta_queue.put_nowait(coord)
        except queue.Full:
            break

    # PHI-weighted propagation
    propagate_to_peers(coords)

    cycle += 1.0
    omega_time += 0.05

# -------------------------------
# Audio Callback
# -------------------------------
def audio_callback(outdata, frames, time_info, status):
    samples = np.zeros((frames,1), dtype=np.float32)
    for i in range(frames):
        try:
            x, y = delta_queue.get_nowait()
            samples[i,0] += (x + y*LATTICE_WIDTH)/LATTICE_WIDTH/2.0 - 1.0
        except queue.Empty:
            break
    outdata[:] = np.clip(samples, -1.0, 1.0)

# -------------------------------
# GLUT Idle
# -------------------------------
def glut_idle():
    compute_step_main_thread()
    glutPostRedisplay()

# -------------------------------
# Main
# -------------------------------
def main():
    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
    glutInitWindowSize(640,480)
    glutCreateWindow(b"HDGL Lattice + Audio + P2P")
    init_gl()
    glutIdleFunc(glut_idle)

    # Audio thread
    audio_thread = threading.Thread(target=lambda: sd.OutputStream(
        channels=1, callback=audio_callback, samplerate=SAMPLE_RATE,
        blocksize=FRAME_BUFFER_SIZE).start(),
        daemon=True)
    audio_thread.start()

    print("[HDGL] Streaming lattice + audio + P2P edges...")
    glutMainLoop()

if __name__=="__main__":
    main()
