#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
TBstream2.1.py
Full HDGL recursive node with streaming-safe Base4096 export (HMAC), 
JSON/binary lattice export, and progressive real-time OpenGL folding.
"""

import sys, json, math, struct, unicodedata, hmac, hashlib
import numpy as np
from base4096 import encode, decode
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileShader, compileProgram

# -------------------------------
# CONFIG
# -------------------------------
LATTICE_WIDTH = 4096
LATTICE_HEIGHT = 4096
CHUNK_HEIGHT = 256            # step per frame
PHI = 1.6180339887
PHI_POWERS = np.array([1.0 / pow(PHI, 7*(i+1)) for i in range(72)], dtype=np.float32)
THRESHOLD = math.sqrt(PHI)
MAX_SLOTS = 16_777_216
RECURSION_DEPTH = 3

EXPORT_JSON = "hdgl_vectors.json"
EXPORT_BINARY = "hdgl_lattice.hdgl"
EXPORT_BASE4096 = "vectors.b4096"
HMAC_KEY = b"ZCHG-Base4096-Signature-Key"

# -------------------------------
# CHAR SLOT HELPERS
# -------------------------------
def is_valid_char(c):
    try:
        cp = ord(c)
        if 0xD800 <= cp <= 0xDFFF:
            return False
        name = unicodedata.name(c)
        cat = unicodedata.category(c)
        if any(bad in name for bad in ['CONTROL','PRIVATE USE','UNASSIGNED','TAG']):
            return False
        if cat in ['Mn','Mc','Me','Cc','Cf','Cs','Cn','Co','Zs']:
            return False
        return True
    except ValueError:
        return False

def hdgl_char(idx):
    h = (idx * 2654435761) % 0x110000
    c = chr(h) if is_valid_char(chr(h)) else chr((h+1)%0x110000)
    return c

# -------------------------------
# RECURSIVE VECTORS
# -------------------------------
def unfold_slot(idx, depth=0):
    val = (idx * 2654435761) % 4096 / 4096.0
    slot = {
        "idx": idx,
        "value": val,
        "char": hdgl_char(idx),
        "children": []
    }
    if depth < RECURSION_DEPTH:
        for offset in [1,2]:
            child_idx = (idx*offset + depth*1337) % MAX_SLOTS
            slot["children"].append(unfold_slot(child_idx, depth+1))
    return slot

def flatten_vector(v):
    if isinstance(v, (list, tuple)):
        return ''.join(flatten_vector(x) for x in v)
    if isinstance(v, dict):
        return ''.join(flatten_vector(v[k]) for k in ['char','value','children'] if k in v)
    return str(v)

def build_recursive_vectors(num_samples=1024):
    return [unfold_slot(idx) for idx in range(num_samples)]

# -------------------------------
# BASE4096 SURROGATE-SAFE
# -------------------------------
def string_to_base4096_bytes(s):
    data = bytearray()
    for c in s:
        cp = ord(c)
        if 0xD800 <= cp <= 0xDFFF or cp > 0x10FFFF:
            continue
        data.extend(cp.to_bytes(3, byteorder='big', signed=False))
    return bytes(data)

def base4096_bytes_to_string(b):
    out = []
    for i in range(0, len(b), 3):
        chunk = int.from_bytes(b[i:i+3], 'big')
        if chunk >= 0x110000:
            raise ValueError(f"Invalid codepoint {chunk}")
        out.append(chr(chunk))
    return ''.join(out)

# -------------------------------
# EXPORT FUNCTIONS
# -------------------------------
def export_recursive_vectors_base4096(vectors, outfile=EXPORT_BASE4096, hmac_key=HMAC_KEY):
    h = hmac.new(hmac_key, digestmod=hashlib.sha256)
    with open(outfile, "w", encoding="utf-8") as f:
        for vector in vectors:
            combined = flatten_vector(vector)
            data = string_to_base4096_bytes(combined)
            h.update(data)
            encoded = encode(data)
            f.write(encoded + "\n")
        signature = h.digest()
        sig_encoded = encode(signature)
        f.write(f"#HMAC:{sig_encoded}\n")
    print(f"✅ Exported vectors to Base4096 with embedded HMAC: {outfile}")

def export_recursive_vectors_json(vectors, outfile=EXPORT_JSON):
    def filter_surrogates(obj):
        if isinstance(obj, dict):
            return {k: filter_surrogates(v) for k,v in obj.items()}
        if isinstance(obj, list):
            return [filter_surrogates(x) for x in obj]
        if isinstance(obj, str):
            return ''.join(c for c in obj if 0xD800 > ord(c) or ord(c) > 0xDFFF)
        return obj
    safe_vectors = filter_surrogates(vectors)
    with open(outfile, "w", encoding="utf-8") as f:
        json.dump(safe_vectors, f, ensure_ascii=False, indent=2)
    print(f"✅ Exported {len(vectors)} vectors to JSON: {outfile}")

def export_binary_lattice(num_samples=LATTICE_WIDTH, outfile=EXPORT_BINARY):
    with open(outfile, "wb") as f:
        for idx in range(num_samples):
            val = (idx * 2654435761) % 4096 / 4096.0
            packed = struct.pack("fI", val, idx)
            f.write(packed)
    print(f"✅ Exported {num_samples} lattice slots to binary: {outfile}")

# -------------------------------
# SHADERS
# -------------------------------
VERTEX_SRC = """
#version 450 core
layout(location = 0) in vec2 pos;
out vec2 texCoord;
void main(){ texCoord = (pos + 1.0)*0.5; gl_Position = vec4(pos,0,1); }
"""

FRAGMENT_SRC = """
#version 450 core
in vec2 texCoord;
out vec4 fragColor;

uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform int latticeWidth;
uniform int latticeHeight;
uniform int yOffset;

float hash_float(int i, int seed) {
    uint ui = uint(i*374761393 + seed*668265263u);
    return float(ui & 0xFFFFFFFFu)/4294967295.0;
}

vec3 computeVectorColor(int idx, float slot) {
    float hue = hash_float(idx,0)*360.0;
    float grad_hue = mod(hue + 180.0,360.0);
    return vec3(hue/360.0, grad_hue/360.0, slot);
}

float hdgl_slot(float val, float r_dim, float omega, int x, int y, int idx){
    float resonance = (x%4==0?0.1*sin(omegaTime*0.05 + float(y)):0.0);
    float wave = (x%3==0?0.3:(x%3==1?0.0:-0.3));
    float omega_inst = phiPowers[y%72];
    float rec = r_dim*val*0.5 + 0.25*sin(omegaTime*r_dim + float(x));
    float new_val = val + omega_inst + resonance + wave + rec + omega*0.05;
    return new_val>threshold?1.0:0.0;
}

void main() {
    int x = int(texCoord.x * float(latticeWidth));
    int y = int(texCoord.y * float(latticeHeight)) + yOffset;
    int idx = y*latticeWidth + x;
    float val = hash_float(idx,12345);
    float r_dim = 0.3 + 0.01*float(y);
    float slot = hdgl_slot(val, r_dim, sin(omegaTime), x, y, idx);
    vec3 color = computeVectorColor(idx, slot);
    fragColor = vec4(color.rgb,1.0);
}
"""

# -------------------------------
# OPENGL MAIN LOOP
# -------------------------------
omega_time = 0.0
y_offset = 0
shader = None

def display():
    global omega_time, y_offset
    glClear(GL_COLOR_BUFFER_BIT)
    glUseProgram(shader)
    glUniform1f(glGetUniformLocation(shader, "omegaTime"), omega_time)
    omega_time += 0.01
    glUniform1i(glGetUniformLocation(shader, "yOffset"), y_offset)
    y_offset = (y_offset + CHUNK_HEIGHT) % LATTICE_HEIGHT
    glBegin(GL_TRIANGLES)
    glVertex2f(-1,-1)
    glVertex2f( 3,-1)
    glVertex2f(-1, 3)
    glEnd()
    glutSwapBuffers()

def idle():
    glutPostRedisplay()

def init_gl():
    global shader
    vs = compileShader(VERTEX_SRC, GL_VERTEX_SHADER)
    fs = compileShader(FRAGMENT_SRC, GL_FRAGMENT_SHADER)
    shader = compileProgram(vs, fs)
    glUseProgram(shader)
    loc = glGetUniformLocation(shader, "phiPowers")
    glUniform1fv(loc, len(PHI_POWERS), PHI_POWERS)
    glUniform1f(glGetUniformLocation(shader, "threshold"), THRESHOLD)
    glUniform1i(glGetUniformLocation(shader, "latticeWidth"), LATTICE_WIDTH)
    glUniform1i(glGetUniformLocation(shader, "latticeHeight"), LATTICE_HEIGHT)
    glUniform1i(glGetUniformLocation(shader, "yOffset"), 0)

def main():
    print("🚀 Starting HDGL recursive node...")

    vectors = build_recursive_vectors(num_samples=32)
    export_recursive_vectors_base4096(vectors)
    export_recursive_vectors_json(vectors)
    export_binary_lattice(num_samples=4096)

    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
    glutInitWindowSize(1280,720)
    glutCreateWindow(b"HDGL Streaming Node - Progressive Folding")

    init_gl()
    glutDisplayFunc(display)
    glutIdleFunc(idle)
    print("🖥 OpenGL progressive folding running...")
    glutMainLoop()

if __name__ == "__main__":
    main()
