#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
HDGL Full Node - Self-contained
Author: Josef Kulovany - ZCHG.org
Features:
- Base-16,777,216 deterministic procedural characters
- GPU lattice HDGL folding with PHI-based evolution
- Procedural vector/glyph generation (rect, circle, line, triangle)
- Self-unfolding, signed JSON export
- Fully self-contained in a single Python file
"""

import sys, time, hashlib, json, unicodedata, os
import numpy as np
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileProgram, compileShader

# Optional GPU utilities
try:
    import GPUtil
except ImportError:
    GPUtil = None

# ---------------- CONFIG ----------------
MAX_CHARS = 4096  # 2^24
SEED = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz!@#$%^&*()-_+=[{]};:',\"<>?/"

WINDOW_WIDTH = 1280
WINDOW_HEIGHT = 720
LATTICE_WIDTH = 32
SUB_TILE_HEIGHT = 256
SCALE_INTERVAL = 2.0
MAX_TEX_HEIGHT = 2048
PHI = 1.6180339887

# ---------------- Alphabet ----------------
def is_valid_char(c):
    try:
        name = unicodedata.name(c)
        return not any(x in name for x in ['CONTROL','PRIVATE USE','SURROGATE','UNASSIGNED','TAG'])
    except ValueError:
        return False

def generate_base16m_alphabet(seed):
    seen = set()
    alphabet = []
    for ch in seed:
        if ch not in seen:
            alphabet.append(ch)
            seen.add(ch)
    codepoint = 0x20
    while len(alphabet) < MAX_CHARS:
        try:
            c = chr(codepoint)
            if c not in seen and is_valid_char(c):
                alphabet.append(c)
                seen.add(c)
        except:
            pass
        codepoint += 1
        if codepoint > 0x10FFFF:
            codepoint = 0x20
    return ''.join(alphabet[:MAX_CHARS])

def hash_to_vector(c):
    shapes = ['rect','circle','line','triangle']
    h = int(hashlib.sha256(c.encode('utf-8')).hexdigest(),16)
    shape = shapes[h%len(shapes)]
    rotate = (h>>8)%360
    scale = ((h>>16)%100)/100 + 0.25
    offset_x = ((h>>24)%200)/100 -1.0
    offset_y = ((h>>32)%200)/100 -1.0
    hue = (h>>40)%360
    gradient_hue = (hue+180)%360
    stroke = ((h>>48)%10)/10 + 0.5
    return {
        'shape': shape,
        'rotate': rotate,
        'scale': scale,
        'offset_x': offset_x,
        'offset_y': offset_y,
        'hue': hue,
        'gradient_hue': gradient_hue,
        'stroke': stroke
    }

# ---------------- Encoder / Decoder ----------------
def encode(data: bytes, alphabet):
    num = int.from_bytes(data, 'big')
    result = []
    base = len(alphabet)
    while num > 0:
        num, rem = divmod(num, base)
        result.append(alphabet[rem])
    return ''.join(reversed(result)) or alphabet[0]

def decode(encoded: str, alphabet):
    idx_map = {c:i for i,c in enumerate(alphabet)}
    base = len(alphabet)
    num = 0
    for c in encoded:
        num = num*base + idx_map[c]
    length = (num.bit_length()+7)//8
    return num.to_bytes(length,'big')

# ---------------- HDGL GPU Engine ----------------
VERTEX_SRC = """
#version 330
layout(location = 0) in vec2 pos;
out vec2 texCoord;
void main(){ texCoord = (pos+1.0)*0.5; gl_Position=vec4(pos,0,1);}
"""
FRAGMENT_SRC = """
#version 330
in vec2 texCoord;
out vec4 fragColor;
uniform sampler2D latticeTex;
uniform float cycle;
uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform int latticeHeight;
uniform int yOffset;
float hdgl_slot(float val,float r_dim,float omega,int x,int y){
    float resonance = (x%4==0?0.1*sin(cycle*0.05+float(y)):0.0);
    float wave = (x%3==0?0.3:(x%3==1?0.0:-0.3));
    float omega_inst = phiPowers[y%72];
    float rec = r_dim*val*0.5+0.25*sin(cycle*r_dim+float(x));
    float new_val = val+omega_inst+resonance+wave+rec+omega*0.05;
    return new_val>threshold?1.0:0.0;
}
void main(){
    int x=int(texCoord.x*32.0);
    int y=int(texCoord.y*float(latticeHeight))+yOffset;
    float val=texelFetch(latticeTex,ivec2(x,y-yOffset),0).r;
    float r_dim=0.3+0.01*float(y);
    float new_val=hdgl_slot(val,r_dim,sin(omegaTime),x,y);
    fragColor=vec4(new_val,sin(omegaTime),0.0,1.0);
}
"""

# Globals
window = None
shader = None
vao = None
textures = []
fbos = []
current = 0
cycle = 0.0
omega_time = 0.0
tile_heights = []
tile_count = 0
num_instances_base = 50_000
frame_times = []

phi_powers = np.array([1.0/pow(PHI,7*(i+1)) for i in range(72)],dtype=np.float32)
threshold = np.sqrt(PHI)

# GPU Detection
def compute_max_instances(lattice_width):
    if GPUtil:
        gpus = GPUtil.getGPUs()
        if gpus:
            gpu = gpus[0]
            free_vram = gpu.memoryFree*1024**2
            return int(free_vram*0.9/(lattice_width*4*4))
    return 50_000

num_instances_max = compute_max_instances(LATTICE_WIDTH)

# ---------------- OpenGL Init ----------------
def init_gl():
    global shader, vao, textures, fbos, tile_heights, tile_count
    shader = compileProgram(compileShader(VERTEX_SRC, GL_VERTEX_SHADER),
                            compileShader(FRAGMENT_SRC, GL_FRAGMENT_SHADER))
    verts=np.array([-1,-1,1,-1,-1,1,1,-1,1,1,-1,1],dtype=np.float32)
    vao = glGenVertexArrays(1)
    glBindVertexArray(vao)
    vbo = glGenBuffers(1)
    glBindBuffer(GL_ARRAY_BUFFER,vbo)
    glBufferData(GL_ARRAY_BUFFER,verts.nbytes,verts,GL_STATIC_DRAW)
    glVertexAttribPointer(0,2,GL_FLOAT,GL_FALSE,0,None)
    glEnableVertexAttribArray(0)
    reinit_lattice(num_instances_base)

def reinit_lattice(new_num_instances):
    global textures, fbos, tile_heights, tile_count
    for tex_pair in textures: glDeleteTextures(tex_pair)
    for fbo_pair in fbos: glDeleteFramebuffers(2,fbo_pair)
    textures.clear(); fbos.clear()
    tile_count=(new_num_instances+MAX_TEX_HEIGHT-1)//MAX_TEX_HEIGHT
    tile_heights[:] = [min(MAX_TEX_HEIGHT,new_num_instances-i*MAX_TEX_HEIGHT) for i in range(tile_count)]
    for t,th in enumerate(tile_heights):
        tex_pair = glGenTextures(2)
        fbo_pair = glGenFramebuffers(2)
        for i in range(2):
            glBindTexture(GL_TEXTURE_2D,tex_pair[i])
            glTexImage2D(GL_TEXTURE_2D,0,GL_RGBA32F,LATTICE_WIDTH,th,0,GL_RGBA,GL_FLOAT,None)
            glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST)
            glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST)
            glBindFramebuffer(GL_FRAMEBUFFER,fbo_pair[i])
            glFramebufferTexture2D(GL_FRAMEBUFFER,GL_COLOR_ATTACHMENT0,GL_TEXTURE_2D,tex_pair[i],0)
        textures.append(tex_pair)
        fbos.append(fbo_pair)
        for start in range(0,th,SUB_TILE_HEIGHT):
            end=min(start+SUB_TILE_HEIGHT,th)
            init_chunk=np.zeros((end-start,LATTICE_WIDTH,4),dtype=np.float32)
            glBindTexture(GL_TEXTURE_2D,tex_pair[0])
            glTexSubImage2D(GL_TEXTURE_2D,0,0,start,LATTICE_WIDTH,end-start,GL_RGBA,GL_FLOAT,init_chunk)
    glBindFramebuffer(GL_FRAMEBUFFER,0)

def display():
    global cycle, omega_time, current
    next_idx=1-current
    for t,th in enumerate(tile_heights):
        for y_start in range(0,th,SUB_TILE_HEIGHT):
            h=min(SUB_TILE_HEIGHT,th-y_start)
            glBindFramebuffer(GL_FRAMEBUFFER,fbos[t][next_idx])
            glViewport(0,0,LATTICE_WIDTH,h)
            glUseProgram(shader)
            glUniform1i(glGetUniformLocation(shader,"latticeTex"),0)
            glUniform1f(glGetUniformLocation(shader,"cycle"),cycle)
            glUniform1f(glGetUniformLocation(shader,"omegaTime"),omega_time)
            glUniform1fv(glGetUniformLocation(shader,"phiPowers"),72,phi_powers)
            glUniform1f(glGetUniformLocation(shader,"threshold"),threshold)
            glUniform1i(glGetUniformLocation(shader,"latticeHeight"), h)
            glUniform1i(glGetUniformLocation(shader,"yOffset"), y_start)
            glActiveTexture(GL_TEXTURE0)
            glBindTexture(GL_TEXTURE_2D,textures[t][current])
            glBindVertexArray(vao)
            glDrawArrays(GL_TRIANGLES,0,6)
    glBindFramebuffer(GL_FRAMEBUFFER,0)
    glViewport(0,0,WINDOW_WIDTH,WINDOW_HEIGHT)
    for t,th in enumerate(tile_heights):
        glBindTexture(GL_TEXTURE_2D,textures[t][next_idx])
        glDrawArrays(GL_TRIANGLES,0,6)
    glutSwapBuffers()
    cycle+=1; omega_time+=0.05; current=next_idx

def idle():
    global num_instances_base
    glutPostRedisplay()
    now = time.time()
    if now%SCALE_INTERVAL<0.05 and num_instances_base<num_instances_max:
        step = int(max(1_000,(1.0-num_instances_base/num_instances_max)*50_000))
        num_instances_base+=step
        if num_instances_base>num_instances_max: num_instances_base=num_instances_max
        reinit_lattice(num_instances_base)

# ---------------- Main ----------------
def main():
    global alphabet, vectors
    alphabet = generate_base16m_alphabet(SEED)
    vectors = [hash_to_vector(c) for c in alphabet[:1024]]  # for export sample
    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA|GLUT_DOUBLE)
    glutInitWindowSize(WINDOW_WIDTH,WINDOW_HEIGHT)
    glutCreateWindow(b"HDGL Full Node - Base16M")
    init_gl()
    glutDisplayFunc(display)
    glutIdleFunc(idle)
    glutMainLoop()

if __name__=="__main__":
    main()
