#!/usr/bin/env python3
# HDGL Base16M GPU Renderer
import sys, time
import numpy as np
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileProgram, compileShader

# ---------- Shader Sources ----------
VERTEX_SRC = """
#version 450 core
layout(location=0) in vec2 pos;
out vec2 texCoord;
void main(){
    texCoord = (pos+1.0)*0.5;
    gl_Position = vec4(pos,0,1);
}
"""

FRAGMENT_SRC = """
#version 450 core
in vec2 texCoord;
out vec4 fragColor;

uniform float omegaTime;
uniform int latticeWidth;
uniform int latticeHeight;

// deterministic hash [0,1]
float hash_float(int i, int seed){
    return float((uint(i*374761393u + seed*668265263u)) & 0xFFFFFFFFu)/4294967295.0;
}

// procedural color
vec3 computeVectorColor(int idx, float slot){
    float hue = hash_float(idx,0)*360.0;
    float grad_hue = mod(hue+180.0,360.0);
    return vec3(hue/360.0, grad_hue/360.0, slot);
}

// HDGL slot logic
float hdgl_slot(float val, float r_dim, float omega, int x, int y, int idx){
    float resonance = (x%4==0?0.1*sin(omegaTime*0.05+float(y)):0.0);
    float wave = (x%3==0?0.3:(x%3==1?0.0:-0.3));
    float rec = r_dim*val*0.5 + 0.25*sin(omegaTime*r_dim+float(x));
    float new_val = val + resonance + wave + rec + omega*0.05;
    return new_val;
}

void main(){
    int x = int(texCoord.x*float(latticeWidth));
    int y = int(texCoord.y*float(latticeHeight));
    int idx = y*latticeWidth+x;
    float val = hash_float(idx,12345);
    float r_dim = 0.3+0.01*float(y);
    float slot = hdgl_slot(val,r_dim,sin(omegaTime),x,y,idx);
    vec3 color = computeVectorColor(idx,slot);
    fragColor = vec4(color,1.0);
}
"""

# ---------- Globals ----------
window = None
shader = None
vao = None
lattice_width = 4096            # 4096 slots horizontally
num_slots = 16_777_216          # 16M slots
lattice_height = (num_slots + lattice_width - 1)//lattice_width
omegaTime = 0.0

# ---------- OpenGL Init ----------
def init_gl():
    global shader, vao
    shader = compileProgram(
        compileShader(VERTEX_SRC,GL_VERTEX_SHADER),
        compileShader(FRAGMENT_SRC,GL_FRAGMENT_SHADER)
    )
    verts = np.array([-1,-1,1,-1,-1,1,1,-1,1,1,-1,1],dtype=np.float32)
    vao = glGenVertexArrays(1)
    glBindVertexArray(vao)
    vbo = glGenBuffers(1)
    glBindBuffer(GL_ARRAY_BUFFER,vbo)
    glBufferData(GL_ARRAY_BUFFER,verts.nbytes,verts,GL_STATIC_DRAW)
    glVertexAttribPointer(0,2,GL_FLOAT,GL_FALSE,0,None)
    glEnableVertexAttribArray(0)

# ---------- Display ----------
def display():
    global omegaTime
    glClear(GL_COLOR_BUFFER_BIT)
    glUseProgram(shader)
    glUniform1f(glGetUniformLocation(shader,"omegaTime"),omegaTime)
    glUniform1i(glGetUniformLocation(shader,"latticeWidth"),lattice_width)
    glUniform1i(glGetUniformLocation(shader,"latticeHeight"),lattice_height)
    glBindVertexArray(vao)
    glDrawArrays(GL_TRIANGLES,0,6)
    glutSwapBuffers()
    omegaTime += 0.05

# ---------- Main ----------
def main():
    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
    glutInitWindowSize(1280,720)
    glutCreateWindow(b"HDGL Base16M GPU")
    init_gl()
    glutDisplayFunc(display)
    glutIdleFunc(display)
    glutMainLoop()

if __name__=="__main__":
    main()
