#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import sys, time
import numpy as np
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileShader, compileProgram

# ---------- Shader Sources ----------
VERTEX_SRC = """
#version 450 core
layout(location = 0) in vec2 pos;
out vec2 texCoord;
void main(){
    texCoord = (pos + 1.0)*0.5;
    gl_Position = vec4(pos,0,1);
}
"""

FRAGMENT_SRC = """
#version 450 core
in vec2 texCoord;
out vec4 fragColor;

uniform sampler2D prevTex;
uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform int latticeWidth;
uniform int latticeHeight;
uniform int yOffset;

// Simple deterministic hash to [0,1]
float hash_float(int i, int seed) {
    uint ui = uint(i*374761393 + seed*668265263u);
    return float(ui & 0xFFFFFFFFu)/4294967295.0;
}

// Procedural vector color
vec3 computeVectorColor(int idx, float slot) {
    float hue = hash_float(idx,0)*360.0;
    float grad_hue = mod(hue + 180.0,360.0);
    return vec3(hue/360.0, grad_hue/360.0, slot);
}

// HDGL slot computation
float hdgl_slot(float val, float r_dim, float omega, int x, int y, int idx){
    float resonance = (x%4==0?0.1*sin(omegaTime*0.05 + float(y)):0.0);
    float wave = (x%3==0?0.3:(x%3==1?0.0:-0.3));
    float omega_inst = phiPowers[y%72];
    float rec = r_dim*val*0.5 + 0.25*sin(omegaTime*r_dim + float(x));
    float new_val = val + omega_inst + resonance + wave + rec + omega*0.05;
    return new_val>threshold?1.0:0.0;
}

void main() {
    int x = int(texCoord.x * float(latticeWidth));
    int y = int(texCoord.y * float(latticeHeight)) + yOffset;
    int idx = y*latticeWidth + x;
    float val = hash_float(idx,12345);
    float r_dim = 0.3 + 0.01*float(y);
    float slot = hdgl_slot(val, r_dim, sin(omegaTime), x, y, idx);
    vec3 color = computeVectorColor(idx, slot);
    fragColor = vec4(color.rgb,1.0);
}
"""

# ---------- Globals ----------
window = None
shader = None
vao = None
textures = []
fbos = []
current = 0
cycle = 0.0
omega_time = 0.0

# ---------- Lattice Parameters ----------
lattice_width = 4096  # x-dim of lattice
lattice_height_total = 4096  # y-dim total
max_tile_height = 1024  # Each FBO tile
threshold = np.sqrt(1.6180339887)
phi = 1.6180339887
phi_powers = np.array([1.0 / pow(phi, 7*(i+1)) for i in range(72)], dtype=np.float32)

tile_heights = []
tile_count = 0
sub_tile_height = 256
scale_interval = 2.0
last_scale_time = time.time()
frame_times = []

# ---------- OpenGL Init ----------
def init_gl():
    global shader, vao, textures, fbos, tile_heights, tile_count

    shader = compileProgram(
        compileShader(VERTEX_SRC, GL_VERTEX_SHADER),
        compileShader(FRAGMENT_SRC, GL_FRAGMENT_SHADER)
    )

    verts = np.array([-1,-1,1,-1,-1,1,1,-1,1,1,-1,1], dtype=np.float32)
    vao = glGenVertexArrays(1)
    glBindVertexArray(vao)
    vbo = glGenBuffers(1)
    glBindBuffer(GL_ARRAY_BUFFER, vbo)
    glBufferData(GL_ARRAY_BUFFER, verts.nbytes, verts, GL_STATIC_DRAW)
    glVertexAttribPointer(0,2,GL_FLOAT,GL_FALSE,0,None)
    glEnableVertexAttribArray(0)

    # Determine tiles
    global tile_heights, tile_count
    tile_count = (lattice_height_total + max_tile_height - 1) // max_tile_height
    tile_heights[:] = [min(max_tile_height, lattice_height_total - i*max_tile_height) for i in range(tile_count)]

    # Allocate textures + FBOs per tile
    for th in tile_heights:
        tex_pair = glGenTextures(2)
        fbo_pair = glGenFramebuffers(2)
        for i in range(2):
            glBindTexture(GL_TEXTURE_2D, tex_pair[i])
            glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA32F, lattice_width, th, 0, GL_RGBA, GL_FLOAT, None)
            glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST)
            glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST)
            glBindFramebuffer(GL_FRAMEBUFFER, fbo_pair[i])
            glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, tex_pair[i],0)
        textures.append(tex_pair)
        fbos.append(fbo_pair)

# ---------- Display ----------
def display():
    global cycle, omega_time, current

    next_idx = 1-current

    for t, th in enumerate(tile_heights):
        glBindFramebuffer(GL_FRAMEBUFFER, fbos[t][next_idx])
        glViewport(0,0,lattice_width,th)
        glUseProgram(shader)
        glUniform1i(glGetUniformLocation(shader,"prevTex"),0)
        glUniform1f(glGetUniformLocation(shader,"omegaTime"),omega_time)
        glUniform1fv(glGetUniformLocation(shader,"phiPowers"),72,phi_powers)
        glUniform1f(glGetUniformLocation(shader,"threshold"),threshold)
        glUniform1i(glGetUniformLocation(shader,"latticeWidth"),lattice_width)
        glUniform1i(glGetUniformLocation(shader,"latticeHeight"),th)
        glUniform1i(glGetUniformLocation(shader,"yOffset"), t*max_tile_height)
        glActiveTexture(GL_TEXTURE0)
        glBindTexture(GL_TEXTURE_2D, textures[t][current])
        glBindVertexArray(vao)
        glDrawArrays(GL_TRIANGLES,0,6)

    # Blit to default framebuffer
    glBindFramebuffer(GL_FRAMEBUFFER,0)
    glViewport(0,0,1280,720)
    for t, th in enumerate(tile_heights):
        glBindTexture(GL_TEXTURE_2D, textures[t][next_idx])
        glDrawArrays(GL_TRIANGLES,0,6)

    glutSwapBuffers()
    start = time.time()
    cycle += 1
    omega_time += 0.05
    current = next_idx
    frame_times.append(time.time()-start)
    if len(frame_times) > 100: frame_times.pop(0)

# ---------- Idle ----------
def idle():
    glutPostRedisplay()
    if frame_times:
        avg = sum(frame_times)/len(frame_times)
        print(f"[Perf] Avg frame: {avg*1000:.2f} ms")

# ---------- Main ----------
def main():
    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
    glutInitWindowSize(1280,720)
    glutCreateWindow(b"HDGL 16M Slots Actualized")
    init_gl()
    glutDisplayFunc(display)
    glutIdleFunc(idle)
    glutMainLoop()

if __name__=="__main__":
    main()
