#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import sys, time, json, math
import numpy as np
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileShader, compileProgram
import unicodedata

# -------------------------------
# CONFIG
# -------------------------------
LATTICE_WIDTH = 4096
LATTICE_HEIGHT_TOTAL = 4096
MAX_TILE_HEIGHT = 1024
SUB_TILE_HEIGHT = 256
PHI = 1.6180339887
PHI_POWERS = np.array([1.0 / pow(PHI, 7*(i+1)) for i in range(72)], dtype=np.float32)
THRESHOLD = math.sqrt(PHI)
MAX_SLOTS = 16_777_216  # procedural alphabet slots

EXPORT_JSON = "hdgl_vectors.json"  # procedural vector export

# -------------------------------
# PROCEDURAL ALPHABET
# -------------------------------
def hdgl_char(idx):
    """Deterministic procedural 'character' mapping for idx in [0, MAX_SLOTS-1]."""
    h = (idx * 2654435761) % 0x110000
    c = chr(h) if is_valid_char(chr(h)) else chr((h+1)%0x110000)
    return c

def is_valid_char(c):
    """Skip control, private use, surrogate, combining, unassigned, tag."""
    try:
        name = unicodedata.name(c)
        cat = unicodedata.category(c)
        if any(bad in name for bad in ['CONTROL','PRIVATE USE','SURROGATE','UNASSIGNED','TAG']):
            return False
        if cat in ['Mn','Mc','Me','Cc','Cf','Cs','Cn','Co','Zs']:
            return False
        return True
    except ValueError:
        return False

# -------------------------------
# VECTOR EXPORT
# -------------------------------
def export_vectors(num_samples=4096):
    """Export procedural vectors for TS/JSON consumption."""
    vectors = []
    for idx in range(num_samples):
        val = (idx * 2654435761) % 4096 / 4096.0
        angle = 2.0 * math.pi * val
        vec = [math.cos(angle), math.sin(angle), val]
        vectors.append({"idx": idx, "vector": vec, "char": hdgl_char(idx)})
    with open(EXPORT_JSON, "w", encoding="utf-8") as f:
        json.dump(vectors, f, ensure_ascii=False, indent=2)
    print(f"✅ Exported {num_samples} procedural vectors to {EXPORT_JSON}")

# -------------------------------
# SHADERS
# -------------------------------
VERTEX_SRC = """
#version 450 core
layout(location = 0) in vec2 pos;
out vec2 texCoord;
void main(){ texCoord = (pos + 1.0)*0.5; gl_Position = vec4(pos,0,1); }
"""

FRAGMENT_SRC = """
#version 450 core
in vec2 texCoord;
out vec4 fragColor;

uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform int latticeWidth;
uniform int latticeHeight;
uniform int yOffset;

float hash_float(int i, int seed) {
    uint ui = uint(i*374761393 + seed*668265263u);
    return float(ui & 0xFFFFFFFFu)/4294967295.0;
}

vec3 computeVectorColor(int idx, float slot) {
    float hue = hash_float(idx,0)*360.0;
    float grad_hue = mod(hue + 180.0,360.0);
    return vec3(hue/360.0, grad_hue/360.0, slot);
}

float hdgl_slot(float val, float r_dim, float omega, int x, int y, int idx){
    float resonance = (x%4==0?0.1*sin(omegaTime*0.05 + float(y)):0.0);
    float wave = (x%3==0?0.3:(x%3==1?0.0:-0.3));
    float omega_inst = phiPowers[y%72];
    float rec = r_dim*val*0.5 + 0.25*sin(omegaTime*r_dim + float(x));
    float new_val = val + omega_inst + resonance + wave + rec + omega*0.05;
    return new_val>threshold?1.0:0.0;
}

void main() {
    int x = int(texCoord.x * float(latticeWidth));
    int y = int(texCoord.y * float(latticeHeight)) + yOffset;
    int idx = y*latticeWidth + x;
    float val = hash_float(idx,12345);
    float r_dim = 0.3 + 0.01*float(y);
    float slot = hdgl_slot(val, r_dim, sin(omegaTime), x, y, idx);
    vec3 color = computeVectorColor(idx, slot);
    fragColor = vec4(color.rgb,1.0);
}
"""

# -------------------------------
# OPENGL + LATTICE
# -------------------------------
window = None
shader = None
vao = None
textures = []
fbos = []
current = 0
cycle = 0.0
omega_time = 0.0
tile_heights = []
tile_count = 0
frame_times = []

def init_gl():
    global shader, vao, textures, fbos, tile_heights, tile_count

    shader = compileProgram(
        compileShader(VERTEX_SRC, GL_VERTEX_SHADER),
        compileShader(FRAGMENT_SRC, GL_FRAGMENT_SHADER)
    )

    verts = np.array([-1,-1,1,-1,-1,1,1,-1,1,1,-1,1], dtype=np.float32)
    vao = glGenVertexArrays(1)
    glBindVertexArray(vao)
    vbo = glGenBuffers(1)
    glBindBuffer(GL_ARRAY_BUFFER, vbo)
    glBufferData(GL_ARRAY_BUFFER, verts.nbytes, verts, GL_STATIC_DRAW)
    glVertexAttribPointer(0,2,GL_FLOAT,GL_FALSE,0,None)
    glEnableVertexAttribArray(0)

    global tile_heights, tile_count
    tile_count = (LATTICE_HEIGHT_TOTAL + MAX_TILE_HEIGHT - 1)//MAX_TILE_HEIGHT
    tile_heights[:] = [min(MAX_TILE_HEIGHT, LATTICE_HEIGHT_TOTAL - i*MAX_TILE_HEIGHT) for i in range(tile_count)]

    for th in tile_heights:
        tex_pair = glGenTextures(2)
        fbo_pair = glGenFramebuffers(2)
        for i in range(2):
            glBindTexture(GL_TEXTURE_2D, tex_pair[i])
            glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA32F, LATTICE_WIDTH, th, 0, GL_RGBA, GL_FLOAT, None)
            glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST)
            glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST)
            glBindFramebuffer(GL_FRAMEBUFFER, fbo_pair[i])
            glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, tex_pair[i],0)
        textures.append(tex_pair)
        fbos.append(fbo_pair)

# -------------------------------
# DISPLAY + IDLE
# -------------------------------
def display():
    global cycle, omega_time, current
    next_idx = 1-current
    for t, th in enumerate(tile_heights):
        glBindFramebuffer(GL_FRAMEBUFFER, fbos[t][next_idx])
        glViewport(0,0,LATTICE_WIDTH,th)
        glUseProgram(shader)
        glUniform1f(glGetUniformLocation(shader,"omegaTime"),omega_time)
        glUniform1fv(glGetUniformLocation(shader,"phiPowers"),72,PHI_POWERS)
        glUniform1f(glGetUniformLocation(shader,"threshold"),THRESHOLD)
        glUniform1i(glGetUniformLocation(shader,"latticeWidth"),LATTICE_WIDTH)
        glUniform1i(glGetUniformLocation(shader,"latticeHeight"),th)
        glUniform1i(glGetUniformLocation(shader,"yOffset"), t*MAX_TILE_HEIGHT)
        glActiveTexture(GL_TEXTURE0)
        glBindTexture(GL_TEXTURE_2D, textures[t][current])
        glBindVertexArray(vao)
        glDrawArrays(GL_TRIANGLES,0,6)

    glBindFramebuffer(GL_FRAMEBUFFER,0)
    glViewport(0,0,1280,720)
    for t, th in enumerate(tile_heights):
        glBindTexture(GL_TEXTURE_2D, textures[t][next_idx])
        glDrawArrays(GL_TRIANGLES,0,6)

    glutSwapBuffers()
    start = time.time()
    cycle += 1
    omega_time += 0.05
    current = next_idx
    frame_times.append(time.time()-start)
    if len(frame_times) > 100: frame_times.pop(0)

def idle():
    glutPostRedisplay()
    if frame_times:
        avg = sum(frame_times)/len(frame_times)
        print(f"[Perf] Avg frame: {avg*1000:.2f} ms")

# -------------------------------
# MAIN
# -------------------------------
def main():
    print("✅ Procedural HDGL node starting...")
    export_vectors()
    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
    glutInitWindowSize(1280,720)
    glutCreateWindow(b"HDGL 16M Slots Node")
    init_gl()
    glutDisplayFunc(display)
    glutIdleFunc(idle)
    glutMainLoop()

if __name__=="__main__":
    main()
