#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import unicodedata
import json
import hashlib

# -------- CONFIG --------
OUTPUT_TS_FILE = "frozen_base32768_alphabet.ts"
SEED = (
    "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz"
    "!@#$%^&*()-_+=[{]};:',\"<>?/`|~"
)
MAX_CHARS = 32768

# -------- DUMMY ALPHABET GENERATOR --------
# For simplicity, assume the alphabet is deterministic and precomputed
# Replace this with your generate_frozen_base32768() if needed
def generate_alphabet(seed, max_chars):
    alphabet = []
    for codepoint in range(0x20, 0x110000):
        if len(alphabet) >= max_chars:
            break
        c = chr(codepoint)
        try:
            name = unicodedata.name(c)
            cat = unicodedata.category(c)
            if cat in ['Zs','Cc','Cf','Cs','Cn','Co']:
                continue
            if c in seed:
                alphabet.append(c)
        except ValueError:
            continue
    # pad with seed first
    for ch in seed:
        if ch not in alphabet:
            alphabet.append(ch)
    return ''.join(alphabet[:max_chars])

# -------- COLOR/GRADIENT GENERATION --------
def style_for_char(c):
    """Deterministically generate a color or gradient for a character."""
    h = int(hashlib.sha256(c.encode("utf-8")).hexdigest(), 16) % 360  # hue
    s = 70  # saturation
    l = 50  # lightness
    color = f"hsl({h},{s}%,{l}%)"
    
    # Optional gradient: combine two hues deterministically
    h2 = (h + 180) % 360
    gradient = f"linear-gradient(hsl({h},{s}%,{l}%),hsl({h2},{s}%,{l}%))"
    return {"color": color, "gradient": gradient}

# -------- TYPE-SCRIPT EXPORT --------
def export_typescript(alphabet, ts_file):
    styles = {c: style_for_char(c) for c in alphabet}
    with open(ts_file, "w", encoding="utf-8") as f:
        f.write("// Auto-generated Base-32,768 alphabet with colors/gradients\n\n")
        f.write(f'export const BASE32768_ALPHABET = {json.dumps(alphabet)};\n\n')
        f.write("export const ALPHABET_STYLES: Record<string, {color:string, gradient?:string}> = ")
        json.dump(styles, f, ensure_ascii=False, indent=2)
        f.write(";\n")
    print(f"✅ TypeScript export written to {ts_file}")

# -------- MAIN --------
def main():
    alphabet = generate_alphabet(SEED, MAX_CHARS)
    export_typescript(alphabet, OUTPUT_TS_FILE)

if __name__ == "__main__":
    main()
