#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import json
import hashlib

# -------- CONFIG --------
OUTPUT_TS_FILE = "base524288_vectors.ts"
MAX_CHARS = 524_288  # Base-524,288
SEED = "DeterministicVectorBase524288"

# -------- VECTOR GENERATION --------
def generate_vector_for_index(i: int):
    """
    Deterministically generate a unique vector for character index i.
    Returns a dict of parameters for procedural glyph rendering.
    """
    # Hash seed + index to get deterministic bytes
    h = hashlib.sha256(f"{SEED}-{i}".encode("utf-8")).digest()

    # Extract deterministic parameters from hash bytes
    rotate = int.from_bytes(h[0:2], 'big') % 360          # 0-359 degrees
    scale = (h[2] / 255) * 0.5 + 0.5                      # 0.5 - 1.0
    stroke = (h[3] / 255) * 3 + 1                         # 1-4 px
    hue = int.from_bytes(h[4:6], 'big') % 360             # 0-359 degrees
    gradient_hue = (hue + 180) % 360                      # opposite hue
    seed_offset = int.from_bytes(h[6:8], 'big')           # extra param for uniqueness

    return {
        "rotate": rotate,
        "scale": round(scale, 3),
        "stroke": round(stroke, 2),
        "hue": hue,
        "gradient_hue": gradient_hue,
        "seed_offset": seed_offset
    }

# -------- TYPE-SCRIPT EXPORT --------
def export_typescript(max_chars: int, ts_file: str):
    vectors = {i: generate_vector_for_index(i) for i in range(max_chars)}
    with open(ts_file, "w", encoding="utf-8") as f:
        f.write("// Auto-generated Base-524,288 vectorized alphabet\n\n")
        f.write(f"export const BASE524288_ALPHABET = Array.from({{length:{max_chars}}}, (_,i)=>i);\n\n")
        f.write("export const ALPHABET_VECTORS: Record<number, {rotate:number, scale:number, stroke:number, hue:number, gradient_hue:number, seed_offset:number}> = ")
        json.dump(vectors, f, ensure_ascii=False, indent=2)
        f.write(";\n")
    print(f"✅ TypeScript export written to {ts_file}")
    print(f"Total characters: {max_chars}")

# -------- MAIN --------
def main():
    export_typescript(MAX_CHARS, OUTPUT_TS_FILE)

if __name__ == "__main__":
    main()
