#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
hdgl_snapshot.py
KISS version of HDGL node:
- Fixed lattice of 16,777,216 slots (Base4096-ready)
- Stateless snapshot(modulation)
- Exports JSON/Base4096 snapshots
- Minimal OpenGL folding (optional)
"""

import sys, os, time, json, math, hashlib, hmac, struct
import numpy as np
from base4096 import encode
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GL.shaders import compileShader, compileProgram

# -------------------------------
# CONFIG
# -------------------------------
CHANNELS = 131_072
MAX_SLOTS = 16_777_216
SAMPLES_PER_CHANNEL = 32
LATTICE_WIDTH = 1920
LATTICE_HEIGHT = 1080
PHI = 1.6180339887
PHI_POWERS = np.array([1.0 / pow(PHI, 7*(i+1)) for i in range(72)], dtype=np.float32)
THRESHOLD = math.sqrt(PHI)

HMAC_KEY = b"ZCHG-Base4096-Signature-Key"

EXPORT_BASE4096 = "snapshot_modulations.b4096"
EXPORT_JSON = "snapshot.json"
EXPORT_BINARY = "lattice_fixed.hdgl"

# -------------------------------
# SNAPSHOT (STATELESS)
# -------------------------------
def snapshot_modulation(channel: int):
    """Produce stateless snapshot for a given channel"""
    indices = np.arange(SAMPLES_PER_CHANNEL, dtype=np.uint32) + (channel * MAX_SLOTS) // CHANNELS
    values = (indices * 2654435761 % 4096) / 4096.0
    data_bytes = b''.join(int(v*4096).to_bytes(2, 'big') for v in values)

    digest = hmac.new(HMAC_KEY, data_bytes, hashlib.sha256).digest()
    snapshot = {
        "meta": {
            "channel": channel,
            "timestamp": time.time(),
            "samples": SAMPLES_PER_CHANNEL,
            "phi_precision": len(PHI_POWERS),
        },
        "modulation": encode(data_bytes),
        "hmac": encode(digest),
    }
    return snapshot

def export_snapshot(snapshot, out_json=EXPORT_JSON, out_base4096=EXPORT_BASE4096):
    with open(out_json, "w", encoding="utf-8") as f:
        json.dump(snapshot, f, indent=2)
    with open(out_base4096, "a", encoding="utf-8") as f:
        f.write(f"#Channel:{snapshot['meta']['channel']}\n")
        f.write(snapshot["modulation"] + "\n")
        f.write(f"#HMAC:{snapshot['hmac']}\n")
    print(f"✅ Exported stateless snapshot for channel {snapshot['meta']['channel']}")

# -------------------------------
# FIXED LATTICE (ONE-TIME EXPORT)
# -------------------------------
def export_fixed_lattice(outfile=EXPORT_BINARY, num_slots=MAX_SLOTS):
    with open(outfile, "wb") as f:
        for idx in range(num_slots):
            val = (idx * 2654435761) % 4096 / 4096.0
            packed = struct.pack("fI", val, idx)
            f.write(packed)
    print(f"✅ Exported fixed lattice of {num_slots} slots to {outfile}")

# -------------------------------
# OPENGL VISUALIZER (OPTIONAL)
# -------------------------------
VERTEX_SRC = """
#version 450 core
layout(location=0) in vec2 pos;
out vec2 texCoord;
void main(){ texCoord=(pos+1.0)*0.5; gl_Position=vec4(pos,0,1); }
"""

FRAGMENT_SRC = """
#version 450 core
in vec2 texCoord;
out vec4 fragColor;
uniform float omegaTime;
uniform float threshold;

float hash_float(int i,int seed){ 
    uint ui=uint(i*374761393 + seed*668265263u);
    return float(ui & 0xFFFFFFFFu)/4294967295.0;
}

void main(){
    int x=int(texCoord.x*1920.0);
    int y=int(texCoord.y*1080.0);
    int idx=y*1920+x;
    float val=hash_float(idx,12345)+sin(omegaTime*0.1);
    fragColor=val>threshold?vec4(1,0,0,1):vec4(0,0,0,1);
}
"""

omega_time = 0.0
shader = None

def display():
    global omega_time
    glClear(GL_COLOR_BUFFER_BIT)
    glUseProgram(shader)
    glUniform1f(glGetUniformLocation(shader, "omegaTime"), omega_time)
    glUniform1f(glGetUniformLocation(shader, "threshold"), THRESHOLD)
    omega_time += 0.01
    glBegin(GL_TRIANGLES)
    glVertex2f(-1, -1)
    glVertex2f(3, -1)
    glVertex2f(-1, 3)
    glEnd()
    glutSwapBuffers()

def idle():
    glutPostRedisplay()

def init_gl():
    global shader
    vs = compileShader(VERTEX_SRC, GL_VERTEX_SHADER)
    fs = compileShader(FRAGMENT_SRC, GL_FRAGMENT_SHADER)
    shader = compileProgram(vs, fs)

# -------------------------------
# MAIN
# -------------------------------
def main():
    print(f"🚀 Starting KISS HDGL snapshot node with {CHANNELS} channels...")
    export_fixed_lattice()

    # Take first snapshot (example)
    snap = snapshot_modulation(0)
    export_snapshot(snap)

    # Minimal OpenGL loop
    try:
        glutInit(sys.argv)
        glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
        glutInitWindowSize(640, 480)
        glutCreateWindow(b"KISS HDGL Visualizer")
        init_gl()
        glutDisplayFunc(display)
        glutIdleFunc(idle)
        print("🖥 Running OpenGL visualizer...")
        glutMainLoop()
    except Exception as e:
        print(f"⚠️ Skipping OpenGL: {e}")

if __name__ == "__main__":
    main()
