import sys, time
import numpy as np
import sounddevice as sd
from OpenGL.GL import *
from OpenGL.GL.shaders import compileProgram, compileShader
from OpenGL.GLUT import *

# ---------- Globals ----------
window = None
compute_shader = None

lattice_width = 32
lattice_height = 256
threshold = np.sqrt(1.6180339887)
delta_threshold = 0.01
phi = 1.6180339887
phi_powers = np.array([1.0/pow(phi,7*(i+1)) for i in range(72)], dtype=np.float32)

cycle = 0.0
omega_time = 0.0

# ---------- Audio params ----------
sample_rate = 44100
frame_buffer_size = 1024

# ---------- Compute Shader ----------
COMPUTE_SRC = """
#version 430
layout(local_size_x=16, local_size_y=16) in;

layout(rgba32f, binding=0) uniform image2D latticeTex;
layout(rgba32f, binding=1) uniform image2D prevTex;

layout(std430, binding=2) buffer DeltaBuffer {
    uint count;
    ivec2 coords[];
};

uniform float cycle;
uniform float omegaTime;
uniform float phiPowers[72];
uniform float threshold;
uniform float deltaThreshold;

float hdgl_slot(float val, float r_dim, float omega, int x, int y){
    float resonance = (x % 4 == 0 ? 0.1 * sin(cycle*0.05 + float(y)) : 0.0);
    float wave = (x % 3 == 0 ? 0.3 : (x % 3 == 1 ? 0.0 : -0.3));
    float omega_inst = phiPowers[y % 72];
    float rec = r_dim*val*0.5 + 0.25*sin(cycle*r_dim + float(x));
    float new_val = val + omega_inst + resonance + wave + rec + omega*0.05;
    return new_val > threshold ? 1.0 : 0.0;
}

void main() {
    ivec2 coord = ivec2(gl_GlobalInvocationID.xy);
    ivec2 size = imageSize(latticeTex);
    if(coord.x >= size.x || coord.y >= size.y) return;

    vec4 prev = imageLoad(prevTex, coord);
    vec4 val = imageLoad(latticeTex, coord);

    float r_dim = 0.3 + 0.01*float(coord.y);
    float new_val = hdgl_slot(val.r, r_dim, sin(omegaTime), coord.x, coord.y);
    imageStore(latticeTex, coord, vec4(new_val,0.0,0.0,1.0));

    if(abs(new_val - prev.r) > deltaThreshold){
        uint idx = atomicAdd(count, 1);
        coords[idx] = coord;
    }

    imageStore(prevTex, coord, vec4(new_val,0.0,0.0,1.0));
}
"""

# ---------- OpenGL Init ----------
def init_gl():
    global compute_shader, lattice_tex, prev_tex, delta_ssbo
    compute_shader = compileShader(COMPUTE_SRC, GL_COMPUTE_SHADER)
    compute_shader = compileProgram(compute_shader)

    lattice_tex = glGenTextures(1)
    glBindTexture(GL_TEXTURE_2D, lattice_tex)
    glTexImage2D(GL_TEXTURE_2D,0,GL_RGBA32F,lattice_width,lattice_height,0,GL_RGBA,GL_FLOAT,None)
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST)
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST)

    prev_tex = glGenTextures(1)
    glBindTexture(GL_TEXTURE_2D, prev_tex)
    glTexImage2D(GL_TEXTURE_2D,0,GL_RGBA32F,lattice_width,lattice_height,0,GL_RGBA,GL_FLOAT,None)
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST)
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST)

    max_deltas = lattice_width*lattice_height
    delta_ssbo = glGenBuffers(1)
    glBindBuffer(GL_SHADER_STORAGE_BUFFER, delta_ssbo)
    glBufferData(GL_SHADER_STORAGE_BUFFER, 4 + max_deltas*8, None, GL_DYNAMIC_DRAW)
    glBindBufferBase(GL_SHADER_STORAGE_BUFFER, 2, delta_ssbo)

# ---------- Compute Step ----------
def compute_step():
    global cycle, omega_time
    glUseProgram(compute_shader)
    glBindImageTexture(0, lattice_tex, 0, GL_FALSE, 0, GL_READ_WRITE, GL_RGBA32F)
    glBindImageTexture(1, prev_tex, 0, GL_FALSE, 0, GL_READ_WRITE, GL_RGBA32F)

    glUniform1f(glGetUniformLocation(compute_shader,"cycle"),cycle)
    glUniform1f(glGetUniformLocation(compute_shader,"omegaTime"),omega_time)
    glUniform1fv(glGetUniformLocation(compute_shader,"phiPowers"),72,phi_powers)
    glUniform1f(glGetUniformLocation(compute_shader,"threshold"),threshold)
    glUniform1f(glGetUniformLocation(compute_shader,"deltaThreshold"),delta_threshold)

    work_x = (lattice_width + 15)//16
    work_y = (lattice_height + 15)//16
    glDispatchCompute(work_x, work_y,1)
    glMemoryBarrier(GL_SHADER_STORAGE_BARRIER_BIT | GL_SHADER_IMAGE_ACCESS_BARRIER_BIT)

    glBindBuffer(GL_SHADER_STORAGE_BUFFER, delta_ssbo)
    ptr = glMapBuffer(GL_SHADER_STORAGE_BUFFER, GL_READ_ONLY)
    count = np.frombuffer(np.core.multiarray.int_asbuffer(ptr, 4), dtype=np.uint32)[0]
    coords = np.frombuffer(np.core.multiarray.int_asbuffer(ptr+4, count*8), dtype=np.uint32).reshape(-1,2)
    glUnmapBuffer(GL_SHADER_STORAGE_BUFFER)

    cycle += 1.0
    omega_time += 0.05
    return coords

# ---------- Audio Callback ----------
def audio_callback(outdata, frames, time_info, status):
    deltas = compute_step()
    if len(deltas) == 0:
        outdata[:] = np.zeros((frames,1), dtype=np.float32)
        return
    samples = np.zeros((frames,1), dtype=np.float32)
    for i, (x,y) in enumerate(deltas):
        idx = i % frames
        samples[idx,0] += (x + y*lattice_width)/4096.0*2.0 - 1.0
    # Normalize
    samples = np.clip(samples, -1.0, 1.0)
    outdata[:] = samples

# ---------- Main ----------
def main():
    glutInit(sys.argv)
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE)
    glutInitWindowSize(640,480)
    glutCreateWindow(b"HDGL Delta -> Real-time Audio")
    init_gl()

    # Stream audio
    with sd.OutputStream(channels=1, callback=audio_callback,
                         samplerate=sample_rate, blocksize=frame_buffer_size):
        print("[Audio] Streaming... Press Ctrl+C to stop.")
        while True:
            time.sleep(0.1)

if __name__=="__main__":
    main()
